import copy
from complexity.infrastructures.unidirectional_corridor import UnidirectionalCorridor
from complexity.computational_complexity import TestCase
from complexity.models.social_force import WEIDEMANN_FD, SocialForceModel, closest_point_on_line

def test_closest_point_on_line():
    line = [(0.0,0.0), (2.0,0.0), 1/4]
    pos_2_point = [
        ((0,0.5), (0,0)),
        ((2,0.5), (2,0)),
        ((0.6,0.5), (0.6,0)),
        ((2.6,0.5), (2.0,0)),
        ((-1.6,0.5), (0.0,0)),
    ]

    for pos, point in pos_2_point:
        closest_x, closest_y = closest_point_on_line(line, pos)
        assert round(closest_x, 12) == point[0]
        assert round(closest_y, 12) == point[1]

def test_single_ped_init():
    unidir_corridor = UnidirectionalCorridor(2, 1.2)
    mean_density = 1/unidir_corridor.area    
    density_distr_factor = 0
    sf_test_case = TestCase(unidir_corridor, mean_density, density_distr_factor)
    sf_model = SocialForceModel(sf_test_case)
    sf_model.initialize()
    assert len(sf_model.pedestrians) == 1
    assert len(sf_model.obstacles) == 2

    ped = sf_model.pedestrians[0]
    for line in sf_model.obstacles:
        closest_x, closest_y = closest_point_on_line(line, ped.pos)
        assert closest_y == 0.0 or closest_y == 1.2
        assert round(closest_x,12) == round(ped.pos[0],12)

    dir_x, dir_y = sf_model.get_pref_dir(ped.pos)
    assert dir_x == 1
    assert dir_y == 0
        
def test_single_ped_run():
    unidir_corridor = UnidirectionalCorridor(2, 1.2)
    mean_density = 1/unidir_corridor.area    
    density_distr_factor = 0
    sf_test_case = TestCase(unidir_corridor, mean_density, density_distr_factor)
    sf_model = SocialForceModel(sf_test_case, {'obs_force_mag':0.0})
    sf_model.initialize()
    assert len(sf_model.pedestrians) == 1
    
    ped = sf_model.pedestrians[0]
    org_pos = copy.copy(ped.pos)
    ped.vel[0] = ped.pref_speed

    sf_model.run()
    
    assert round(ped.vel[0],12) ==  round(ped.pref_speed,12)
    assert round(ped.vel[1],12) == 0.0

    expected_pos = org_pos[0] + ped.pref_speed*sf_model.DELTA_T
    assert round(expected_pos,12) == round(ped.pos[0],12)
    assert round(org_pos[1],12) == round(ped.pos[1],12)


def test_single_ped_run_obstacle():
    unidir_corridor = UnidirectionalCorridor(2, 1.2)
    mean_density = 1/unidir_corridor.area    
    density_distr_factor = 0
    sf_test_case = TestCase(unidir_corridor, mean_density, density_distr_factor)
    ped_params = {
        'obs_force_mag':1.0,
        'obs_force_sig':1.0}

    sf_model = SocialForceModel(sf_test_case)
    sf_model.initialize()
    assert len(sf_model.pedestrians) == 1
    
    ped = sf_model.pedestrians[0]
    ped.vel[0] = 1
    ped.pos = [1.0, 0.4]

    sf_model.run()
    
    assert round(ped.vel[0],6) == 1.034000
    assert round(ped.vel[1],6) == 0.013496

    assert round(ped.pos[0],6) == 1.051700
    assert round(ped.pos[1],6) == 0.400675


def test_double_ped_run_no_obstacle():
    unidir_corridor = UnidirectionalCorridor(2, 1.2)
    mean_density = 2/unidir_corridor.area    
    density_distr_factor = 0
    sf_test_case = TestCase(unidir_corridor, mean_density, density_distr_factor)
    ped_params = {
        'obs_force_mag':0.0,
        'ped_force_mag':1.0,
        'ped_force_sig':1.0}

    sf_model = SocialForceModel(sf_test_case, ped_params)
    sf_model.initialize()
    assert len(sf_model.pedestrians) == 2
    
    ped_1 = sf_model.pedestrians[0]
    ped_1.pos = [1.0, 0.4]
    ped_1.vel[0] = 1
    ped_2 = sf_model.pedestrians[1]
    ped_2.pos = [1.5, 0.6]
    ped_2.vel[0] = 1

    sf_model.run()
    
    assert round(ped_1.vel[0],6) == 0.993581
    assert round(ped_1.vel[1],6) == -0.016168
    assert round(ped_1.pos[0],6) == 1.049679
    assert round(ped_1.pos[1],6) == 0.399192

    assert round(ped_2.vel[0],6) == 1.074419
    assert round(ped_2.vel[1],6) == 0.016168
    assert round(ped_2.pos[0],6) == 1.553721
    assert round(ped_2.pos[1],6) == 0.600808
