from math import ceil
from typing import Sequence
import numpy as np
import numpy.typing as npt


def is_valid_rectangle(coords: npt.NDArray) -> bool:
    if len(coords) != 4:
        return False
    x = coords[:, 0]
    y = coords[:, 1]
    try:
        x_min, x_max = np.unique(x)
    except ValueError:
        return False
    try:
        y_min, y_max = np.unique(y)
    except ValueError:
        return False

    if sum(x == x_min) != 2 or sum(x == x_max) != 2:
        return False
    if sum(y == y_min) != 2 or sum(y == y_max) != 2:
        return False

    return True


class RoutingFloorField():

    def __init__(self,
                 walkable_space: list[tuple[float, float]],
                 destination: list[tuple[float, float]],
                 grid_size: float) -> None:
        self.walkable_space = np.array(walkable_space)
        self.destination = np.array(destination)
        self.grid_size = grid_size

        self._geom_checks()
        self._create_cost_matrix()

    def _create_cost_matrix(self) -> None:
        x_min, x_max = self._get_min_max('x')
        y_min, y_max = self._get_min_max('y')
        x_cell_count = ceil((x_max-x_min)/self.grid_size)
        y_cell_count = ceil((y_max-y_min)/self.grid_size)
        max_walking_cost = (x_cell_count+y_cell_count)*4
        self.cost_matrix = np.ones(
            (x_cell_count, y_cell_count), dtype=float)*max_walking_cost

    def _set_destination_cost(self) -> None:
        min_x_coord = round(min(self.destination[:, 0])/self.grid_size)
        max_x_coord = round(max(self.destination[:, 0])/self.grid_size)
        min_y_coord = round(min(self.destination[:, 1])/self.grid_size)
        max_y_coord = round(max(self.destination[:, 1])/self.grid_size)

        self.cost_matrix[min_x_coord:max_x_coord, min_y_coord:max_y_coord] = 0

    def _flood_cost_matrix(self) -> None:
        pass

    def _geom_checks(self) -> None:
        if not is_valid_rectangle(self.walkable_space):
            raise Exception(
                'RoutingFloorField only works for walkable spaces that are rectangular geometries starting at (0,0)!'
            )
        if not is_valid_rectangle(self.destination):
            raise Exception(
                'RoutingFloorField only works for destinations that are rectangular geometries starting at (0,0)!'
            )

    def _get_min_max(self, axis: str) -> tuple[float, float]:
        if axis == 'x':
            column = 0
        elif axis == 'y':
            column = 1

        max_val = max(max(self.walkable_space[:, column]),
                      max(self.destination[:, column]))
        min_val = min(min(self.walkable_space[:, column]),
                      min(self.destination[:, column]))
        return min_val, max_val


def create_floor_field(walkable_space: list[tuple[float, float]],
                       destination: list[tuple[float, float]],
                       grid_size: float) -> None:
    # Create a routing floor field using a flooding algorithm -> this works fine for simple infrastructures
    pass
