from __future__ import annotations
from dataclasses import astuple, dataclass, field
from typing import TYPE_CHECKING

from complexity.constants import *

import numpy.typing as npt
import numpy as np

if TYPE_CHECKING:
    from complexity.computational_complexity import Experiment

class Infrastructure():

    def __init__(self, area: float, obstacle_count: int) -> None:
        self.area = area
        self.obstacles: list = []

        self.obstacle_count = obstacle_count

    def get_ped_pos_continuous(
            self, ped_count: int, jam_density: float,
            local_density: float|DistrType) -> tuple[list[CONT_POS], float]:
        raise NotImplementedError

    def get_ped_pos_cell(
            self, ped_count: int, jam_density: float,
            local_density: float|DistrType) -> tuple[list[CELL_POS], float]:
        raise NotImplementedError

    def get_density_per_cell(self,
                             cell_size: float,
                             ped_count: int,
                             jam_density: float,
                             local_density: float|DistrType) -> npt.NDArray[np.float64]:
        raise NotImplementedError

    def get_route_floor_field(self,
                              cell_size: float) -> npt.NDArray[np.float64]:
        raise NotImplementedError

    def get_obstacles(self) -> list[Obstacle]:
        raise NotImplementedError

    def get_grid_representation(
            self,
            cell_size: float,
            z_size: int = 1,
            dtype: npt.DTypeLike = float,
            default_val: object = 1) -> tuple[npt.NDArray, int, int]:
        raise NotImplementedError

    def get_grid_representation_as_dict(
            self, grid_size: float) -> dict[tuple[int, int], list]:
        raise NotImplementedError

    @staticmethod
    def get_instance(step_count: int, obstacle_count: int) -> Infrastructure:
        raise NotImplementedError

    @property
    def min_obstacle_count(self) -> int:
        raise NotImplementedError

    @property
    def tag(self) -> str:
        raise NotImplementedError

    def __str__(self) -> str:
        raise NotImplementedError


class PedModel():

    def __init__(self, infrastructure: Infrastructure) -> None:
        self.infra = infrastructure
        self.initialize()

    def set_state(self, experiment: Experiment) -> None:
        # Rest to initial state (however, do enure randomization)
        raise NotImplementedError

    def run(self) -> None:
        # Run the code that needs to be timed
        raise NotImplementedError

    def initialize(self) -> None:
        # Convert test case defintion to model specific definition
        # Including computing the initial velocities
        raise NotImplementedError

    @property
    def label(self) -> str:
        raise NotImplementedError

    @staticmethod
    def get_tag() -> str:
        raise NotImplementedError

    @property
    def tag(self) -> str:
        return self.get_tag()

    def __str__(self) -> str:
        return f'{self.label} - {self.infra}'
