/**
 * @class Surface
 * @brief A class representing a surface with geometry and statistical properties.
 * 
 * The Surface class stores the geometric data and statistical properties (e.g., autocorrelation length,
 * mu and sigma coefficients) required for surface modeling and interaction in simulations.
 * 
 * @tparam T Type parameter for numerical data (typically float or double).
 */
template <typename T>
class Surface {

    private:

        /** 
         * @brief Geometry of the surface.
         */
        Geometry<T> geometry;

        /** 
         * @brief Mu coefficients for the surface properties.
         */
        Matrix<T> mu_coefficients;

        /** 
         * @brief Sigma coefficients for the surface properties.
         */
        Matrix<T> sigma_coefficients;

        /** 
         * @brief Autocorrelation length of the surface.
         */
        T ac_length;

        /** 
         * @brief Local parameters associated with the surface.
         */
        T local_parameters[4];

    public:

        /** 
         * @brief Default constructor for Surface class.
         * 
         * Initializes the autocorrelation length to 0.
         */
        Surface();

        /**
         * @brief Parameterized constructor for Surface class.
         * 
         * @param geometry_file The file containing surface geometry data.
         * @param properties_file The file containing surface properties (e.g., mu, sigma coefficients).
         */
        Surface(std::string geometry_file, std::string properties_file);

        /**
         * @brief Destructor for Surface class.
         * 
         * Resets the autocorrelation length to 0.
         */
        ~Surface();

        /**
         * @brief Imports surface properties from a file.
         * 
         * Reads the surface properties, such as mu and sigma coefficients, and autocorrelation length
         * from the provided properties file.
         * 
         * @param properties_file The file containing surface properties.
         */
        void import_properties(std::string properties_file);

        /**
         * @brief Retrieves the mu coefficients for the surface.
         * 
         * @return The mu coefficients as a Matrix object.
         */
        Matrix<T>& get_mu_coefficients() { return mu_coefficients; }

        /**
         * @brief Retrieves the sigma coefficients for the surface.
         * 
         * @return The sigma coefficients as a Matrix object.
         */
        Matrix<T>& get_sigma_coefficients() { return sigma_coefficients; }

        /**
         * @brief Retrieves the geometry of the surface.
         * 
         * @return A reference to the geometry object of the surface.
         */
        Geometry<T>& get_geometry() { return this->geometry; }

        /**
         * @brief Retrieves the autocorrelation length of the surface.
         * 
         * @return The autocorrelation length of the surface.
         */
        T autocorrelation_length() { return ac_length; }

        /**
         * @brief Retrieves the local parameters of the surface.
         * 
         * @return A pointer to the local parameters array.
         */
        T* get_local_parameters() { return local_parameters; }
};

/**
 * @brief Default constructor for Surface.
 * 
 * Initializes the autocorrelation length to 0.
 */
template <typename T>
Surface<T>::Surface() : ac_length(0.0) {}

/**
 * @brief Parameterized constructor for Surface.
 * 
 * Initializes the geometry from a file and imports surface properties from another file.
 * 
 * @param geometry_file The file containing surface geometry data.
 * @param properties_file The file containing surface properties (e.g., mu, sigma coefficients).
 */
template <typename T>
Surface<T>::Surface(std::string geometry_file, std::string properties_file) {
    this->geometry = Geometry<T>(geometry_file);
    import_properties(properties_file);
}

/**
 * @brief Destructor for Surface.
 * 
 * Sets the autocorrelation length to 0 upon destruction.
 */
template <typename T>
Surface<T>::~Surface() {
    ac_length = 0.0;
}

/**
 * @brief Imports the surface properties from a file.
 * 
 * Reads the mu and sigma coefficients and autocorrelation length from the properties file
 * and stores them in the appropriate member variables.
 * 
 * @param properties_file The file containing surface properties.
 */
template <typename T>
void Surface<T>::import_properties(std::string properties_file) {

    unsigned int order, local_param_number;

    std::ifstream myfile;
    myfile.open(properties_file);
    myfile >> order;

    Matrix<T> mu_k(order, 1);
    Matrix<T> sigma_k(order, 1);

    // Read mu coefficients from the file.
    for(auto i = 0; i < order; i++) {
        myfile >> mu_k(i, 0);
    }

    // Read sigma coefficients from the file.
    for(auto i = 0; i < order; i++) {
        myfile >> sigma_k(i, 0);
    }

    // Read autocorrelation length from the file.
    myfile >> this->ac_length;

    this->mu_coefficients = mu_k;
    this->sigma_coefficients = sigma_k;

    myfile.close();
}
