/**
 * @file main.cpp
 * @brief Main program for running a ray tracing and kernel sampling simulation.
 * 
 * This program runs a single or batch simulation using a ray tracer and PG kernel. It parses command-line 
 * arguments to determine the simulation mode and project name. Results are saved after running the simulation.
 */

#include <iostream>
#include <fstream>
#include <cmath>
#include <random>
#include <string>
#include "Constants.h"
#include "Matrix.h"
#include "Hermite_Tools.h"
#include "Local_Kernels.h"
#include "Random_Sampler.h"
#include "Geometry.h"
#include "Gas.h"
#include "Ray.h"
#include "Surface.h"
#include "PG_Kernel.h"
#include "Raytracer.h"
#include "Importer.h"

/**
 * @brief Main function for ray tracing and kernel sampling simulation.
 * 
 * This function parses the command-line arguments, imports the project data, runs the kernel and ray tracing 
 * simulations, and saves the results to output files.
 * 
 * @param argc The number of command-line arguments.
 * @param argv The array of command-line arguments.
 * @return int Returns 0 upon successful completion.
 */
int main(int argc, char* argv[]) {

    bool batch = false; ///< Flag indicating if the simulation is run in batch mode.
    std::string project_name; ///< Name of the project for which the simulation is being run.

    // Parse command-line arguments to check for batch mode and project name.
    for(auto i = 0; i < argc; i++) {
        std::string arg = argv[i];
        if(arg == "--batch") batch = true; ///< Set batch mode if "--batch" argument is provided.
        if(arg.find("--project=") != std::string::npos) {
            project_name = arg.substr(10, arg.length()); ///< Extract project name from the "--project=" argument.
        }
    }

    // Import the project which contains the kernel and ray tracer data.
    std::tuple<PG_Kernel<double>, Raytracer<double>> project_tuple = import_single<double>(project_name);

    PG_Kernel<double> kernel = std::get<0>(project_tuple); ///< Extract the PG_Kernel object from the project tuple.
    Raytracer<double> raytracer = std::get<1>(project_tuple); ///< Extract the Raytracer object from the project tuple.

    // Run the kernel simulation.
    std::vector<trajectory<double>> trajectories_kernel = kernel.sample_batch();

    // Run the ray tracing simulation.
    std::vector<trajectory<double>> trajectories_raytracer = raytracer.simulate();

    // Save the results of the kernel simulation.
    kernel.save(trajectories_kernel, project_name + "/results/kernel_data.dat");

    // Save the results of the ray tracer simulation.
    raytracer.save(trajectories_raytracer, project_name + "/results/raytracer_data.dat");

    return 0;
}
