/**
 * @class Raytracer
 * @brief A class that simulates ray tracing through a surface and gas environment.
 * 
 * The Raytracer class is responsible for simulating the propagation of particles (rays) through 
 * a surface and gas medium, tracking their interactions, and saving the resulting trajectories.
 * 
 * @tparam T Type parameter for numerical data (typically float or double).
 */
template <typename T>
class Raytracer {

    private:

        /** 
         * @brief The surface through which rays will propagate.
         */
        Surface<T> surface;

        /** 
         * @brief The gas medium in which the rays interact.
         */
        Gas<T> gas;

        /** 
         * @brief Function pointer to the local kernel used for ray interactions with the gas.
         */
        void (*local_kernel)(T*, T*, T*, T*, T*);

        /** 
         * @brief Incident velocity matrix for the ray propagation.
         */
        Matrix<T> incident_velocity;

        /** 
         * @brief Name of the simulation.
         */
        std::string sim_name;

        /** 
         * @brief The number of particles (rays) to simulate.
         */
        unsigned long num_particles;

    public:

        /** 
         * @brief Default constructor.
         * 
         * Initializes the Raytracer class with default values.
         */
        Raytracer() {}

        /**
         * @brief Parameterized constructor for the Raytracer class.
         * 
         * Initializes the raytracer class with a surface, gas type, local kernel, incident particle velocity, 
         * number of particles, and a simulation name.
         * 
         * @param surface The surface object used by the simulation.
         * @param gas The gas object used by the simulation.
         * @param local_kernel The kernel describing the local dynamics of the gas-surface interaction.
         * @param incident_velocity The incident velocity vector.
         * @param num_particles The total number of rays to be simulated.
         * @param sim_name Name of the simulation.
         */
        Raytracer(Surface<T>& surface, Gas<T> gas, void (*local_kernel)(T*, T*, T*, T*, T*), Matrix<T> incident_velocity, unsigned long num_particles, std::string sim_name);

        /** 
         * @brief Destructor for the Raytracer class.
         */
        ~Raytracer() {}

        /**
         * @brief Simulates ray tracing and generates ray trajectories.
         * 
         * This method propagates the rays through the surface and gas medium, tracks their positions 
         * and velocities, and returns the trajectories for each particle.
         * 
         * @return A vector of trajectories, each containing the positions and velocities of the particles.
         */
        std::vector<trajectory<T>> simulate();

        /**
         * @brief Imports trajectory data from a file.
         * 
         * Reads simulation data from the specified file and returns the corresponding particle trajectories.
         * 
         * @param filename The name of the file containing trajectory data.
         * @return A vector of imported trajectories.
         */
        std::vector<trajectory<T>> import_data(std::string filename);

        /**
         * @brief Saves the simulated trajectories to a file.
         * 
         * This method saves the positions and velocities of each particle to the specified file.
         * 
         * @param trajectories The vector of particle trajectories to be saved.
         * @param filename The name of the file to save the data.
         */
        void save(std::vector<trajectory<T>> trajectories, std::string filename);

        /**
         * @brief Combines output files from multiple processes into a single file.
         * 
         * This method is used in parallel simulations to combine partial results from multiple 
         * processes into one final output file.
         * 
         * @param filename The base name of the files to combine.
         * @param num_procs The number of processes (files) to combine.
         */
        void combine_files(std::string filename, int num_procs);

        /**
         * @brief Sets the number of particles to simulate.
         * 
         * @param num_particles The number of particles for the simulation.
         */
        void set_num_particles(unsigned long num_particles) { this->num_particles = num_particles; }

        /**
         * @brief Retrieves the name of the simulation.
         * 
         * @return A reference to the simulation name.
         */
        std::string& get_sim_name() { return sim_name; }

        /**
         * @brief Retrieves the number of particles in the simulation.
         * 
         * @return The number of particles to be simulated.
         */
        unsigned long get_num_particles() { return num_particles; }

};

template <typename T>
Raytracer<T>::Raytracer(Surface<T>& surface, Gas<T> gas, void (*local_kernel)(T*, T*, T*, T*, T*), Matrix<T> incident_velocity, unsigned long num_particles, std::string sim_name) {

    this->surface = surface;
    this->gas = gas;
    this->local_kernel = local_kernel;
    this->incident_velocity = incident_velocity;
    this->num_particles = num_particles;
    this->sim_name = sim_name;

}

template <typename T>
std::vector<trajectory<T>> Raytracer<T>::simulate() {

    Matrix<T> unit_incident = this->incident_velocity / this->incident_velocity.norm() * (-1.0);

    T theta_i = std::acos(- this->incident_velocity(2, 0) / this->incident_velocity.norm());

    long num_col = 0;
    bool collided = true;
    unsigned int collision_index;

    std::vector<trajectory<T>> trajectories;

    std::random_device rd;
    std::seed_seq fullSeed{rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd()};
    std::mt19937 rng(fullSeed);
    std::uniform_real_distribution<T> uniformDist(0.0f, 1.0f);
    std::normal_distribution<T> normDist(0.0f, 1.0f);

    T offset_size = 15.0;

    for(long i = 0; i < this->num_particles; i ++) {
        
        std::vector<Matrix<T>> p_positions;
        std::vector<Matrix<T>> p_velocities;

        T m_dot = 1.0;
        num_col = 0;

        collided = true;

        T rotation_angle = uniformDist(rng) * 2.0 * M_PI;

        surface.get_geometry().rotate(2, rotation_angle);

        Matrix<T> origin = unit_incident / std::cos(theta_i) * 100.0;
        Matrix<T> offset(3, 1, {2.0 * (uniformDist(rng) - 0.5) * offset_size, 2.0 * (uniformDist(rng) - 0.5) * offset_size, 0.0});
        origin += offset;
        Ray<T> ray(origin, incident_velocity, m_dot, 1, -1);
        while (collided) {

            std::vector<Ray<T>> new_rays = ray.propagate(this->surface.get_geometry(), this->gas, this->local_kernel, &collision_index);
            if(new_rays.size() == 0) {
                collided = false;
            }
            else {
                ray = new_rays.back();
                p_velocities.push_back(ray.get_velocity());
                p_positions.push_back(ray.get_origin());
            }
        num_col ++;
        }

        trajectory<T> trajectory = {p_velocities, p_positions};
        trajectories.push_back(trajectory);
    }
    return trajectories;
}

template<typename T>
void Raytracer<T>::save(std::vector<trajectory<T>> trajectories, std::string filename) {

    std::ofstream myfile;
    myfile.open(filename);
    
    unsigned long num_particles = trajectories.size();

    for(auto i = 0; i < num_particles; i ++) {

        trajectory<T> p_trajectory = trajectories[i];
        unsigned long num_bounces = p_trajectory.positions.size();

        myfile << num_bounces << "\n";

        for(auto j = 0; j < 3; j ++) {
            for(auto k = 0; k < num_bounces; k ++) {
                myfile << p_trajectory.positions[k](j, 0) << " ";
            }
            myfile << "\n";
        }

        for(auto j = 0; j < 3; j ++) {
            for(auto k = 0; k < num_bounces; k ++) {
                myfile << p_trajectory.velocities[k](j, 0) << " ";
            }
            myfile << "\n";
        }

    }
    myfile.close();

}

template <typename T>
std::vector<trajectory<T>> Raytracer<T>::import_data(std::string filename) {

    std::ifstream linecounter(filename);
    std::string line;
    std::vector<trajectory<T>> trajectories;

    long num_lines = 0;

    while( std::getline(linecounter, line)) {
        ++ num_lines;
    }

    long num_particles = num_lines / 7;
    long num_bounces;

    std::ifstream myfile(filename);

    for(long i = 0; i < num_particles; i ++) {

        myfile >> num_bounces;
        Matrix<T> position(3, 1);
        Matrix<T> velocity(3, 1);
        std::vector<Matrix<T>> p_positions(num_bounces, Matrix<T>(3, 1));
        std::vector<Matrix<T>> p_velocities(num_bounces, Matrix<T>(3, 1));

        for(long k = 0; k < 3; k ++) {
            for(long j = 0; j < num_bounces; j ++) {
                myfile >> p_positions[j](k, 0);
            }
        }
         for(long k = 0; k < 3; k ++) {
            for(long j = 0; j < num_bounces; j ++) {
                myfile >> p_velocities[j](k, 0);
            }
        }
        trajectory<T> trajectory = {p_velocities, p_positions};
        trajectories.push_back(trajectory);
    }
    
    return trajectories;
}

template <typename T>
void Raytracer<T>::combine_files(std::string filename, int num_procs) {

    std::vector<trajectory<T>> trajectories_total;

    for( long i = 0; i < num_procs; i ++) {
        std::vector<trajectory<T>> trajectories = import_data(filename + "_" + std::to_string(i) + ".dat");
        trajectories_total.insert(trajectories_total.end(), trajectories.begin(), trajectories.end());
        std::remove((filename + "_" + std::to_string(i) + ".dat").c_str()); 
    }
    save(trajectories_total, filename + ".dat");
}