/**
 * @brief Computes the sign of a value.
 * 
 * This function returns -1 if the input value is negative, 1 if the input value is positive, 
 * and 0 if the input value is zero.
 * 
 * @tparam T Type of the input value (must be comparable).
 * @param val The value whose sign is to be determined.
 * @return int -1 if val < 0, 1 if val > 0, and 0 if val == 0.
 */
template <typename T>
int sgn(T val) {
    return (T(0) < val) - (val < T(0));
}

/**
 * @brief Samples a reflection velocity based on the CLL (Cercignani-Lampis-Lord) model.
 * 
 * This function computes the reflection velocity (`vel_refl`) of a particle 
 * based on its input velocity (`vel_in`) and various parameters related to gas, 
 * surface, and GSI (Gas-Surface Interaction) models. The CLL model incorporates 
 * tangential and normal accommodation coefficients.
 * 
 * @tparam T Type of the input values (e.g., float, double).
 * @param vel_in Input velocity vector (3 components).
 * @param vel_refl Output reflected velocity vector (3 components).
 * @param GSI_parameters Parameters of the Gas-Surface Interaction (GSI) model:
 *        - GSI_parameters[0]: alpha_n (normal accommodation coefficient),
 *        - GSI_parameters[1]: sigma_t (tangential momentum accommodation coefficient).
 * @param Gas_parameters Parameters related to the gas:
 *        - Gas_parameters[2]: Molar mass of the gas.
 * @param Surface_parameters Parameters related to the surface:
 *        - Surface_parameters[0]: T_W (surface temperature).
 */
template <typename T>
void sample_CLL(T* vel_in, T* vel_refl, T* GSI_parameters, T* Gas_parameters, T* Surface_parameters) {
    
    // Retrieve GSI and surface parameters
    T alpha_n = GSI_parameters[0]; // Normal accommodation coefficient
    T sigma_t = GSI_parameters[1]; // Tangential momentum accommodation coefficient
    T alpha_t = sigma_t * (2 - sigma_t); // Tangential accommodation coefficient
    T T_W = Surface_parameters[0]; // Wall temperature

    // Random number generator setup
    std::random_device rd;
    std::seed_seq fullSeed{rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd()};
    std::mt19937 rng(fullSeed); // Random number generator
    std::uniform_real_distribution<T> uniformDist(0.0f, 1.0f); // Uniform distribution in [0, 1]

    // Calculate velocity scale based on gas properties and surface temperature
    T vel_wall = sqrt(2.0 * R_gas / Gas_parameters[2] * T_W);

    // Generate random numbers for sampling
    T x1 = uniformDist(rng), x2 = uniformDist(rng), x3 = uniformDist(rng), 
      x4 = uniformDist(rng), x5 = uniformDist(rng), x6 = uniformDist(rng);

    // Calculate reflection parameters
    T r1 = sqrt(-alpha_n * log(x1));
    T r3 = sqrt(-alpha_t * log(x3));
    T r5 = sqrt(-alpha_t * log(x5));
    T theta2 = 2.0 * M_PI * x2;
    T theta4 = 2.0 * M_PI * x4;
    T theta6 = 2.0 * M_PI * x6;

    // Compute the magnitude of the input velocity components
    T vel_norm_m = fabs(vel_in[2] / vel_wall) * sqrt(1 - alpha_n);
    T vel_tan1_m = vel_in[0] / vel_wall * sqrt(1 - alpha_t);

    // Reflect the velocity in normal and tangential directions
    vel_refl[2] = vel_wall * sqrt(r1 * r1 + vel_norm_m * vel_norm_m + 2 * r1 * vel_norm_m * cos(theta2)); // Normal component
    vel_refl[0] = vel_wall * (vel_tan1_m + r3 * cos(theta4)); // Tangential component 1
    vel_refl[1] = vel_wall * (r5 * cos(theta6)); // Tangential component 2
}

/**
 * @brief Samples a reflection velocity based on the DRIA (Diffuse Reflection Incomplete Accommodation) model.
 * 
 * This function computes the reflection velocity (`vel_refl`) of a particle 
 * based on its input velocity (`vel_in`) and various parameters related to gas, 
 * surface, and GSI models. The DRIA model assumes incomplete accommodation of 
 * the incoming velocity to the surface.
 * 
 * @tparam T Type of the input values (e.g., float, double).
 * @param vel_in Input velocity vector (3 components).
 * @param vel_refl Output reflected velocity vector (3 components).
 * @param GSI_parameters Parameters of the Gas-Surface Interaction (GSI) model:
 *        - GSI_parameters[0]: alpha (accommodation coefficient).
 * @param Gas_parameters Parameters related to the gas:
 *        - Gas_parameters[0]: T_G (gas temperature),
 *        - Gas_parameters[2]: Molar mass of the gas,
 *        - Gas_parameters[3]: Gas speed.
 * @param Surface_parameters Parameters related to the surface:
 *        - Surface_parameters[0]: T_W (surface temperature).
 */
template <typename T>
void sample_DRIA(T* vel_in, T* vel_refl, T* GSI_parameters, T* Gas_parameters, T* Surface_parameters) {

    // Compute the magnitude of the input velocity
    T vel_in_norm = sqrt(vel_in[0] * vel_in[0] + vel_in[1] * vel_in[1] + vel_in[2] * vel_in[2]);

    // Retrieve GSI and surface parameters
    T alpha = GSI_parameters[0]; // Accommodation coefficient
    T T_W = Surface_parameters[0]; // Wall temperature
    T T_G = Gas_parameters[0]; // Gas temperature
    T gas_speed = Gas_parameters[3]; // Gas speed

    // Compute characteristic gas temperature based on speed
    T T_K = gas_speed * gas_speed * Gas_parameters[2] / 3.0 / R_gas;

    // Random number generator setup
    std::random_device rd;
    std::seed_seq fullSeed{rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd()};
    std::mt19937 rng(fullSeed); // Random number generator
    std::uniform_real_distribution<T> uniformDist(0.0f, 1.0f); // Uniform distribution in [0, 1]
    std::normal_distribution<T> normDist(0.0f, 1.0f); // Normal distribution for random velocities

    // Compute reflected temperature based on accommodation coefficient
    T T_R = (1.0 - alpha) * T_K + alpha * T_W;

    // Calculate velocity scale based on gas properties and reflected temperature
    T vel_wall = sqrt(2.0 * R_gas / Gas_parameters[2] * T_R);

    // Generate random numbers for sampling
    T x1 = uniformDist(rng), x2 = uniformDist(rng), x3 = uniformDist(rng), 
      x4 = uniformDist(rng), x5 = uniformDist(rng), x6 = uniformDist(rng);

    // Calculate reflection parameters
    T r1 = sqrt(-log(x1));
    T r3 = sqrt(-log(x3));
    T r5 = sqrt(-log(x5));
    T theta2 = 2.0 * M_PI * x2;
    T theta4 = 2.0 * M_PI * x4;
    T theta6 = 2.0 * M_PI * x6;

    // Reflect the velocity in normal and tangential directions
    vel_refl[2] = vel_wall * r1; // Normal component
    vel_refl[0] = sqrt(R_gas * T_R / Gas_parameters[2]) * normDist(rng); // Tangential component 1
    vel_refl[1] = sqrt(R_gas * T_R / Gas_parameters[2]) * normDist(rng); // Tangential component 2
}
