/**
 * @brief Class representing a Gas with properties such as density, temperature, molar mass, and speed.
 * 
 * This class models the behavior of a gas and provides methods to calculate gas properties such as 
 * thermal velocity.
 * 
 * @tparam T The data type used for gas properties (e.g., float, double).
 */
template <typename T>
class Gas {

    private:
        T density;      ///< Density of the gas (in kg/m³).
        T temperature;  ///< Temperature of the gas (in K).
        T molar_mass;   ///< Molar mass of the gas (in kg/mol).
        T speed;        ///< Speed of the gas (in m/s).

    public:

        /**
         * @brief Default constructor for the Gas class.
         * 
         * Initializes the gas properties to zero.
         */
        Gas() : density(0.0), temperature(0.0), molar_mass(0.0), speed(0.0) {}

        /**
         * @brief Parameterized constructor for the Gas class.
         * 
         * Initializes the gas properties with specified values.
         * 
         * @param density Initial density of the gas.
         * @param temperature Initial temperature of the gas.
         * @param molar_mass Initial molar mass of the gas.
         * @param speed Initial speed of the gas.
         */
        Gas(T density, T temperature, T molar_mass, T speed) : density(density), temperature(temperature), molar_mass(molar_mass), speed(speed) {}

        /**
         * @brief Copy constructor for the Gas class.
         * 
         * Initializes a new gas object by copying properties from another gas object.
         * 
         * @param other The gas object to copy from.
         */
        Gas(const Gas<T>& other) : density(other.density), temperature(other.temperature), molar_mass(other.molar_mass), speed(other.speed) {}

        /**
         * @brief Move constructor for the Gas class.
         * 
         * Initializes a new gas object by moving properties from another gas object.
         * The source object is reset to zero after the move.
         * 
         * @param other The gas object to move from.
         */
        Gas(Gas<T>&& other) : density(other.density), temperature(other.temperature), molar_mass(other.molar_mass), speed(other.speed) {
            other.density = 0.0;
            other.temperature = 0.0;
            other.molar_mass = 0.0;
            other.speed = 0.0;
        }

        /**
         * @brief Destructor for the Gas class.
         * 
         * Resets the gas properties to zero.
         */
        ~Gas() {
            density = 0.0;
            temperature = 0.0;
            molar_mass = 0.0;
            speed = 0.0;
        }

        /**
         * @brief Copy assignment operator for the Gas class.
         * 
         * Copies the properties from another gas object.
         * 
         * @param other The gas object to copy from.
         * @return Gas& Reference to the current gas object.
         */
        Gas& operator=(const Gas& other) {
            density = other.density;
            temperature = other.temperature;
            molar_mass = other.molar_mass;
            speed = other.speed;
            return *this;
        }

        /**
         * @brief Move assignment operator for the Gas class.
         * 
         * Moves the properties from another gas object, and resets the source object to zero.
         * 
         * @param other The gas object to move from.
         * @return Gas& Reference to the current gas object.
         */
        Gas& operator=(Gas&& other) {
            density = other.density;
            temperature = other.temperature;
            molar_mass = other.molar_mass;
            speed = other.speed;

            other.density = 0.0;
            other.temperature = 0.0;
            other.molar_mass = 0.0;
            other.speed = 0.0;
            return *this;
        }

        /**
         * @brief Computes the thermal velocity of the gas in three dimensions.
         * 
         * This function generates random velocities in the x, y, and z directions based on 
         * a normal distribution and the temperature and molar mass of the gas.
         * 
         * @return Matrix<T> A 3x1 matrix containing the thermal velocity components (v_x, v_y, v_z).
         */
        Matrix<T> get_thermal_velocity();

        /**
         * @brief Returns the temperature of the gas.
         * 
         * @return T The current temperature of the gas.
         */
        T get_temperature() const { return temperature; }

        /**
         * @brief Returns the density of the gas.
         * 
         * @return T The current density of the gas.
         */
        T get_density() const { return density; }

        /**
         * @brief Returns the molar mass of the gas.
         * 
         * @return T The current molar mass of the gas.
         */
        T get_molar_mass() const { return molar_mass; }

        /**
         * @brief Returns the speed of the gas.
         * 
         * @return T The current speed of the gas.
         */
        T get_speed() const { return speed; }
};

/**
 * @brief Computes the thermal velocity of the gas.
 * 
 * This function generates random thermal velocities in the x, y, and z directions using
 * a normal distribution based on the gas temperature and molar mass.
 * 
 * @return Matrix<T> A 3x1 matrix containing the thermal velocity components (v_x, v_y, v_z).
 */
template <typename T>
Matrix<T> Gas<T>::get_thermal_velocity() {
    // Set up random number generation
    std::random_device rd;
    std::seed_seq fullSeed{rd(), rd(), rd(), rd(), rd(), rd(), rd(), rd()};
    std::mt19937 rng(fullSeed);
    std::normal_distribution<T> normDist(0.0f, 1.0f);

    // Compute the thermal velocities in each direction
    T v_x = sqrt(R_gas * temperature / molar_mass) * normDist(rng);
    T v_y = sqrt(R_gas * temperature / molar_mass) * normDist(rng);
    T v_z = sqrt(R_gas * temperature / molar_mass) * normDist(rng);

    // Return the thermal velocity as a 3x1 matrix
    Matrix<T> vel(3, 1, {v_x, v_y, v_z});
    return vel;
}
