classdef MiscTest < matlab.unittest.TestCase
    % MISCTEST  Contains unit tests for loose functions.


    methods (Test) % copy_into
        function copy_into__out_is_not_handle_if_to_is_not_handle(testCase)
            from = struct();
            to = struct(title = "foo");

            out = copy_into(from, to);

            to.title = "bar";  %#ok<STRNU> Intentional
            testCase.verifyEqual(out.title, "foo");
        end

        function copy_into__empty_struct_to_empty_struct(testCase)
            from = struct();
            to = struct();

            out = copy_into(from, to);

            testCase.verifyEqual(numel(fieldnames(out)), 0);
        end

        function copy_into__from_has_new_field(testCase)
            from = struct(foo = "bar");
            to = struct();

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, "bar");
        end

        function copy_into__to_has_old_field(testCase)
            from = struct();
            to = struct(foo = "bar");

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, "bar");
        end

        function copy_into__from_and_to_have_different_fields(testCase)
            from = struct(foo = "bar");
            to = struct(baz = "qux");

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, "bar");
            testCase.verifyEqual(out.baz, "qux");
        end

        function copy_into__from_overwrites_field(testCase)
            from = struct(foo = "baz");
            to = struct(foo = "bar");

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, "baz");
        end

        function copy_into__from_has_new_struct_field(testCase)
            from = struct(foo = struct(bar = "baz"));
            to = struct();

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, struct(bar = "baz"));
        end

        function copy_into__struct_field_overwrites_non_struct_field(testCase)
            from = struct(foo = struct(bar = "baz"));
            to = struct(foo = "qux");

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, struct(bar = "baz"));
        end

        function copy_into__struct_field_gets_merged(testCase)
            from = struct(foo = struct(bar = "baz", qux = "quux"));
            to = struct(foo = struct(qux = "quuux", plugh = "xyzzy"));

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, struct(bar = "baz", qux = "quux", plugh = "xyzzy"));
        end

        function copy_into__from_has_new_array_field(testCase)
            from = struct(foo = [1, 2, 3]);
            to = struct();

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 2, 3]);
        end

        function copy_into__array_field_overwrites_non_array_field(testCase)
            from = struct(foo = [1, 2, 3]);
            to = struct(foo = "bar");

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 2, 3]);
        end

        function copy_into__array_field_overwrites_smaller_array_field(testCase)
            from = struct(foo = [1, 2, nan, 4]);
            to = struct(foo = [5, 6]);

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 2, nan, 4]);
        end

        function copy_into__array_field_overwrites_larger_array_field(testCase)
            from = struct(foo = [1, 2]);
            to = struct(foo = [3, 4, 5, 6]);

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 2]);
        end

        function copy_into__array_field_overwrites_same_size_array_field(testCase)
            from = struct(foo = [1, 2, 3]);
            to = struct(foo = [4, 5, 6]);

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 2, 3]);
        end

        function copy_into__array_field_merges_non_nan_values_in_same_size_array(testCase)
            from = struct(foo = [1, nan, 2, nan, 3]);
            to = struct(foo = [4, 5, 6, 7, 8]);

            out = copy_into(from, to);

            testCase.verifyEqual(out.foo, [1, 5, 2, 7, 3]);
        end

        function copy_into__out_is_handle_if_to_is_handle(testCase)
            from = struct();
            to = PlotConfig(struct(title = "foo"));

            out = copy_into(from, to);

            to.title = "bar";
            testCase.verifyEqual(out.title, "bar");
        end
    end

    methods (Test) % hash
        function hash_self(testCase)
            value = "a";

            testCase.verifyEqual(hash(value), hash(value));
        end

        function hash_same(testCase)
            testCase.verifyEqual(hash("a"), hash("a"));
        end

        function hash_different(testCase)
            testCase.verifyNotEqual(hash("a"), hash("b"));
        end

        function hash_struct_self(testCase)
            s = struct(a = 1, b = 1);

            testCase.verifyEqual(hash(s), hash(s));
        end

        function hash_struct_same(testCase)
            s1 = struct(a = 1, b = 1);
            s2 = struct(a = 1, b = 1);

            testCase.verifyEqual(hash(s1), hash(s2));
        end

        function hash_struct_different_values(testCase)
            s1 = struct(a = 1, b = 1);
            s2 = struct(a = 1, b = 2);

            testCase.verifyNotEqual(hash(s1), hash(s2));
        end

        function hash_struct_different_fields(testCase)
            s1 = struct(a = 1, b = 2);
            s2 = struct(a = 1, c = 2);

            testCase.verifyNotEqual(hash(s1), hash(s2));
        end

        function hash_struct_different_recursion(testCase)
            s1 = struct(a = 1, b = 2);
            s2 = struct(a = 1, b = struct(c = 2));

            testCase.verifyNotEqual(hash(s1), hash(s2));
        end
    end

    methods (Test) % max_rng
        function max_rng_empty(testCase)
            [value, idx] = max_rng([]);

            testCase.verifyEmpty(value);
            testCase.verifyEmpty(idx);
        end

        function max_rng_single(testCase)
            [value, idx] = max_rng(5);

            testCase.verifyEqual(value, 5);
            testCase.verifyEqual(idx, 1);
        end

        function max_rng_row(testCase)
            [value, idx] = max_rng([5, 3, 4, 7, 1]);

            testCase.verifyEqual(value, 7);
            testCase.verifyEqual(idx, 4);
        end

        function max_rng_column(testCase)
            [value, idx] = max_rng([6; 1; 8; 2; 3; 5]);

            testCase.verifyEqual(value, 8);
            testCase.verifyEqual(idx, 3);
        end

        function max_rng_square(testCase)
            [value, idx] = max_rng([8, 1, 2; 4, 3, 9; 7, 6, 5]);

            testCase.verifyEqual(value, 9);
            testCase.verifyEqual(idx, 8);
        end

        function max_rng_multiple(testCase)
            rng("shuffle");
            r = 100;

            [values, idxs] = arrayfun(@(~) max_rng([7, 5, 3, 7, 2, 1, 4]), 1:r);

            testCase.verifyEqual(unique(values), 7, info());
            testCase.verifyEqual(unique(idxs), [1, 4], info());
        end
    end

    methods (Test) % min_rng
        function min_rng_empty(testCase)
            [value, idx] = min_rng([]);

            testCase.verifyEmpty(value);
            testCase.verifyEmpty(idx);
        end

        function min_rng_single(testCase)
            [value, idx] = min_rng(5);

            testCase.verifyEqual(value, 5);
            testCase.verifyEqual(idx, 1);
        end

        function min_rng_row(testCase)
            [value, idx] = min_rng([4, 6, 2, 1]);

            testCase.verifyEqual(value, 1);
            testCase.verifyEqual(idx, 4);
        end

        function min_rng_column(testCase)
            [value, idx] = min_rng([6; 2; 4; 8; 9; 5]);

            testCase.verifyEqual(value, 2);
            testCase.verifyEqual(idx, 2);
        end

        function min_rng_square(testCase)
            [value, idx] = min_rng([4, 8, 9; 1, 3, 5; 6, 7, 2]);

            testCase.verifyEqual(value, 1);
            testCase.verifyEqual(idx, 2);
        end

        function min_rng_multiple(testCase)
            rng("shuffle");
            r = 100;

            [values, idxs] = arrayfun(@(~) min_rng([5, 6, 5, 2, 2, 3]), 1:r);

            testCase.verifyEqual(unique(values), 2, info());
            testCase.verifyEqual(unique(idxs), [4, 5], info());
        end
    end

    methods (Test) % mustContainSubstring
        function mustContainSubstring_error(testCase)
            testCase.verifyError(@() mustContainSubstring("aeiou", "q"), "");
        end
    end

    methods (Test) % randelem
        function randelem(testCase)
            rng("shuffle");
            r = 100;

            values = [5, 7, 2];
            samples = arrayfun(@(~) randelem(values), 1:r);

            testCase.verifyEqual(unique(samples), unique(values), info());
        end
    end

    methods (Test) % randf
        function randf_size(testCase)
            size = [14, 8, 4, 8];
            values = randf([0, 1], size);
            testCase.verifySize(values, size);
        end

        function randf_size_dimensions(testCase)
            size = [1, 1, 1, 1, 5, 1, 1];
            values = randf([0, 1], size);
            testCase.verifySize(values, size);
        end

        function randf_range(testCase)
            values = randf([14, 38], [1, 1000000]);
            testCase.verifyEqual(min(values), 14, AbsTol = 0.001);
            testCase.verifyEqual(max(values), 38, AbsTol = 0.001);
        end
    end

    methods (Test) % rgb2hex
        function rgb2hex_empty(testCase)
            testCase.verifyEqual(rgb2hex([]), "" + zeros([0, 1]));
        end

        function rgb2hex_zeroes(testCase)
            testCase.verifyEqual(rgb2hex([0, 0, 0]), "#000000");
        end

        function rgb2hex_ffs(testCase)
            testCase.verifyEqual(rgb2hex([1, 1, 1]), "#FFFFFF");
        end

        function rgb2hex_single(testCase)
            testCase.verifyEqual(rgb2hex([128, 255, 64] ./ 255), "#80FF40");
        end

        function rgb2hex_multiple(testCase)
            testCase.verifyEqual(rgb2hex([128, 255, 64; 115, 185, 201] ./ 255), ["#80FF40"; "#73B9C9"]);
        end
    end
end
