classdef GraphsTest < matlab.unittest.TestCase
    % GRAPHSTEST  Unit tests for [Graphs].


    methods (Test) % is_unweighted
        function is_unweighted_double_empty(testCase)
            G = graph(ones(0));

            testCase.verifyFalse(Graphs.is_unweighted(G));
        end

        function is_unweighted_double(testCase)
            A = ones(5);
            A(eye(height(A), "logical")) = 0;
            G = graph(A);

            testCase.verifyFalse(Graphs.is_unweighted(G));
        end

        function is_unweighted_logical_empty(testCase)
            G = graph(false(0));

            testCase.verifyTrue(Graphs.is_unweighted(G));
        end

        function is_unweighted_logical(testCase)
            A = true(5);
            A(eye(height(A), "logical")) = 0;
            G = graph(A);

            testCase.verifyTrue(Graphs.is_unweighted(G));
        end
    end

    methods (Test) % is_simple
        function is_simple_empty(testCase)
            G = graph([]);

            testCase.verifyTrue(Graphs.is_simple(G));
        end

        function is_simple_complete(testCase)
            A = true(8);
            A(eye(height(A), "logical")) = 0;
            G = graph(A);

            testCase.verifyTrue(Graphs.is_simple(G));
        end

        function is_simple_self_loop(testCase)
            G = graph(false(11));
            G = addedge(G, 1, 1);

            testCase.verifyFalse(Graphs.is_simple(G));
        end

        function is_simple_double_edge(testCase)
            G = graph(false(8));
            G = addedge(G, [1, 1], [2, 2]);

            testCase.verifyFalse(Graphs.is_simple(G));
        end

        function is_simple_weighted(testCase)
            G = graph(zeros(6));

            testCase.assertFalse(Graphs.is_unweighted(G));
            testCase.verifyTrue(Graphs.is_simple(G));
        end

        function is_simple_half_edges(testCase)
            G = graph(false(4));
            G = addedge(G, [2, 2], [3, 3], [0.5, 0.5]);

            testCase.verifyFalse(Graphs.is_simple(G));
        end
    end

    methods (Test) % is_connected
        function is_connected_empty(testCase)
            G = graph([]);

            testCase.verifyError(@() Graphs.is_connected(G), "MATLAB:assertion:failed");
        end

        function is_connected_without_edges(testCase)
            G = graph(false(10));

            testCase.verifyFalse(Graphs.is_connected(G));
        end

        function is_connected_complete(testCase)
            A = true(8);
            A(eye(height(A), "logical")) = 1;
            G = graph(A);

            testCase.verifyTrue(Graphs.is_connected(G));
        end

        function is_connected_components(testCase)
            A = true(5);
            A(eye(height(A), "logical")) = 1;
            G = graph(blkdiag(A, A));

            testCase.verifyFalse(Graphs.is_connected(G));
        end
    end


    methods (Test) % generate_empty
        function generate_empty(testCase)
            r = 25;
            ns = @() randi([1, 500]);

            for i = 1:r
                rng("shuffle");

                n = ns();
                G = Graphs.generate_empty(n);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyEqual(numedges(G), 0, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end

    methods (Test) % generate_complete
        function generate_complete(testCase)
            r = 25;
            ns = @() randi([1, 500]);

            for i = 1:r
                rng("shuffle");

                n = ns();
                G = Graphs.generate_complete(n);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyEqual(numedges(G), n * (n - 1) / 2, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end

    methods (Test) % generate_erdos_renyi
        function generate_erdos_renyi(testCase)
            r = 50;
            ns = @() randi([1, 500]);
            ps = @(n) randf([0, 1]);

            for i = 1:r
                rng("shuffle");

                n = ns();
                p = ps(n);
                G = Graphs.generate_erdos_renyi(n, p);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end

    methods (Test) % generate_watts_strogatz
        function generate_watts_strogatz(testCase)
            r = 25;
            ns = @() randi([1, 500]);
            ks = @(n) randi([0, floor((n - 1) / 2)]);
            ps = @(n) randf([0, 1]);

            for i = 1:r
                rng("shuffle");

                n = ns();
                k = ks(n);
                p = ps(n);
                G = Graphs.generate_watts_strogatz(n, k, p);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyEqual(numedges(G), n * k, info(i));
                testCase.verifyEqual(mean(degree(G)), 2 * k, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end

    methods (Test) % generate_barabasi_albert
        function generate_barabasi_albert(testCase)
            r = 25;
            ns = @() randi([1, 500]);
            ms = @(n) randi([0, n - 1]);

            for i = 1:r
                rng("shuffle");

                n = ns();
                m = ms(n);
                G = Graphs.generate_barabasi_albert(n, m);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyEqual(numedges(G), m + (n - (m + 2) + 1) * m, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end

    methods (Test) % generate_geometric_random
        function generate_geometric_random(testCase)
            r = 50;
            ds = @() randi([1, 10]);
            ns = @() randi([1, 500]);
            rs = @(n, d) randf([0, sqrt(d)]);

            for i = 1:r
                rng("shuffle");

                d = ds();
                n = ns();
                r = rs(n, d);
                G = Graphs.generate_geometric_random(d, n, r);

                testCase.verifyEqual(numnodes(G), n, info(i));
                testCase.verifyTrue(Graphs.is_simple(G), info(i));
                testCase.verifyTrue(Graphs.is_unweighted(G), info(i));
            end
        end
    end


    methods (Test) % fully_connected_graph_cycle_count
        function fully_connected_graph_cycle_count_low_length(testCase)
            testCase.verifyError(@() Graphs.fully_connected_graph_cycle_count(4, 2), "");
            testCase.verifyError(@() Graphs.fully_connected_graph_cycle_count(9, 0), "");
            testCase.verifyError(@() Graphs.fully_connected_graph_cycle_count(6, -3), "");
        end

        function fully_connected_graph_cycle_count_various(testCase)
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(3, 3), 1);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(3, 4), 0);

            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(4, 3), 4);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(4, 4), 3);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(4, 5), 0);

            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(5, 3), 10);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(5, 4), 15);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(5, 5), 12);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(5, 6), 0);

            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(6, 3), 20);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(6, 4), 45);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(6, 5), 72);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(6, 6), 60);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(6, 7), 0);

            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 3), 35);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 4), 105);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 5), 252);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 6), 420);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 7), 360);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(7, 8), 0);

            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 3), 56);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 4), 210);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 5), 672);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 6), 1680);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 7), 2880);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 8), 2520);
            testCase.verifyEqual(Graphs.fully_connected_graph_cycle_count(8, 9), 0);
        end
    end

    methods (Test) % girth
        function girth_empty_one_node(testCase)
            G = Graphs.generate_empty(1);

            testCase.verifyEqual(Graphs.girth(G), Inf);
        end

        function girth_empty_multiple_nodes(testCase)
            G = Graphs.generate_empty(7);

            testCase.verifyEqual(Graphs.girth(G), Inf);
        end

        function girth_complete_has_triangle(testCase)
            G = Graphs.generate_complete(6);

            testCase.verifyEqual(Graphs.girth(G), 3);
        end

        function girth_takes_shortest_cycle(testCase)
            e = [1 2; 2 3; 3 4; 4 5; 5 6; 6 7; 7 1; 4 1];
            G = graph(e(:, 1), e(:, 2));

            testCase.verifyEqual(Graphs.girth(G), 4);
        end
    end


    methods (Test) % stretched
        function stretched_removes_nothing_for_low_girth(testCase)
            G1 = Graphs.generate_erdos_renyi(13, 0.5);

            G2 = Graphs.stretch(G1, girth = 2);

            testCase.verifyTrue(isisomorphic(G1, G2));
        end

        function stretched_removes_all_cycles(testCase)
            G1 = Graphs.generate_erdos_renyi(13, 0.3);

            G2 = Graphs.stretch(G1, girth = -1);

            testCase.verifyEqual(Graphs.girth(G2), Inf);
        end

        function stretched_removes_different_numbers_of_edges(testCase)
            r = 50;

            for i = 1:r
                rng("shuffle");

                G = Graphs.generate_erdos_renyi(25, 0.5);

                G1 = Graphs.stretch(G, girth = 5, method = "least_cycles_steps");
                G2 = Graphs.stretch(G, girth = 5, method = "most_cycles_steps");

                testCase.verifyLessThan(numedges(G1), numedges(G2), info(i));
            end
        end

        function stretched_removes_edges_in_steps_suboptimally(testCase)
            r = 25;

            for i = 1:r
                rng("shuffle");

                G = Graphs.generate_erdos_renyi(25, 0.7);

                G1 = Graphs.stretch(G, girth = 5, method = "most_cycles_steps");
                G2 = Graphs.stretch(G, girth = 5, method = "most_cycles");

                % Note: This test is ever so slightly flaky! Don't worry!
                testCase.verifyLessThanOrEqual(numedges(G1), numedges(G2), info(i));
            end
        end
    end


    methods (Test) % update_adjacency
        function update_adjacency_add_nonexisting_edge(testCase)
            A = false(5, 5);

            A_upd = Graphs.update_adjacency(A, 2, 4, true);
            G_upd = addedge(graph(A), 2, 4);

            testCase.verifyEqual(A_upd, full(logical(adjacency(G_upd))));
        end

        function update_adjacency_remove_existing_edge(testCase)
            A = false(8, 8);
            A(2, 8) = 1;
            A(8, 2) = 1;

            A_upd = Graphs.update_adjacency(A, 2, 8, false);
            G_upd = rmedge(graph(A), 2, 8);

            testCase.verifyEqual(A_upd, full(logical(adjacency(G_upd))));
        end
    end

    methods (Test) % update_degree
        function update_degree_add_nonexisting_edge(testCase)
            G = graph([1, 4, 7, 8], [2, 6, 4, 7]);
            d = degree(G);

            d_upd = Graphs.update_degree(d, 2, 8, true);
            G_upd = addedge(G, 2, 8);

            testCase.verifyEqual(d_upd, degree(G_upd));
        end

        function update_degree_remove_existing_edge(testCase)
            e = [1 2; 1 3; 2 6; 4 5; 6 4];
            G = graph(e(:, 1), e(:, 2));
            d = degree(G);

            d_upd = Graphs.update_degree(d, 2, 6, false);
            G_upd = rmedge(G, 2, 6);

            testCase.verifyEqual(d_upd, degree(G_upd));
        end
    end

    methods (Test) % update_distances
        function update_distances_add_nonexisting_edge(testCase)
            G = Graphs.generate_watts_strogatz(6, 2, 0);  % 6-node ring lattice, with regular degree 4
            D = distances(G);

            D_upd = Graphs.update_distances(D, adjacency(G), 1, 4, true);
            G_upd = addedge(G, 1, 4);

            testCase.verifyNotEqual(D, D_upd);
            testCase.verifyEqual(D_upd, distances(G_upd));
        end

        function update_distances_remove_existing_edge(testCase)
            G = Graphs.generate_watts_strogatz(6, 2, 0);  % 6-node ring lattice, with regular degree 4
            D = distances(G);

            D_upd = Graphs.update_distances(D, adjacency(rmedge(G, 1, 3)), 1, 3, false);
            G_upd = rmedge(G, 1, 3);

            testCase.verifyNotEqual(D, D_upd);
            testCase.verifyEqual(D_upd, distances(G_upd));
        end
    end
end
