classdef ConfigTest < matlab.unittest.TestCase
    % CONFIGTEST  Unit tests for [Config].

    methods (Test) % general
        function config_is_handle(testCase)
            conf1 = DummyConfig(struct(foo = 1));

            conf2 = conf1;
            conf2.foo = 2;

            testCase.verifyEqual(conf1.foo, 2);
        end
    end


    methods (Test) % constructor
        function constructor_struct_empty(testCase)
            conf = DummyConfig(struct());

            testCase.verifyEqual(conf.foo, "foo");
        end

        function constructor_struct_incomplete(testCase)
            conf = DummyConfig(struct(foo = "dog"));

            testCase.verifyEqual(conf.foo, "dog");
        end

        function constructor_struct_complete(testCase)
            conf = DummyConfig(struct(foo = "one", bar = "two"));

            testCase.verifyEqual(conf.foo, "one");
            testCase.verifyEqual(conf.bar, "two");
        end
    end

    methods (Test) % set
        function set_overrides_single(testCase)
            conf = DummyConfig(struct(foo = "foo1"));

            new_conf = conf.set(struct(foo = "foo2"));

            testCase.verifyEqual(conf.foo, "foo2");
            testCase.verifyEqual(new_conf.foo, "foo2");
        end

        function set_overrides_multiple(testCase)
            conf = DummyConfig(struct(foo = "foo1", bar = "bar1"));

            new_conf = conf.set(struct(foo = "foo2", bar = "bar2"));

            testCase.verifyEqual(conf.foo, "foo2");
            testCase.verifyEqual(conf.bar, "bar2");
            testCase.verifyEqual(new_conf.foo, "foo2");
            testCase.verifyEqual(new_conf.bar, "bar2");
        end
    end

    methods (Test) % cache_cull
        function cache_cull_values(testCase)
            conf = DummyConfig(struct(foo = "foo1", bar = "bar1"));

            culled = conf.cache_cull();

            testCase.verifyEqual(culled.foo, "foo1");
            testCase.verifyEqual(culled.bar, "bar1");
        end

        function cache_cull_field_names(testCase)
            conf = DummyConfig(struct());

            culled = conf.cache_cull();

            testCase.verifyEqual(fieldnames(culled), {'foo'; 'bar'; 'ignored'});
        end

        function cache_cull_ignores(testCase)
            conf = DummyConfig(struct(ignored = "other"));

            culled = conf.cache_cull();

            testCase.verifyNotEqual(culled.ignored, "other");
        end
    end

    methods (Test) % cache_id
        function cache_id_self(testCase)
            conf = DummyConfig(struct());

            testCase.verifyEqual(conf.cache_id(), conf.cache_id());
        end

        function cache_id_same(testCase)
            conf1 = DummyConfig(struct(foo = "foo1"));
            conf2 = DummyConfig(struct(foo = "foo1"));

            testCase.verifyEqual(conf1.cache_id(), conf2.cache_id());
        end

        function cache_id_same_except_ignored(testCase)
            conf1 = DummyConfig(struct(ignored = "ignored1"));
            conf2 = DummyConfig(struct(ignored = "ignored2"));

            testCase.verifyEqual(conf1.cache_id(), conf2.cache_id());
        end

        function cache_id_different(testCase)
            conf1 = DummyConfig(struct(foo = "foo1"));
            conf2 = DummyConfig(struct(foo = "foo2"));

            testCase.verifyNotEqual(conf1.cache_id(), conf2.cache_id());
        end
    end


    methods (Test) % copy
        function copy_independent(testCase)
            conf1 = DummyConfig(struct(foo = 1, bar = 2));

            conf2 = copy(conf1);
            conf2.foo = 3;

            testCase.verifyEqual(conf1.foo, 1);
        end

        function copy_shallow(testCase)
            conf1 = DummyConfig(struct(foo = DummyConfig(struct(foo = 1))));

            conf2 = copy(conf1);
            conf2.foo.foo = 2;

            testCase.verifyEqual(conf1.foo.foo, 2);
        end

        function copy_ignored_fields(testCase)
            conf = DummyConfig(struct(ignored = "ignored1"));

            testCase.verifyEqual(copy(conf).ignored, "ignored1");
        end
    end


    methods (Test) % combinations
        function combinations_single(testCase)
            s1 = {struct()};

            combos = Config.combinations(s1);

            testCase.verifyEqual(combos, s1);
        end

        function combinations_one_cell(testCase)
            s1 = {struct(a = 1), struct(a = 2)};

            combos = Config.combinations(s1);

            testCase.verifyEqual(combos, s1);
        end

        function combinations_merges(testCase)
            s1 = {struct(a = 1)};
            s2 = {struct(b = 2)};

            combos = Config.combinations(s1, s2);

            testCase.verifyEqual(combos, {struct(a = 1, b = 2)});
        end

        function combinations_creates_both(testCase)
            s1 = {struct(a = 1)};
            s2 = {struct(b = 2), struct(b = 3)};

            combos = Config.combinations(s1, s2);

            testCase.verifyEqual(combos, {struct(a = 1, b = 2), struct(a = 1, b = 3)});
        end

        function combinations_creates_all(testCase)
            s1 = {struct(a = 1), struct(a = 2)};
            s2 = {struct(b = 2), struct(b = 3)};

            combos = Config.combinations(s1, s2);

            testCase.verifyEqual(combos, {struct(a = 1, b = 2), struct(a = 2, b = 2), ...
                                          struct(a = 1, b = 3), struct(a = 2, b = 3)});
        end

        function combinations_ragged_options(testCase)
            s1 = {struct(a = 1), struct(a = 2)};
            s2 = {struct(b = 1), struct(b = 2, c = 3)};

            combos = Config.combinations(s1, s2);

            testCase.verifyEqual(combos, {struct(a = 1, b = 1), struct(a = 2, b = 1), ...
                                          struct(a = 1, b = 2, c = 3), struct(a = 2, b = 2, c = 3)});
        end
    end
end
