% # Question
% How close does greedy optimisation get to the global optimum?
%
% # Setup
% Compare a greedy search to a search of the entire solution space.
%
% # Conclusion
% In graphs up to 7 nodes, greedy optimisation typically gets quite close to the
% optimum.

clear;


%% Config
reps = 8;
n = 7;
g = 4;


%% Code
scores_gr = nan([1, reps]);
scores_gl = nan([1, reps]);

edges_per_rep = (n * (n - 1)) / 2;
candidates_per_rep = 2^edges_per_rep;

pm = ProgressBar(reps * candidates_per_rep, taskname = "GreedyOptimality");
parfor rep = 1:reps
    G_raw = graph(logical(eye(2)));
    while ~Graphs.is_connected(G_raw); G_raw = Graphs.generate_erdos_renyi(n, 1); end
    G_stretched = Graphs.stretch(G_raw, girth = g, method = "most_cycles_steps");

    score_old = Graphs.eigenratio(G_stretched);
    G_gr = Graphs.optimise(G_stretched, metric = "eigenratio", direction = "maximum", girth = g);
    scores_gr(rep) = Graphs.eigenratio(G_gr);

    triu_mask = logical(triu(ones(n), 1));
    score_gl = 0;
    for candidate = 0:(candidates_per_rep - 1)
        count(pm);

        edges = int2bit(candidate, edges_per_rep);
        if numel(find(edges)) < n - 1; continue; end  % Too few edges to be connected

        A = false(n);
        A(triu_mask) = edges;
        G_gl = graph(A + A');

        g_gl = Graphs.girth(G_gl);
        if isinf(g_gl) || g_gl < g; continue; end

        score_gl = max(score_gl, Graphs.eigenratio(G_gl));
    end
    scores_gl(rep) = score_gl;
end
delete(pm);


%% Output
mean(scores_gr)
mean(scores_gl)
