% # Question
% Is simulated annealing (SA) better than greedy (GR) optimisation?
%
% # Setup
% Run both methods on stretched graphs. For SA, vary hyperparameters and choose
% only the best results. Compare the achieved scores and the number of changes
% across both methods.
%
% # Conclusion
% SA achieves a slightly more optimal optimum than GR, and does so in the same
% number of edges. However, the change does not appear to be significant, and is
% more complex.

clear;
rng(17, "twister");

reps = 5;
n = 25;
g = 10;
t_max_list = 200:100:500;
d_thr_list = -6:1:0;

[t_max_grid, d_thr_grid] = ndgrid(t_max_list, d_thr_list);
par_gr_scores = nan([reps, numel(t_max_grid)]);
par_sa_scores = nan([reps, numel(t_max_grid)]);
par_gr_numedges = nan([reps, numel(t_max_grid)]);
par_sa_numedges = nan([reps, numel(t_max_grid)]);
for i = 1:numel(t_max_grid)
    for rep = 1:reps
        G_raw = graph(logical(eye(2)));
        while ~Graphs.is_connected(G_raw); G_raw = Graphs.generate_erdos_renyi(n, log(n) / n); end
        G_stretched = Graphs.stretch(G_raw, girth = g, method = "most_cycles_steps");

        G_gr = Graphs.optimise( ...
            G_stretched, ...
            method = "greedy", ...
            metric = "eigenratio", ...
            direction = "maximum", ...
            girth = g ...
        );
        par_gr_scores(rep, i) = Graphs.eigenratio(G_gr);
        par_gr_numedges(rep, i) = numedges(G_gr);

        G_sa = Graphs.optimise( ...
            G_stretched, ...
            method = "simulated_annealing", ...
            metric = "eigenratio", ...
            direction = "maximum", ...
            girth = g, ...
            sa_t_max = t_max_grid(i), ...
            sa_d_thr = d_thr_grid(i) ...
        );
        par_sa_scores(rep, i) = Graphs.eigenratio(G_sa);
        par_sa_numedges(rep, i) = numedges(G_sa);
    end
end

[max_val, max_idx] = max(mean(par_sa_scores, 1), [], "all");
best_t_max = t_max_grid(max_idx);
best_d_thr = d_thr_grid(max_idx);

fprintf("Optimal parameters: t_max = %d, d_thr = %.2f.\n", best_t_max, best_d_thr);
fprintf("Annealing score: %.5f.\n", max_val);
fprintf("Greedy score: %.5f.\n", mean(par_gr_scores, "all"));
