function to = copy_into(from, to)
    % COPY_INTO  Copies all fields from [from] to [to] recursively.
    %
    % Unless [to] is a handle, [from] is merely copied into a shallow copy of
    % [to].
    %
    % If a field is present only in [to], the field is retained. If a field is
    % present only in [from], then the field is copied into [to]. (A shallow
    % copy, unless that field is a handle.) If the field is present in both
    % [from] and [to], then the action taken depends on the types. If the field
    % is a struct or object (but not a string or a table) in both [from] and
    % [to], then [copy_into] is recursively called on that field. If the field
    % is a numeric matrix with the same size in both [from] and [to], then all
    % non-nan values in [from] are copied into [to]. If the field has a
    % different type in [from] and [to], then the field of [from] overwrites
    % that in [to].

    fields = fieldnames(from);
    for i = 1:numel(fields)
        field = fields{i};

        from_value = from.(field);
        to_has_field = isfield(to, field) || isprop(to, field);
        if to_has_field; to_value = to.(field); end

        if isobjectlike(from_value) && to_has_field && isobjectlike(to_value)
            to.(field) = copy_into(from_value, to_value);
        elseif isnumericmatrix(from_value) && to_has_field && isnumericmatrix(to_value) && ...
                all(size(from_value) == size(to_value))
            idx = ~isnan(from_value);
            to_value(idx) = from_value(idx);
            to.(field) = to_value;
        else
            to.(field) = from_value;
        end
    end
end


function isobjectlike = isobjectlike(obj)
    isobjectlike = isstruct(obj) || isobject(obj) && ~isstring(obj) && ~istable(obj);
end

function isnumericmatrix = isnumericmatrix(obj)
    isnumericmatrix = ismatrix(obj) && isnumeric(obj);
end
