classdef PlotConfig < Config
    % PLOTCONFIG  Configuration for plotting the results of a [Laboratory].

    properties
        % TITLE  The title of the plot.
        %
        % The title is not saved in any created figures, but determines the filename.
        title (1, 1) {mustBeText} = "";
        % TYPE  The type of plot to draw.
        %
        % If set to "tiles", settings relating to `exp`, `line`, and `legend` are
        % ignored, but settings starting with "tiles_" are activated.
        %
        % Must be one of: "line", "confidence", "boxplot", "scatter", "tiles".
        type (1, 1) {mustBeMember(type, ["line", "confidence", "boxplot", "scatter", "tiles"])} = "confidence";
        % SAVE_FORMATS  The file formats to save this plot as.
        %
        % Must be a file extension supported by [saveas], or "bin.tikz" to use
        % [matlab2tikz].
        save_formats (1, :) {mustBeText} = ["svg", "epsc", "png"];

        % FIG_CONF  The config to copy into the `gcf` object of the figure.
        %
        % If `fig_conf.UserData` contains the string `"PreserveAspectRatio"`, then
        % the figure's aspect ratio is preserved if `fig_conf.Position` is changed.
        fig_conf (1, 1) struct;
        % AX_CONF  The config to copy into the `gca` object of the figure.
        ax_conf (1, 1) struct;

        % FILTER_EXP  Determines whether an [Experiment] is included in this plot.
        filter_exp = @(~) true;  % ((1, 1) Experiment) -> (1, 1) logical
        % EXP_TO_LINE  Maps an [Experiment] to the string identifier of the line the
        % [Experiment] is part of.
        %
        % Use this to combine multiple [Experiment]s into a single line.
        %
        % The default value returns a unique string depending on the [ExperimentConfig].
        exp_to_line = @(it) it.conf.cache_id();  % ((1, 1) Experiment) -> (1, 1) {mustBeText};
        % EXP_TO_DATA  Maps an [Experiment] to the `(x, y)` point it represents on the
        % line.
        %
        % For line-style plots (see [#type]), all [Experiment]s that map to the same `x`
        % in a line are averaged.
        exp_to_data;  % ((1, 1) Experiment) -> (:, 2) double;
        % LINE_TO_LABEL  Returns the label corresponding to a line based on the
        % [ExperimentConfig] of any of the filtered [Experiment]s.
        line_to_label = 0;  % ((1, 1) Experiment) -> (1, 1) {mustBeText};
        % LINE_TO_STYLE  Returns the style to apply to the line based on the
        % [ExperimentConfig] of any of the filtered [Experiment]s.
        line_to_style = 0;  % ((1, 1) Experiment) -> (1, :) {mustBeText};

        % CONFIDENCE_CONF  If [type] is `confidence`, then this is the style that
        % should be copied into the confidence interval's shaded area object.
        confidence_conf (1, 1) struct;

        % TILES_FIGS  The [PlotConfig]s to embed into this tiled figure.
        %
        % The matrix of titles must be square. The order in the matrix is the same
        % as how they will be arranged. The contained [PlotConfig]s should not be
        % tiled figures themselves.
        tiles_confs (:, :) cell = {};  % cell<PlotConfig>
        % TILES_ROW_TITLES  The title to display next to each row, or an empty
        % string to not give that row a title, or an empty matrix to not give any
        % row a title.
        tiles_row_titles (1, :) {mustBeText} = strings(0);
        % TILES_COL_TITLES  The title to display above each column, or an empty
        % string to not give that column a title, or an empty matrix to not give
        % any column a title.
        tiles_col_titles (1, :) {mustBeText} = strings(0);

        % LEGEND_ONLY  Exports the legend (and only the legend).
        legend_only (1, 1) logical = false;
        % LEGEND_ENTRIES  If non-empty, shows only the given legend entries, where
        % the first column is the label and the second label is the arguments to
        % pass to [plot] to determine the label's style.
        legend_entries = @() {};
        % LEGEND_CONF  The config to copy into the legend object of the figure.
        %
        % Applied only if there is a non-zero number of lines or [legend_entries] is
        % defined, or [legend_only] is `true`.
        legend_conf (1, 1) struct;

        % FONT_SIZE  The font size (in points) to use in the figure, or `nan` to use the
        % default.
        font_size (1, 1) {mustBeNumeric} = nan;
    end


    methods
        function obj = PlotConfig(strct)
            arguments% (Input)
                strct struct = struct();
            end
            % arguments (Output)
            %     obj (1, 1) PlotConfig;
            % end

            obj@Config(strct);
        end


        function culled = cache_culled(~)
            arguments% (Input)
                ~;
            end
            % arguments (Output)
            %     culled (1, 1) struct;
            % end

            culled = struct();
        end
    end
end
