% MYFIGURE  Utility methods for creating and exporting figures from MATLAB to Tikz.
classdef MyFigure
    properties (Constant)
        % VISIBLE  Whether figures are visible by default.
        visible = false;
        % LINE_WIDTH  The default line width to use.
        line_width = 1.2;
        % LEG_TOKEN_SIZE  The default value to set for a legend's `ItemTokenSize`.
        leg_token_size = 12;
    end

    methods (Static)
        % CREATE  Creates a new figure handle, with useful default settings.
        function f = create()
            warning("off", "MATLAB:handle_graphics:exceptions:SceneNode");  % False positives about encoding
            warning("off", "MATLAB:structOnObject");  % Not relevant when converting heatmap axes to struct

            f = figure(Visible = MyFigure.visible);

            set(f, DefaultTextInterpreter = "latex");
            set(f, DefaultLegendInterpreter = "latex");
            set(f, DefaultAxesTickLabelInterpreter = "latex");

            set(f, DefaultLineLineWidth = MyFigure.line_width);
        end

        % HEATMAP_CBAR_LABEL_FORMAT  Sets the format of the labels of the colorbar of the heatmap [h] to [format].
        function heatmap_cbar_label_format(h, format)
            sh = struct(h);
            sh.Colorbar.TickLabels = arrayfun(format, sh.Colorbar.Ticks);  %#ok<STRNU>
        end

        % HEATMAP_X_TICK_LABEL_FILTER  Removes all x-tick labels from [h] that match the given [filter].
        function heatmap_x_tick_label_filter(h, filter)
            l = string(h.XDisplayLabels);
            l(arrayfun(filter, l)) = " ";
            h.XDisplayLabels = l;
        end

        % HEATMAP_Y_TICK_LABEL_FILTER  Removes all y-tick labels from [h] that match the given [filter].
        function heatmap_y_tick_label_filter(h, filter)
            l = string(h.YDisplayLabels);
            l(arrayfun(filter, l)) = " ";
            h.YDisplayLabels = l;
        end

        % EXPORT  Exports the figure [f] as a Tikz figure with filename [args], after applying the optional [args].
        %
        % The value of [args.format] determines as what file type the image is saved:
        % * `""`: The image is saved as a TikZ image using the filename [name] without modification.
        % * `"tikz"`/`".tikz"`/`"bin.tikz"`/`".bin.tikz"`: The image is saved as a TikZ image using the filename [name]
        %                                                  with [args.format] used as the file extension.
        % * `"tex"`/`".tex"`: The image is saved as a standalone LaTeX file.
        % * Any format supported by MATLAB's `saveas` function: The image is stored in that format, without any special
        %                                                       pre- or post-processing.
        % With the exception of the case in which [args.format] is the empty string, this function ensures the file uses
        % [args.format] as the file extension: If [name] does not end in [args.format], then [args.format] is appended
        % (with a `.` in between if [args.format] does not already start with one); otherwise, [name] is taken as is.
        function export(f, name, args)
            arguments% (Input)
                f (1, 1) handle;
                name (1, 1) {mustBeText};
                args.extra_axis_options (1, :) cell = {};
                args.extra_x_ticks (1, :) {mustBeNumeric} = [];
                args.extra_y_ticks (1, :) {mustBeNumeric} = [];
                args.format (1, 1) {mustBeText} = "";
                args.y_ticks_scale (1, 1) {mustBeText, mustBeMember(args.y_ticks_scale, ["fixed", "scientific"])} = "fixed";
            end


            if ~ismember(args.format, ["", "tikz", ".tikz", "bin.tikz", ".bin.tikz", "tex", ".tex"])
                % `saveas` is preferred over `exportgraphics` because the latter has ugly fonts
                saveas(f, name, args.format);
                return;
            end

            if ~exist("matlab2tikz", "file")
                error("You do not have the required matlab2tikz add-on installed.");
            end


            % Determine image type
            child_classes = arrayfun(@(it) string(class(it)), f.Children);
            is_tiles = any(child_classes == "matlab.graphics.layout.TiledChartLayout");
            is_heatmap = any(child_classes == "matlab.graphics.chart.HeatmapChart");


            % Validate inputs
            if is_heatmap && ismember(args.format, ["tex", ".tex"])
                error("Cannot export heatmap as standalone figure.");
            end


            % Get handles
            if is_tiles
                tiles = f.Children(find(child_classes == "matlab.graphics.layout.TiledChartLayout"));
                ax = tiles.Children(end);  % last sub-graph
                if class(tiles.Children(1)) == "matlab.graphics.illustration.Legend"
                    leg = tiles.Children(1);
                else
                    leg.Visible = false;
                end
            elseif is_heatmap
                ax = f.CurrentAxes;
                leg.Visible = false;
            else
                ax = f.CurrentAxes;
                leg = legend(ax, "toggle"); legend(ax, "toggle");
            end

            % Process legend
            if leg.Visible
                leg.ItemTokenSize = [MyFigure.leg_token_size, nan];
            end

            extra_axis_options = convertContainedStringsToChars(args.extra_axis_options);
            if leg.Visible && leg.NumColumns ~= 1
                extra_axis_options = [ ...
                    extra_axis_options, ...
                    sprintf('legend style={legend columns=%d}', leg.NumColumns)
                ];
            end

            if leg.Visible && ~isempty(leg.Title.String)
                if leg.NumColumns == 1
                    leg_options = "";
                    leg_wrap = "";
                else
                    leg_options = "text width=0pt,text depth=";
                    leg_wrap = "\makebox[0pt][l]";
                end

                leg_block = ...
                    "\addlegendimage{empty legend}" + ...
                    "\addlegendentry[" + leg_options + "]{" + ...
                        leg_wrap + "{\hspace{-.7cm}\textbf{" + leg.Title.String + "}}" + ...
                    "}" + ...
                    join(repmat("\addlegendimage{empty legend}\addlegendentry{}", [1, leg.NumColumns - 1]), "");

                extra_axis_options = [ ...
                    extra_axis_options, ...
                    sprintf('execute at begin axis={%s}', leg_block) ...
                ];
            end

            % Process ticks and axes
            if ~is_heatmap
                if args.y_ticks_scale == "fixed"
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        'scaled y ticks=false', ...
                        'y tick label style={/pgf/number format/.cd, fixed, precision=9}' ...
                    ];
                end
                if ax.XGrid
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        'xtick style={draw=none}' ...
                    ];
                end
                if ax.YGrid
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        'ytick style={draw=none}' ...
                    ];
                end
                if ax.YMinorGrid
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        'minor y grid style={densely dotted}' ...
                    ];
                end
                if ~isempty(args.extra_x_ticks)
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        sprintf('extra x ticks={%s}', join(string(args.extra_x_ticks), ", ")) ...
                    ];
                end
                if ~isempty(args.extra_y_ticks)
                    extra_axis_options = [ ...
                        extra_axis_options, ...
                        sprintf('extra y ticks={%s}', join(string(args.extra_y_ticks), ", ")) ...
                    ];
                end
            end


            % Export
            if ismember(args.format, ["tikz", ".tikz"])
                standalone = false;
                if ~endsWith(name, ".tikz"); name = name + ".tikz"; end
            elseif ismember(args.format, ["bin.tikz", ".bin.tikz"])
                standalone = false;
                if ~endsWith(name, ".bin.tikz"); name = name + ".bin.tikz"; end
            elseif ismember(args.format, ["tex", ".tex"])
                standalone = false;
                if ~endsWith(name, ".tex"); name = name + ".tex"; end
            else
                standalone = false;
            end

            if is_heatmap
                writelines(MyFigure.heatmap_to_tikz(ax), name);
            else
                cleanfigure('handle', f, 'targetResolution', inf);
                matlab2tikz( ...
                    'figurehandle', f, ...
                    'filename', convertStringsToChars(name), ...
                    'showInfo', false, ...
                    'showWarnings', false, ...
                    'standalone', standalone, ...
                    'noSize', true, ...
                    'extraCode', {'\providecommand*{\myfigurefontsize}{\scriptsize}\myfigurefontsize'}, ...
                    'extraAxisOptions', extra_axis_options, ...
                    'checkForUpdates', false ...
                );
            end


            % Post-process
            text = fileread(name);

            % Rename colors, and allow to be overridden by document's colors
            text = replace(text, "mycolor", "matlab");
            text = replace(text, "definecolor", "providecolor");

            if is_tiles
                % Create matrix of arranged axes
                d = tiles.GridSize;

                text = replace(text, "\usetikzlibrary{arrows.meta}", sprintf("\\usetikzlibrary{arrows.meta}\n\\usetikzlibrary{matrix}"));
                text = replace(text, sprintf("\\begin{tikzpicture}"), sprintf("\\begin{tikzpicture}\n\\matrix{"));
                text = replace(text, "\end{tikzpicture}", sprintf("};\n\\end{tikzpicture}"));
                text = replace(text, "\\[1ex]", "\\[2ex]");
                parts = split(string(text), "\end{axis}");
                text = join(parts, "\end{axis} " + repmat(horzcat(repmat("&", [1, d(2) - 1]), "\\"), [1, d(1)])');
            end
            if ~is_heatmap && ~is_tiles
                % Enable layered drawing, to ensure grid lines are behind axis lines
                % This MUST be done AFTER adding the line `\matrix{` when `is_tiles` is `true`
                text = replace(text, sprintf("\\begin{tikzpicture}"), sprintf("\\begin{tikzpicture}\n\\pgfplotsset{set layers}"));
            end

            % Save post-processed file
            fp = fopen(name, "w");
            fprintf(fp, "%s", text);
            fclose(fp);
        end
    end

    methods (Static, Access = private)
        % HEATMAP_TO_TIKZ  Converts the heatmap axes [h] to file contents.
        function contents = heatmap_to_tikz(h)
            [t, xlabels, ylabels] = MyFigure.heatmap_parse(h);
            sh = struct(h);

            cm = h.Colormap;
            cs = round(linspace(1, height(cm), 4));
            c = join(arrayfun(@(it) sprintf("rgb(%d)=(%f, %f, %f)", it, cm(it, 1), cm(it, 2), cm(it, 3)), cs), "; ");

            lines = [
                "% This file was created by MyFigure.m."
                "%"
                "\providecommand*{\myfigurefontsize}{\scriptsize}\myfigurefontsize"
                "\pgfplotstableset{row sep=\\}"
                "\begin{tikzpicture}"
                "    \begin{axis}["
                "        tick style={draw=none},"
                "        major tick length=0mm,"
                "        minor tick length=0mm,"
                "        %"
                "        xlabel={" + h.XLabel + "},"
                "        xmin={" + string(min(t.x)) + " - 0.5},"
                "        xmax={" + string(max(t.x)) + " + 0.5},"
                "        xtick={" + string(min(t.x)) + ", ..., " + string(max(t.x)) + "},"
                "        xticklabels={" + join(xlabels, ", ") + "},"
                "        %"
                "        ylabel={" + h.YLabel + "},"
                "        ymin={" + string(min(t.y)) + " - 0.5},"
                "        ymax={" + string(max(t.y)) + " + 0.5},"
                "        ytick={" + string(min(t.y)) + ", ..., " + string(max(t.y)) + "},"
                "        yticklabels={" + join(ylabels, ", ") + "},"
                "        %"
                "        tick align=outside,"
                "        tick pos=left,"
                "        %"
                "        colormap={blue}{" + c + "},"
                "        colorbar,"
                "        colorbar style={"
                "            point meta min={" + string(h.ColorLimits(1)) + "},"
                "            point meta max={" + string(h.ColorLimits(2)) + "},"
                "            ytick={" + join(string(sh.Colorbar.Ticks), ", ") + "},"
                "            yticklabels={" + replace(join(string(sh.Colorbar.TickLabels), ", "), "%", "\%") + "},"
                "            ytick style={draw=none},"
                "            width={\linewidth / 40},"
                "            xshift={\linewidth / 10 - .65cm},"
                "        },"
                "    ]"
                "        \addplot["
                "            matrix plot,"
                "            point meta=explicit,"
                "            mesh/cols={" + string(max(t.y) - min(t.y) + 1) + "},"
                "        ] table[meta=z] {"
                "            x y z\\" + string(join(cellstr(table2cell( ...
                                rowfun(@(x, y, z) sprintf("%d %d %f", x, y, z), t, InputVariables=["x", "y", "z"]) ...
                            )), "\\")) + "\\"
                "        };"
                "    \end{axis}"
                "\end{tikzpicture}"
            ];

            contents = join(lines, newline);
        end

        % HEATMAP_PARSE  Parses the data of the heatmap axes [h] to a table, with rows sorted in order of ascending x
        % and then y.
        function [t, xlabels, ylabels] = heatmap_parse(h)
            x = round(str2double(h.XDisplayData));
            xlabels = strtrim(string(h.XDisplayLabels));
            y = round(str2double(h.YDisplayData));
            ylabels = strtrim(string(h.YDisplayLabels));
            z = h.ColorData;

            if issorted(x, "strictdescend")
                x = flip(x);
                xlabels = flip(xlabels);
                z = fliplr(z);
            elseif ~issorted(x, "strictascend")
                error("Heatmap x data must be strictly monotonic.");
            end
            if issorted(y, "strictdescend")
                y = flip(y);
                ylabels = flip(ylabels);
                z = flipud(z);
            elseif ~issorted(y, "strictascend")
                error("Heatmap y data must be strictly monotonic.");
            end

            [x, y] = ndgrid(x, y);
            t = table(reshape(x', [], 1), ...
                      reshape(y', [], 1), ...
                      reshape(z, [], 1), ...
                      VariableNames=["x", "y", "z"]);
        end
    end
end
