classdef LaboratoryConfig < Config
    % LABORATORYCONFIG  Configuration for a [Laboratory].

    properties
        % SEED  Seed for random number generator.
        %
        % A negative value indicates no fixed seed should be used.
        %
        % Two runs with the same seed give the same result. However, this does not
        % necessarily hold if the runs are with different MATLAB versions.
        seed (1, 1) {mustBeInteger} = -1;

        % PARALLEL  Whether to run experiments in parallel.
        %
        % Ignored if the MATLAB Parallel Computing Toolbox is not installed.
        parallel (1, 1) logical = true;
        % PARALLEL_MAX_WORKERS  The maximum number of parallel workers.
        %
        % Ignored if [parallel] is `false`.
        parallel_max_workers (1, 1) {mustBeNumeric, mustBeGreaterThanOrEqual(parallel_max_workers, 1)} = Inf;

        % EXP_CONFS  The [ExperimentConfig]s for the experiments to run.
        exp_confs (:, 1) ExperimentConfig;
        % REPEAT_COUNT  The number of times to repeat each [Experiment].
        repeat_count (1, 1) {mustBeInteger, mustBePositive} = 3;

        % PART_COUNT  Number of parts to split the laboratory into, useful if the
        % laboratory is too much work to finish in one go on a cluster.
        %
        % Parts are essentially just an implicit way of splitting up experiments.
        part_count (1, 1) {mustBeInteger, mustBePositive} = 1;
        % PART_IDX  The part to run in this laboratory, or 0 to run all experiments.
        %
        % If [part_idx] is nonzero, plots are disabled.
        part_idx (1, 1) {mustBeInteger, mustBeNonnegative} = 0;

        % CACHE_DIR  The directory to cache any results in.
        cache_dir (1, 1) {mustBeText} = "cache/";
        % SAVE_VERSION  The save file version identifier, to force a re-run despite the
        % presence of appropriate cache files.
        save_version (1, 1) {mustBeText} = "v18";
        % SAVE_ENABLED  `true` if and only if experiments should be saved after
        % completion.
        save_enabled (1, 1) logical = true;
        % SAVE_LAB_ENABLED  `true` if and only if entire laboratories should also be
        % saved.
        %
        % Note that labs are stored much less efficiently than individual experiments
        % because labs typically exceed the maximum file size supported by MATLAB's more
        % efficient storage method.
        %
        % Ignored if [save_enabled] is `false`. See [will_save_lab] for exact details.
        save_lab_enabled (1, 1) logical = false;
        % LOAD_ENABLED  `true` if and only if completed laboratories and experiments
        % should be loaded instead of re-running them.
        load_enabled (1, 1) logical = true;
        % LOAD_EXP_BEHAVIOR  The way existing cached [Experiment]s should be loaded.
        %
        % * `"run"` always re-runs [Experiment]s.
        % * `"load"` loads a cached [Experiment] if it exists, and runs the [Experiment]
        %   otherwise.
        % * `"skip"` skips the [Experiment] if cached ouput exists, and runs the
        %   [Experiment] otherwise. This may result in missing [Experiment]s, so
        %   plotting is disabled.
        %
        % Must be one of: "run", "load", and "skip".
        %
        % Ignored if [load_enabled] is `false`.
        load_exp_behavior (1, 1) {mustBeMember(load_exp_behavior, ["run", "load", "skip"])} = "load";
        % LOAD_VALIDATE  `true` if and only if cached files should be checked for
        % integrity before loading.
        %
        % If a cached file fails the check, the cache file is treated as if it does not
        % exist.
        %
        % Ignored if [load_enabled] is `false` or if [load_exp_behavior] is `"run"`.
        load_validate (1, 1) logical = true;
        % SAVE_LAB_TARGET  The file path relative to [cache_dir] for saving a completed
        % laboratory of experiments, expressed as a format string where (in order) `%s`
        % is the [save_version] and `%s` is the [LaboratoryConfig#cache_id].
        save_lab_target (1, 1) {mustBeText, ...
                                mustContainSubstring(save_lab_target, "%s")} = "lab-%s-%s.mat";
        % SAVE_EXP_TARGET  The file path relative to [cache_dir] for saving experiments,
        % expressed as a format string where (in order) `%s` is the [save_version], `%s`
        % is the [ExperimentConfig#cache_id], and `%d` identifies the repetition number
        % of the experiment.
        save_exp_target (1, 1) { ...
            mustBeText, ...
            mustContainSubstring(save_exp_target, "%s"), ...
            mustContainSubstring(save_exp_target, "%d") ...
        } = "exp-%s-%s-%d.mat";

        % METRICS  Metrics to calculate on the experiments after completion.
        %
        % Each metric is specified as a mapping from its name to a function that maps an
        % [Experiment] to the metric value.
        metrics (1, 1) struct;  % struct<string, (Experiment) -> any>

        % PLOT_DIR  The directory to save plots in.
        plot_dir (1, 1) {mustBeText} = "figs/";
        % PLOT_CONFS  The [PlotConfig]s for the plots to make after completing all
        % [Experiment]s.
        plot_confs (:, 1) PlotConfig = PlotConfig.empty();
        % PLOT_SHOW  `true` if and only if obtained plots should be shown.
        plot_show (1, 1) logical = false;
        % PLOT_SAVE  `true` if and only if obtained plots should be saved to disk.
        %
        % Plots are saved in the [save_directory] folder.
        plot_save (1, 1) logical = true;
    end


    methods
        function obj = LaboratoryConfig(strct)
            arguments% (Input)
                strct struct = struct();
            end
            % arguments (Output)
            %     obj (1, 1) LaboratoryConfig;
            % end

            obj = obj@Config(strct);
        end


        function culled = cache_cull(obj)
            arguments% (Input)
                obj (1, 1) LaboratoryConfig;
            end
            % arguments (Output)
            %     culled (1, 1) struct;
            % end

            culled = struct(seed = obj.seed, ...
                            repeat_count = obj.repeat_count, ...
                            exp_confs = arrayfun(@(it) it.cache_cull(), obj.exp_confs, UniformOutput = false));
        end


        function will_save_lab = will_save_lab(obj)
            % WILL_SAVE_LAB  Returns `true` if and only if a [Laboratory] using this
            % configuration will save the laboratory as per the [save_lab_enabled] field.

            will_save_lab = ...
                obj.part_idx == 0 && ...
                obj.save_enabled && ...
                obj.save_lab_enabled && ...
                obj.load_exp_behavior ~= "skip";
        end

        function will_plot = will_plot(obj)
            % WILL_PLOT  Returns `true` if and only if a [Laboratory] using this
            % configuration will not skip [Laboratory#plot].

            will_plot = ...
                obj.part_idx == 0 && ...
                (obj.plot_show || obj.plot_save) && ...
                obj.load_exp_behavior ~= "skip";
        end
    end
end
