classdef Graphs
    % GRAPHS  Contains static functions for checking, creating, and manipulating
    % graphs.

    methods (Static)
        function is_unweighted = is_unweighted(G)
            % IS_UNWEIGHTED  Returns `true` if and only if [G] is an unweighted graph.
            %
            % Note that a graph with all weights set to `1` is still considered a weighted
            % graph. For a graph to be considered unweighted, edges must not have weights at
            % all.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     is_unweighted (1, 1) logical;
            % end

            % Check that `G.Edges` table has only one column
            is_unweighted = isscalar(G.Edges.Properties.VariableNames);
        end

        function is_simple = is_simple(G)
            % IS_SIMPLE  Returns `true` if and only if [G] is a simple graph, i.e. a
            % directed graph without duplicate edges and without self-loops.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     is_simple (1, 1) logical;
            % end

            is_simple = isisomorphic(G, simplify(G));
        end

        function is_connected = is_connected(G)
            % IS_CONNECTED  Returns `true` if and only if [G] is connected.
            %
            % Throws an error if [G] is the empty graph.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     is_connected (1, 1) logical;
            % end

            comps = conncomp(G);
            assert(numel(comps) > 0);

            is_connected = any(max(comps) == 1);
        end


        function G = generate_empty(n)
            % GENERATE_EMPTY  Outputs a graph with [n] nodes and 0 edges.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            G = graph(false([n, n]));
        end

        function G = generate_complete(n)
            % GENERATE_COMPLETE  Outputs a complete graph with [n] nodes.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            G = graph(true([n, n]), "omitselfloops");
        end

        function G = generate_erdos_renyi(n, p)
            % GENERATE_ERDOS_RENYI  Outputs an (n, p) Erdős-Renyi graph, with [n] nodes and
            % each edge having a probability [p] of being included.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                p (1, 1) {mustBeInRange(p, 0, 1)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            A = false(n);
            [X, Y] = meshgrid(1:n);
            A(X > Y) = rand([(n * (n - 1)) / 2, 1]) < p;
            G = graph(A, "upper");
        end

        function G = generate_watts_strogatz(n, k, p)
            % GENERATE_WATTS_STROGATZ  Outputs a Watts-Strogatz graph with [n] nodes,
            % [n]*[k] edges, mean node degree 2*[k], and rewiring probability [p].
            %
            % An (n, k, p)-Watts-Strogatz graph is constructed by taking a regular ring
            % lattice (where each node is connected to the previous [k] nodes and the next
            % [k] nodes), and then, for each node, for each edge connecting to a subsequent
            % (but not preceding) node, changing the destination of that edge with
            % probability [p], to a uniformly randomly chosen other node, avoiding
            % self-loops and duplicate edges.
            %
            % `p = 0` creates a ring lattice, and `p = 1` creates a random graph.
            %
            % Regardless of [p], `k = 0` creates an edge-free graph, and odd `n` and 
            % `k = (n - 1) / 2` creates the complete graph.
            %
            % Slightly modified from
            % https://mathworks.com/help/matlab/math/build-watts-strogatz-small-world-graph-model.html.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                k (1, 1) {mustBeInteger, mustBeInRange(k, 0, n)};
                p (1, 1) {mustBeFloat, mustBeInRange(p, 0, 1)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            assert(2 * k + 1 <= n, "k must be at most (n - 1) / 2.");

            if k == 0
                G = Graphs.generate_empty(n);
                return;
            end

            % Construct ring lattice
            s = repelem((1:n)', 1, k);
            t = s + repmat(1:k, n, 1);
            t = mod(t - 1, n) + 1;

            % Rewire
            for source = 1:n
                switch_edge = rand(k, 1) < p;

                new_t = rand(n, 1);
                new_t(source) = 0;
                new_t(s(t == source)) = 0;
                new_t(t(source, ~switch_edge)) = 0;

                [~, ind] = sort(new_t, "descend");
                t(source, switch_edge) = ind(1:nnz(switch_edge));
            end

            G = graph(s, t);
        end

        function G = generate_barabasi_albert(n, m)
            % GENERATE_BARABASI_ALBERT  Outputs a Barabási-Albert graph with [n] nodes,
            % using preferential attachment parameter [m].
            %
            % An (n, m)-Barabási-Albert graph is constructed by taking an initial graph, in
            % this case the star graph with `m + 1` nodes, and then iteratively adding
            % nodes, connecting each new nodes to [m] random previous nodes, sampled
            % linearly in their degree. In this implementation, sampling is done without
            % replacement, ensuring that each new node creates [m] new edges.
            %
            % `m = 0` creates an edge-free graph, and `m = n` creates a star graph.

            arguments% (Input)
                n (1, 1) {mustBeInteger, mustBePositive};
                m (1, 1) {mustBeInteger, mustBeGreaterThanOrEqual(m, 0), mustBeLessThanOrEqual(m, n)};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            if m == 0
                G = Graphs.generate_empty(n);
                return;
            elseif m == n
                A = false(n, n);
                A(2:end, 1) = 1;
                A(1, 2:end) = 1;
                G = graph(A);
                return;
            end

            m0 = m + 1;
            A = false([n, n]);

            % Use first `m0` nodes to create star graph
            A(2:m0, 1) = 1;
            A(1, 2:m0) = 1;

            % Populate graph
            for idx = (m0 + 1):n
                selected = datasample(1:(idx - 1), m, Weights = sum(A(1:(idx - 1), :), 2), Replace = false);

                A(selected, idx) = 1;
                A(idx, selected) = 1;
            end

            G = graph(A);
        end

        function G = generate_geometric_random(d, n, r)
            % GENERATE_GEOMETRIC_RANDOM  Outputs a geometric random graph with [n] nodes in
            % a [d]-dimensional unit hypercube with edges if and only if nodes are within
            % [r] Euclidian distance of one another.

            arguments% (Input)
                d (1, 1) {mustBeInteger, mustBePositive};
                n (1, 1) {mustBeInteger, mustBePositive};
                r (1, 1) {mustBeNumeric, mustBeNonnegative};
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            if n == 1
                G = Graphs.generate_empty(1);
                return;
            end

            G = graph(squareform(pdist(rand([n, d]))) <= r, "omitselfloops");
        end


        function count = fully_connected_graph_cycle_count(n, l)
            % FULLY_CONNECTED_GRAPH_CYCLE_COUNT  Returns the number of cycles of length
            % [l] in the fully-connected graph with [n] nodes.

            assert(l >= 3, "Cycles of length below 3 do not exist.");

            if l > n; count = 0; return; end

            count = factorial(n) / (factorial(n - l) * 2 * l);
        end

        function girth = girth(G)
            % GIRTH  Returns the length of the shortest cycle in [G], or `Inf` if [G] has no
            % cycles.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     girth (1, 1) {isNumeric, mustBeNonnegative};
            % end

            if ~hascycles(G)
                girth = Inf;
                return;
            end

            for girth = 3:numnodes(G)
                [~, found_cycles] = allcycles(G, MaxCycleLength = girth, MaxNumCycles = 1);

                if ~isempty(found_cycles); return; end
            end
        end

        function leaf_count = leaf_count(G)
            % LEAF_COUNT  Returns the number of leaves in [G].

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     leaf_count (1, 1) {isInteger, mustBeNonnegative};
            % end

            leaf_count = sum(degree(G) == 1);
        end

        function score = algebraic_connectivity_loose(A, d)
            % ALGEBRAIC_CONNECTIVITY_LOOSE  Returns the algebraic connectivity, i.e.
            % the second-smallest eigenvalue, of the Laplacian matrix of the graph
            % defined by adjacency matrix [A] and degree vector [d].

            arguments% (Input)
                A (:, :) {mustBeNonnegative};
                d (:, 1) {mustBeNonnegative};
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            l = eig(diag(d) - A);
            score = l(2);
        end

        function score = algebraic_connectivity(G)
            % ALGEBRAIC_CONNECTIVITY  Variant of [Graphs.algebraic_connectivity_loose]
            % that takes a graph as input instead.

            score = Graphs.algebraic_connectivity_loose(adjacency(G), degree(G));
        end

        function score = eigenvector_delta_loose(A, d, k, srcs, dsts)
            % EIGENVECTOR_DELTA_LOOSE  Returns the "eigenvector delta" of each edge
            % from [srcs] to [dsts] in the graph defined by adjacency matrix [A] and
            % degree vector [d].
            %
            % The "eigenvector delta" of an edge from `src` to `dst` is 
            % `(u(src) - u(dst))^2` given the eigenvector `u` corresponding to the
            % [k]th smallest eigenvalue.

            [V, ~] = eig(diag(d) - A);
            F = V(:, k);
            score = (F(srcs) - F(dsts)).^2;
        end

        function score = eigenvector_delta(G, k, srcs, dsts)
            % EIGENVECTOR_DELTA  Variant of [Graphs.eigenvector_delta_loose] that takes
            % a graph as input instead.

            arguments% (Input)
                G (1, 1) graph;
                k (1, 1) {mustBeInteger, mustBePositive};
                srcs (:, 1) {mustBeInteger, mustBePositive};
                dsts (:, 1) {mustBeInteger, mustBePositive};
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            score = Graphs.fiedler_loose(adjacency(G), degree(G), k, srcs, dsts);
        end

        function score = eigenratio_loose(A, d)
            % EIGENRATIO_LOOSE  Returns the ratio of the second-smallest eigenvalue to the
            % largest eigenvalue of the Laplacian matrix of the graph defined by adjacency
            % matrix [A] and degree vector [d].

            arguments% (Input)
                A (:, :) {mustBeNonnegative};
                d (:, 1) {mustBeNonnegative};
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            l = eig(diag(d) - A);
            score = l(2) / l(end);
        end

        function score = eigenratio(G)
            % EIGENRATIO  Variant of [Graphs.eigenratio_loose] that takes a graph as input
            % instead.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            score = Graphs.eigenratio_loose(adjacency(G), degree(G));
        end

        function score = closeness_centrality_loose(D)
            % CLOSENESS_CENTRALITY_LOOSE  Returns the closeness centrality of the graph
            % defined by mutual-distance matrix [D].

            arguments% (Input)
                D (:, :) {mustBeNonnegative};
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            n = height(D);
            score = mean((n - 1) ./ sum(D, 2));
        end

        function score = closeness_centrality(G)
            % CLOSENESS_CENTRALITY  Variant of [Graphs.closeness_centrality_loose] that
            % takes a graph as input instead.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            score = Graphs.closeness_centrality_loose(distances(G));
        end

        function score = efficiency_loose(D)
            % EFFICIENCY_LOOSE  Returns the efficiency of the graph defined by
            % mutual-distance matrix [D].

            arguments% (Input)
                D (:, :) {mustBeNonnegative};
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            n = height(D);
            inv = D.^-1;
            inv(eye(n, "logical")) = 0;
            score = 1 / (n * (n - 1)) * sum(inv, "all");
        end

        function score = efficiency(G)
            % EFFICIENCY  Variant of [Graphs.efficiency_loose] that takes a graph as input
            % instead.

            arguments% (Input)
                G (1, 1) graph;
            end
            % arguments (Output)
            %     score (1, 1) {isNumeric};
            % end

            score = Graphs.efficiency_loose(distances(G));
        end


        function G = stretch(G, args)
            % STRETCH  Increases the girth of [G] to [args.girth] using [args.method].
            %
            % The girth of a graph is the length of the shortest cycle. If a graph has no
            % cycles, the girth is infinite.
            %
            % If [args.girth] is negative, all cycles are removed. If [args.girth] is 0, 1,
            % 2, or 3 no cycles are removed, because simple graphs always have a girth of at
            % least 3.
            %
            % The output is connected if and only if the input is connected.
            %
            % The [args.method] must be one of the following:
            % * "random": Removes random edges from random cycles.
            % * "least_cycles_steps": Removes the edge that is in the smallest non-zero
            %                         number of cycles of shortest length.
            % * "least_cycles": Removes the edge that is in the smallest non-zero number of
            %                   cycles of any length. This method requires prohibitive
            %                   amounts of memory for large graphs.
            % * "most_cycles_steps": Removes the edge that is in the largest number of
            %                        cycles of shortest length.
            % * "most_cycles": Removes the edge that is in the largest number of cycles of
            %                  any length. This method requires prohibitive amounts of
            %                  memory for large graphs.

            arguments% (Input)
                G (1, 1) graph;
                args.girth (1, 1) {mustBeInteger} = -1;
                args.method { ...
                    mustBeMember( ...
                        args.method, ...
                        ["random", "least_cycles_steps", "least_cycles", "most_cycles_steps", "most_cycles"] ...
                    ) ...
                } = "random";
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            if args.girth >= 0 && args.girth <= 3; return; end
            if args.girth < 0; args.girth = numnodes(G) + 1; end

            edgecount = numedges(G);
            for length = 3:(args.girth - 1)
                if args.method == "least_cycles" || args.method == "most_cycles"
                    [~, cycles] = allcycles(G, MinCycleLength = 3, MaxCycleLength = args.girth - 1);
                else
                    [~, cycles] = allcycles(G, MinCycleLength = length, MaxCycleLength = length);
                end
                if isempty(cycles); continue; end

                cycle_matrix = false([height(cycles), edgecount]);
                for i = 1:height(cycles); cycle_matrix(i, cycles{i}) = 1; end
                cycle_matrix = sparse(cycle_matrix);

                edges = 1:edgecount;
                removed = false([1, edgecount]);
                while true
                    removable = sum(cycle_matrix, 1) > 0;
                    edges = edges(removable);
                    cycle_matrix = cycle_matrix(:, removable);
                    if isempty(cycle_matrix); break; end

                    switch args.method
                        case "random"
                            best_idx = randi(numel(edges));
                        case {"least_cycles_steps", "least_cycles"}
                            cand_scores = sum(cycle_matrix, 1);
                            [~, best_idx] = min_rng(cand_scores);
                        case {"most_cycles_steps", "most_cycles"}
                            cand_scores = sum(cycle_matrix, 1);
                            [~, best_idx] = max_rng(cand_scores);
                    end

                    removed(edges(best_idx)) = 1;
                    cycle_matrix(cycle_matrix(:, best_idx), :) = [];  %#ok<SPRIX> It's actually faster!
                end

                edges = 1:edgecount;
                G = rmedge(G, edges(removed));
                edgecount = edgecount - sum(removed);
            end

            assert(Graphs.girth(G) >= args.girth, "Failed to stretch graph to desired girth.");
        end

        function G = minimise_leaves(G, args)
            % MINIMISE_LEAVES  Adds edges to minimise the number of nodes with only one
            % neighbour, while ensuring that the graph retains its [args.girth] and remains
            % connected.
            %
            % The algorithm starts by adding edges between leaves. Once no viable candidates
            % remain, edges are added between leaves and non-leaves, again until no viable
            % candidates remain.
            %
            % When multiple candidates exist, the [args.method] determines which candidate
            % is chosen. If the [args.method] has multiple best candidates, a random best
            % candidate is chosen.
            %
            % This function does not guarantee that no leaves remain afterwards. In some
            % graphs (such as star topologies), adding any single edge would reduce the
            % girth below [args.girth].

            arguments% (Input)
                G (1, 1) graph;
                args.method (1, 1) {mustBeMember(args.method, ["random", "closest", "furthest"])};
                args.girth (1, 1) {mustBeInteger} = -1;
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            assert(Graphs.is_connected(G), "Cannot minimise leaves for disconnected graph.");
            if args.girth < 0; return; end

            nodes = 1:numnodes(G);
            leaves = nodes(degree(G) == 1);

            phase = 2;  % 2 = leaf <> leaf. 1 = leaf <> non-leaf. 0 = stop.
            while phase ~= 0 && numel(leaves) > 0
                if phase == 2
                    srcs = leaves';
                    dsts = leaves;
                else
                    srcs = leaves';
                    dsts = nodes;
                end

                dists = distances(G, srcs, dsts);
                dists(dists < args.girth - 1) = NaN;
                dists(srcs == dsts) = NaN;

                candidates = reshape(dists, 1, []);
                if args.method == "closest"
                    [score, winner] = min_rng(candidates);
                elseif args.method == "furthest"
                    [score, winner] = max_rng(candidates);
                elseif args.method == "random"
                    [score, winner] = min_rng(candidates .* 0);
                end
                if isnan(score); phase = phase - 1; continue; end

                [src_idx, dst_idx] = ind2sub(size(dists), winner);
                src = srcs(src_idx);
                dst = dsts(dst_idx);

                G = addedge(G, src, dst);
                leaves(leaves == src | leaves == dst) = [];
            end

            assert(Graphs.girth(G) >= args.girth, "Girth decreased below threshold.");
        end

        function G = optimise(G, args)
            % OPTIMISE  Adds and removes edges from [G] to optimise [args.metric], while
            % retaining the [args.girth], [args.leaves_minimised], and connectivity.
            %
            % Starting with [G], at each iteration, all graphs that differ in exactly one
            % edge from the current best graph (the "candidates") are scored by
            % [args.metric]. These scores are then passed to the chosen optimisation
            % [args.method]. If [args.direction] is "maximal", the [args.method] will
            % maximise the [args.metric], otherwise the [args.method] will minimise the
            % [args.metric].
            %
            % The following optimisation methods are available:
            % * "greedy": At each iteration, choose the optimal candidate. Repeat until all
            %             candidates are worse.
            % * "simulated_annealing": For [args.sa_t_max] iterations, choose one candidate
            %                          at random, with probabilities proportional to their
            %                          scores; optimal scores are more likely. If the score
            %                          is improved or a temperature-based threshold is met,
            %                          pick that candidate and set the threshold to 0.
            %                          Otherwise, increase the threshold proportional to the
            %                          current iteration number and [args.sa_d_thr], then go
            %                          to the next iteration without changing the graph. The
            %                          value [args.sa_d_thr] is typically negative, and the
            %                          threshold increment decreases as iteration number
            %                          goes up. This method is based on that presented in
            %                          doi:10.1063/1.296773.
            %
            % The following optimisation metrics are available:
            % * "eigenratio": The ratio between the second-smallest eigenvalue and the
            %                 largest eigenvalue of the graph's laplacian matrix. Typically
            %                 maximised.
            % * "algebraic_connectivity": The second-smallest eigenvalue. Typically
            %                             maximised.
            % * "closeness_centrality": The mean over each node's reciprocal mean distance
            %                           to each other node. Typically maximised.
            % * "efficiency": The mean of all reciprocal pairwise node distances. Typically
            %                 maximised.
            %
            % To minimise computational costs, the greedy algorithm only updates the data
            % structures [A], [d], and [D] when a new best graph is found. As a result, the
            % function [args.metric] receives the data structures [A], [d], and [D]
            % corresponding to the old best graph, and should update those data structures
            % necessary for its own functioning.
            %
            % The function referred to by [args.metric] should take the following arguments:
            % * [A], the old adjacency matrix,
            % * [d], the old degree vector,
            % * [D], the old node-node distance matrix,
            % * [src], the source of the changed edge, or `0` if nothing changed,
            % * [dst], the destination of the changed edge, or `0` if nothing changed,
            % * [connected], `true` if the edge from [src] to [dst] was added, and `false`
            %   if the edge from [src] to [dst] was removed; should be ignored if [src] or
            %   [dst] is `0`.

            arguments% (Input)
                G (1, 1) graph;
                args.method {mustBeMember(args.method, ["greedy", "simulated_annealing"])} = "greedy";
                args.metric { ...
                    mustBeMember( ...
                        args.metric, ...
                        ["eigenratio", "algebraic_connectivity", "closeness_centrality", "efficiency"] ...
                    ) ...
                } = "eigenratio";
                args.direction {mustBeMember(args.direction, ["minimum", "maximum"])} = "minimum";
                args.girth (1, 1) {mustBeInteger} = -1;
                args.leaves_minimised (1, 1) logical = false;
                args.sa_t_max (1, 1) {mustBeInteger, mustBePositive} = 100;
                args.sa_d_thr (1, 1) {mustBeNumeric} = 0.5;
            end
            % arguments (Output)
            %     G (1, 1) graph;
            % end

            % Parse arguments
            nodecount = numnodes(G);

            if args.girth < 0; args.girth = nodecount + 1; end
            if args.leaves_minimised; leaf_count = Graphs.leaf_count(G); end

            switch args.metric
                case "eigenratio"
                    score_fun = ...
                        @(A, d, ~, src, dst, connected) ...
                            Graphs.eigenratio_loose( ...
                                Graphs.update_adjacency(A, src, dst, connected), ...
                                Graphs.update_degree(d, src, dst, connected) ...
                            );
                case "algebraic_connectivity"
                    score_fun = ...
                        @(A, d, ~, src, dst, connected) ...
                            Graphs.algebraic_connectivity_loose( ...
                                Graphs.update_adjacency(A, src, dst, connected), ...
                                Graphs.update_degree(d, src, dst, connected) ...
                            );
                case "closeness_centrality"
                    score_fun = ...
                        @(A, ~, D, src, dst, connected) ...
                            Graphs.closeness_centrality_loose( ...
                                Graphs.update_distances(... 
                                    D, ...
                                    Graphs.update_adjacency(A, src, dst, connected), ...
                                    src, ...
                                    dst, ...
                                    connected ...
                                ) ...
                            );
                case "efficiency"
                    score_fun = ...
                        @(A, ~, D, src, dst, connected) ...
                            Graphs.efficiency_loose( ...
                                Graphs.update_distances( ...
                                    D, ...
                                    Graphs.update_adjacency(A, src, dst, connected), ...
                                    src, ...
                                    dst, ...
                                    connected ...
                                ) ...
                            );
            end

            if args.direction == "minimum"
                opt_rng = @min_rng;
                sa_score_weights = @(scores) -exp(scores);
                is_better = @(new, old) new < old;
            else
                opt_rng = @max_rng;
                sa_score_weights = @(scores) exp(scores);
                is_better = @(new, old) new > old;
            end

            % Run optimisation
            A = logical(adjacency(G));
            A_row = repmat((1:nodecount)', [1, nodecount]);
            A_col = A_row';
            d = degree(G);
            D = distances(G);

            best_score = score_fun(A, d, D, 0, 0, false);
            sa_t = 0; sa_thr = 0;
            while true
                sa_t = sa_t + 1;
                if args.method == "simulated_annealing" && sa_t > args.sa_t_max; break; end

                % Removals
                [~, cycles] = cyclebasis(G);
                rmv = unique(horzcat(cycles{:}));
                [rmv_srcs, rmv_dsts] = findedge(G, rmv);
                if args.leaves_minimised
                    rmv_eligible = d(rmv_srcs) > 2 & d(rmv_dsts) > 2;
                    rmv = rmv(rmv_eligible);
                    rmv_srcs = rmv_srcs(rmv_eligible);
                    rmv_dsts = rmv_dsts(rmv_eligible);
                end
                if isempty(rmv)
                    rmv_scores = [];
                    rmv_best_score = nan;
                else
                    rmv_scores = arrayfun(@(src, dst) score_fun(A, d, D, src, dst, false), rmv_srcs, rmv_dsts);
                    [rmv_best_score, rmv_best_idx] = opt_rng(rmv_scores);
                end

                % Additions
                add = triu(~A & (D >= args.girth - 1), 1);
                add_srcs = A_row(add);
                add_dsts = A_col(add);
                if isempty(add_srcs)
                    add_scores = [];
                    add_best_score = nan;
                else
                    add_scores = arrayfun(@(src, dst) score_fun(A, d, D, src, dst, true), add_srcs, add_dsts);
                    [add_best_score, add_best_idx] = opt_rng(add_scores);
                end

                % Find best
                if isnan(rmv_best_score) && isnan(add_best_score); break; end

                if args.method == "greedy"
                    if ~isnan(rmv_best_score) && (isnan(add_best_score) || is_better(rmv_best_score, add_best_score))
                        assert(~isnan(rmv_best_score));
                        if ~is_better(rmv_best_score, best_score); break; end
    
                        best_src = rmv_srcs(rmv_best_idx);
                        best_dst = rmv_dsts(rmv_best_idx);
                        best_score = rmv_best_score;
                        best_added = false;
                    else
                        assert(~isnan(add_best_score));
                        if ~is_better(add_best_score, best_score); break; end
    
                        best_src = add_srcs(add_best_idx);
                        best_dst = add_dsts(add_best_idx);
                        best_score = add_best_score;
                        best_added = true;
                    end
                elseif args.method == "simulated_annealing"
                    all_scores = vertcat(rmv_scores, add_scores);
                    all_weights = sa_score_weights(all_scores);

                    found_good = false;
                    while true
                        if sa_t > args.sa_t_max || found_good; break; end

                        batch_size = 1000;  % Sample random values in batches
                        best_idxs = randsample(numel(all_scores), batch_size, true, all_weights)';
                        crossings = rand([1, batch_size]);
    
                        for batch_idx = 1:batch_size
                            if sa_t > args.sa_t_max; break; end

                            best_idx = best_idxs(batch_idx);
                            if best_idx <= numel(rmv)
                                best_src = rmv_srcs(best_idx);
                                best_dst = rmv_dsts(best_idx);
                                best_added = false;
                            else
                                best_src = add_srcs(best_idx - numel(rmv));
                                best_dst = add_dsts(best_idx - numel(rmv));
                                best_added = true;
                            end
                            new_best_score = score_fun(A, d, D, best_src, best_dst, best_added);
                            crosses_threshold = crossings(batch_idx) <= max(0, min(1, sa_thr - new_best_score - best_score));
    
                            if is_better(new_best_score, best_score) || crosses_threshold
                                found_good = true;
                                break;
                            end

                            sa_thr = sa_thr + args.sa_d_thr / log(sa_t + 1);
                            sa_t = sa_t + 1;
                        end
                        if found_good; break; end
                    end
                    if ~found_good; break; end

                    sa_thr = 0;
                    best_score = new_best_score;
                end

                % Update structures
                A = Graphs.update_adjacency(A, best_src, best_dst, best_added);
                G = graph(A);
                d = Graphs.update_degree(d, best_src, best_dst, best_added);
                D = Graphs.update_distances(D, A, best_src, best_dst, best_added);
            end

            G = graph(A);

            assert(Graphs.girth(G) >= args.girth, "Girth decreased below threshold.");
            if args.leaves_minimised; assert(Graphs.leaf_count(G) <= leaf_count, "Leaves were added."); end
        end


        function A = update_adjacency(A, src, dst, connected)
            % UPDATE_ADJACENCY  Updates the adjacency matrix [A]: If [connected] is `true`,
            % the edge between [src] and [dst] is added, and otherwise the edge between
            % [src] and [dst] is removed.
            %
            % This function does not validate the inputs. If an already-existing edge is
            % added, or a non-existing edge is removed, behaviour is undefined.

            if src == 0 || dst == 0; return; end

            A(src, dst) = connected;
            A(dst, src) = connected;
        end

        function d = update_degree(d, src, dst, connected)
            % UPDATE_DEGREE  Updates the degree vector [d]: If [connected] is `true`, the
            % degree of both nodes is increased, and otherwise the degree of both nodes is
            % decreased.
            %
            % This function does not validate the inputs. If an already-existing edge is
            % added, or a non-existing edge is removed, or a self-loop is added, behaviour
            % is undefined.

            if src == 0 || dst == 0; return; end

            both = [src, dst];
            d(both) = d(both) + (connected * 2 - 1);
        end

        function D = update_distances(D, A, src, dst, connected)
            % UPDATE_DISTANCES  Updates the node-node distance matrix [D]: If [connected] is
            % `true`, then for every pair of nodes `(u, v)` the two new paths `u -> src ->
            % dst -> v` and `u -> dst -> src -> v` are considered, and the minimum of the
            % old path and the two new paths is used in the updated [D]; otherwise, if
            % [connected] is `false`, all distances are recalculated based on the given
            % adjacency matrix [A].
            %
            % This function does not validate the inputs. If an already-existing edge is
            % added, or a non-existing edge is removed, behaviour is undefined.

            if src == 0 || dst == 0; return; end

            if connected
                D_ = D(src, :) + D(:, dst) + 1;
                D = min(D, min(D_, D_'));
            else
                D = distances(graph(A));
            end
        end
    end
end
