classdef ExperimentConfig < Config
    % EXPERIMENTCONFIG  Configuration for running an [Experiment].

    properties
        % NODE_COUNT  The closed range from which to pick a uniformly random number of
        % nodes in the network.
        node_count (1, 2) {mustBeInteger, mustBePositive} = [25, 100];
        % NETWORK_LAYOUT  Graph type to use for the network layout.
        %
        % Must be one of: "erdos_renyi", "watts_strogatz", "barabasi_albert",
        % "geometric_random", "complete", "empty".
        network_layout (1, 1) { ...
            mustBeMember( ...
                network_layout, ...
                ["erdos_renyi", "watts_strogatz", "barabasi_albert", "geometric_random", "complete", "empty"] ...
            ) ...
        } = "erdos_renyi";
        % NETWORK_ERDOS_RENYI_P  Maps the number of nodes to the desired `p` parameter
        % for Erdős-Renyi graphs.
        %
        % Ignored unless [network_layout] is `"erdos_renyi"`.
        %
        % See also [Graphs#generate_erdos_renyi].
        network_erdos_renyi_p (1, 1) = @(~) 0.1;
        % NETWORK_WATTS_STROGATZ_K  Maps the number of nodes to the desired `k`
        % parameter for Watts-Strogatz graphs.
        %
        % Ignored unless [network_layout] is `"watts_strogatz"`.
        %
        % See also [Graphs#generate_watts_strogatz].
        network_watts_strogatz_k (1, 1) = @(~) 12;
        % NETWORK_WATTS_STROGATZ_P  Maps the number of nodes to the `p` parameter for
        % Watts-Strogatz graphs.
        %
        % Ignored unless [network_layout] is `"watts_strogatz"`.
        %
        % See also [Graphs#generate_watts_strogatz].
        network_watts_strogatz_p (1, 1) = @(~) 0.5;
        % NETWORK_BARABASI_ALBERT_M  Maps the number of nodes to the `m` parameter for
        % Barabási-Albert graphs.
        %
        % Ignored unless [network_layout] is `"barabasi_albert"`.
        %
        % See also [Graphs#generate_barabasi_albert].
        network_barabasi_albert_m (1, 1) = @(~) 12;
        % NETWORK_GEOMETRIC_RANDOM_D  Maps the number of nodes to the `d` parameter for
        % geometric random graphs.
        %
        % Ignored unless [network_layout] is `"geometric_random"`.
        %
        % See also [Graphs#generate_geometric_random].
        network_geometric_random_d (1, 1) = @(~) 2;
        % NETWORK_GEOMETRIC_RANDOM_R  Maps the number of nodes and dimensions to the `r`
        % parameter for geometric random graphs.
        %
        % Ignored unless [network_layout] is `"geometric_random"`.
        %
        % See also [Graphs#generate_geometric_random].
        network_geometric_random_r (1, 1) = @(~) 0.3;
        % NETWORK_STRETCH_GIRTH  The girth to which the graph should be stretched
        % (at least).
        %
        % Setting this value to -1 corresponds to a girth of infinite, thus
        % removing all cycles. Setting this value to 0, 1, 2, or 3 results in no
        % cycles being removed, since every graph has girth at least 3.
        %
        % See also [Graphs#stretch].
        network_stretch_girth (1, 1) {mustBeInteger} = 0;
        % NETWORK_STRETCH_METHOD  The method by which edges should be selected for
        % removal during stretching.
        %
        % Must be one of: "random", "least_cycles_steps", "least_cycles",
        % "most_cycles_steps", "most_cycles".
        %
        % Does nothing if [network_stretch_girth], but is not ignored by the
        % [Config#cache_id].
        %
        % See also [Graphs#stretch].
        network_stretch_method (1, 1) { ...
            mustBeMember( ...
                network_stretch_method, ...
                ["random", "least_cycles_steps", "least_cycles", "most_cycles_steps", "most_cycles"] ...
            ) ...
        } = "random";
        % NETWORK_MINIMISE_LEAVES_METHOD  The method by which leaves should be connected
        % to each other to minimise the number of leaves.
        %
        % Must be one of: "none", "random", "closest", "furthest".
        %
        % See also [Graphs#remove_leaves].
        network_minimise_leaves_method (1, 1) { ...
            mustBeMember(network_minimise_leaves_method, ["none", "random", "closest", "furthest"]) ...
        } = "none";
        % NETWORK_OPTIMISE_METRIC  The metric by which the graph should be
        % optimised after stretching.
        %
        % Must be one of: "none", "eigenratio", "algebraic_connectivity",
        % "closeness_centrality", "efficiency".
        %
        % See also [Graphs#optimise].
        network_optimise_metric (1, 1) { ...
            mustBeMember( ...
                network_optimise_metric, ...
                ["none", "eigenratio", "algebraic_connectivity", "closeness_centrality", "efficiency"] ...
            ) ...
        } = "none";
        % NETWORK_OPTIMISE_DIRECTION  The direction to optimise in.
        %
        % Must be one of: "minimum", "maximum".
        %
        % Ignored if [network_optimise_metric] is `"none"`.
        %
        % See also [Graphs#optimise].
        network_optimise_direction (1, 1) { ...
            mustBeMember(network_optimise_direction, ["minimum", "maximum"]) ...
        } = "minimum";
        % NETWORK_REQUIRE_CONNECTED  `true` if and only if the created network must
        % consist of a single connected component.
        network_require_connected (1, 1) logical = true;
        % NETWORK_MAX_TRIALS  The maximum number of attempts to generate a network
        % before erroring.
        %
        % This field is not considered when caching. For example, setting this field to
        % 100 may result in loading a cached result that generated a network only in the
        % 200th attempt, even though the new configuration would fail to generate a
        % network altogether.
        network_max_attempts (1, 1) {mustBeInteger, mustBePositive} = 250;

        % DA_REPETITIONS  Number of times to run the distributed averaging protocol.
        da_repetitions (1, 1) {mustBeInteger, mustBePositive} = 1;
        % DA_MAX_ROUNDS  Number of distributed averaging rounds to run, or a negative
        % number to not limit the experiment by the number of rounds.
        da_max_rounds (1, 1) {mustBeInteger} = 50;
        % DA_TIME_MODEL  The coordination model used in the distributed averaging
        % protocol.
        %
        % Must be one of:
        % * "synchronous", to have all users update at the same time, or
        % * "asynchronous", to have one user update in each round.
        da_coordination (1, 1) {mustBeText, ...
                                mustBeMember(da_coordination, ["synchronous", "asynchronous"])} = "asynchronous";
        % DA_NEIGHBOR_METHOD  The method by which to select the neighbors to perform an
        % update with.
        %
        % Must be one of: "single", "multiple", "all".
        %
        % Ignored if [da_coordination] is `"synchronous"`.
        da_neighbor_method (1, 1) {mustBeText, ...
                                   mustBeMember(da_neighbor_method, ["single", "multiple", "all"])} = "single";
        % DA_UPDATE_DIRECTION  The direction in which the averaging calculation affects
        % the involved parties.
        %
        % Must be one of: "push", "pull", "push_pull".
        %
        % Ignored if [da_coordination] is `"synchronous"`.
        da_update_direction (1, 1) { ...
            mustBeText, ...
            mustBeMember(da_update_direction, ["push", "pull", "push_pull"]) ...
        } = "push_pull";
        % DA_CONVERGENCE_THRESHOLD  Largest difference between model parameters before
        % convergence is achieved, or a negative number to not limit the experiment by
        % convergence.
        da_convergence_threshold (1, 1) {mustBeNumeric} = -1;
        % DA_CONVERGENCE_METHOD  The formula to use to calculate the level of
        % convergence that has been achieved.
        %
        % Must be one of:
        % * "mutual_distance", which calculates the largest distance between any two
        %   values,
        % * "max_error", which calculates the largest distance to the actual mean, and
        % * "error_norm", which normalizes the max error by the actual norm.
        da_convergence_method (1, 1) { ...
            mustBeText, ...
            mustBeMember(da_convergence_method, ["mutual_distance", "max_error", "error_norm"]) ...
        } = "mutual_distance";
    end


    methods
        function obj = ExperimentConfig(strct)
            arguments% (Input)
                strct struct = struct();
            end
            % arguments (Output)
            %     obj (1, 1) ExperimentConfig;
            % end

            obj@Config(strct);
        end


        function culled = cache_cull(obj)
            arguments% (Input)
                obj (1, 1) ExperimentConfig;
            end
            % arguments (Output)
            %     culled (1, 1) struct;
            % end

            culled = struct(node_count = obj.node_count, ...
                            network_layout = obj.network_layout, ...
                            network_stretch_girth = obj.network_stretch_girth, ...
                            network_optimise_metric = obj.network_optimise_metric, ...
                            network_minimise_leaves_method = obj.network_minimise_leaves_method, ...
                            network_require_connected = obj.network_require_connected, ...
                            network_max_attempts = 1, ...
                            ...
                            da_repetitions = obj.da_repetitions, ...
                            da_max_rounds = obj.da_max_rounds, ...
                            da_coordination = obj.da_coordination, ...
                            da_convergence_threshold = obj.da_convergence_threshold, ...
                            da_convergence_method = obj.da_convergence_method);

            if culled.network_layout == "erdos_renyi"
                culled.network_erdos_renyi_p = char(obj.network_erdos_renyi_p);
            elseif culled.network_layout == "watts_strogatz"
                culled.network_watts_strogatz_k = char(obj.network_watts_strogatz_k);
                culled.network_watts_strogatz_p = char(obj.network_watts_strogatz_p);
            elseif culled.network_layout == "barabasi_albert"
                culled.network_barabasi_albert_m = char(obj.network_barabasi_albert_m);
            elseif culled.network_layout == "geometric_random"
                culled.network_geometric_random_d = char(obj.network_geometric_random_d);
                culled.network_geometric_random_r = char(obj.network_geometric_random_r);
            end

            culled.network_stretch_method = obj.network_stretch_method;  % Not ignored for low girths, because of plots

            if culled.network_optimise_metric ~= "none"
                culled.network_optimise_direction = obj.network_optimise_direction;
            end

            if culled.da_coordination ~= "synchronous"
                culled.da_neighbor_method = obj.da_neighbor_method;
                culled.da_update_direction = obj.da_update_direction;
            end
        end
    end
end
