classdef (Abstract) Config < matlab.mixin.Copyable  % copyable handle
    % CONFIG  Contains settings.
    %
    % Subclasses should perform appropriate type-checking.


    methods
        function obj = Config(strct)
            % CONFIG  Creates a new [Config] object, assigning properties using the fields
            % in the [strct].

            fn = fieldnames(strct);
            for i = 1:numel(fn)
                obj.(fn{i}) = strct.(fn{i});
            end
        end


        function obj = set(obj, strct)
            % SET  Copies properties from [strct] into this [obj] instance.

            arguments% (Input)
                obj (1, 1) Config;
                strct (1, 1) struct;
            end
            % arguments (Output)
            %     obj (1, 1) Config;
            % end

            fn = fieldnames(strct);
            for i = 1:numel(fn)
                obj.(fn{i}) = strct.(fn{i});
            end
        end

        function culled = cache_cull(obj)
            % CACHE_CULL  Returns a struct from [obj] in which all fields that are not
            % relevant for determining the [cache_id] have been culled.

            arguments% (Input)
                obj (1, 1) Config;
            end
            % arguments (Output)
            %     obj (1, 1) struct;
            % end

            culled = struct(obj);
        end

        function h = cache_id(obj)
            % CACHE_ID  Returns an ID that uniquely identifies the properties of this
            % [Config] in a cache.
            %
            % True uniqueness isn't actually guaranteed. Collisions may still occur. Check
            % yourself before you wreck yourself.

            arguments% (Input)
                obj (1, 1) Config;
            end
            % arguments (Output)
            %     h (1, 1) {mustBeText};
            % end

            h = hash(obj.cache_cull());
        end
    end

    methods (Static)
        function combinations = combinations(varargin)
            % COMBINATIONS  Given multiple structs in [varargin], returns all possible
            % combinations.
            %
            % Each argument is a cell array containing multiple structs, each struct
            % describing a (partially completed) [Config]. Then, all possible combinations
            % are created by selecting one struct from each cell array and concatenating
            % them into a new struct.

            arguments (Repeating)% (Input, Repeating)
                varargin (:, 1) cell {mustBeNonempty};  % cell<struct>  (for each input)
            end
            % arguments (Output)
            %     combinations (1, :) cell;  % cell<struct>
            % end

            argin = cellfun(@(arg) cellfun(@namedargs2cell, arg, UniformOutput = false), ...
                            varargin, ...
                            UniformOutput = false);

            grid = cell(numel(argin), 1);
            [grid{:}] = ndgrid(argin{:});

            arguments = cellfun(@(it) it(:), grid, UniformOutput = false)';
            arguments = [arguments{:}];

            combinations = cell([1, height(arguments)]);
            for it = 1:height(arguments)
                combination = horzcat(arguments{it, :});
                combinations{it} = cell2struct(combination(2:2:end)', string(combination(1:2:end)));
            end
        end
    end
end
