"""
File: logic.py
Description: This script analyzes Wi-Fi signal fluctuations as discussed in Section 4.2 
             ("Wi-Fi Signal Analysis") of the paper "A zone-based Wi-Fi fingerprinting indoor 
             positioning system for factory noise mapping." It processes raw data and evaluates 
             signal stability to support indoor positioning and noise mapping in a factory setting.
             Data collected by Thinkpad P16v and antenna

Author: L. Xiao (l.xiao@utwente.nl)
Date: March 7, 2025

License: This project is licensed under the MIT License - see the LICENSE file for details.

Copyright (c) 2025 L. Xiao
"""

import os
import time
import threading
import csv
import pandas as pd
import winsound

from PyQt5.QtCore import QObject, QThread, pyqtSignal

DEFAULT_SAVE_FOLDER = "./FingerPrintWifi/result_tmp"
DEFAULT_DUPLICATE_ALLOW = False
# Now 60s / 4s = 15, 30 is too long total 2 hour, 10 is enough for the prove of concept
DEFAULT_DATA_AMOUNT = 50
# Ask the subthread for current rssi list, subthread update cycle = 3s, 1.1
DEFAULT_INQUIRE_TIME_S = 1.1   # 3.5 not enough..
# TODO Now change here, must make sure the wifi name (no space)
DEFAULT_BSSID_LIST = [
    # UThings priority
    "48:8b:0a:ca:96:c2:",   # Front right, 2.4GHz
    "48:8b:0a:ca:be:a1:",   # Front left, 2.4G
    "00:a2:ee:fd:96:62:",   # BACK LEFT CORNER, 2.4G
    "48:8b:0a:b3:a9:62:",   # BACK RIGHT CORNER, 2.4G
    "00:ea:bd:73:ea:a3:",   # close entrace, 2.4G
    "00:ea:bd:74:21:03:",   # far entrance, 2.4G
]

DEFAULT_PATH_FLUCT_METRIC = './FingerPrintWifi/result_rssi/wifi_signal_metrics.csv'
PATH_RSSI_FLUCTUATUION = r'./RSSI_fluctuate_6AP_Sample400.csv'
DATA_HEADER_RSSI    = ["AP1", "AP2", "AP3", "AP4", "AP5", "AP6"]

# Only memory to get wifi level from, use lock please
glb_wifi_rssi = 0
# Only memory to store the current rssi list
glb_rssi_list = []
glb_wifi_lock = threading.Lock()


class WifiMonitor(QThread):
    """
    Two backend threads consistently to update noise and wifi level
    """
    update_progress = pyqtSignal(int)
    
    def __init__(self, parent: QObject | None = None) -> None:
        super().__init__(parent)
        # Dataframe type storing the indexed position
        self.__df_pos: pd.DataFrame = None
        self.save_folder = DEFAULT_SAVE_FOLDER
        # Zone or position index read from UI
        self.pos_index = -1
        self.pos_x = -1
        self.pos_y = -1
        self.foldpath = "./result"
        self.inner_thread = None
        self.is_running = True
        self.bssi_list = DEFAULT_BSSID_LIST
    
    def __del__(self):
        print(f"Thread Wifi Monitor RSSI is being destroyed.")

    def rssi_fluctuate(self, pNum):
        """
        For the RSSI fluctuation experiment, the RSSI variations of different APs at the same location were analyzed. 
        The MSE, MSEA, and instability levels were measured.
        
        param pNum how many numbers collect
        """
        if self.is_running == False:
            self.start_querytrd()
        print(f"start collecting fluctuate rssi, num={pNum}")
        data = []
        count = 0
        last_rssi = self.get_rssi()
        while True:
            cur_rssi = self.get_rssi()
            if cur_rssi != None and cur_rssi != last_rssi:
                data.append(cur_rssi)
                last_rssi = cur_rssi
                count = count + 1
                print(f"count = {count}")
            if count == pNum:
                break
            time.sleep(0.2)

        df = pd.DataFrame(data, columns=["AP1", "AP2", "AP3", "AP4", "AP5", "AP6"])
        df.to_csv(PATH_RSSI_FLUCTUATUION, index=False)
        self.visualize_fluc(PATH_RSSI_FLUCTUATUION, DEFAULT_PATH_FLUCT_METRIC)
    
    def visualize_fluc(self, filename, savepath=None):
        import pandas as pd
        import matplotlib.pyplot as plt
        import numpy as np
        df = pd.read_csv(filename)

        std_devs = df.std().round(2)
        ranges = df.max() - df.min()
        mean_values = df.mean().round(2)
        mean_deviations = df.apply(lambda x: np.mean(np.abs(x - mean_values[x.name]))).round(2)

        for column in df.columns:
            print(f"Column '{column}':")
            print(f"mean_values: {mean_values}")
            print(f"  Standard Deviation: {std_devs[column]:.2f}")
            print(f"  Range: {ranges[column]}")
        
        results = [
            df.columns,
            mean_values,
            std_devs,
            ranges,
            mean_deviations
        ]
        row_names = ['AP seq', 'Mean', 'Standard Deviations', 'Ranges', 'Mean Deviations']
        df_results = pd.DataFrame(results)
        df_results.insert(0, 'Metric', row_names)
        if savepath != None:
            os.makedirs(os.path.dirname(savepath), exist_ok=True)
            df_results.to_csv(savepath, index=False, header=False)

        x = range(len(df))
        plt.figure(figsize=(10, 6))
        for i in range(df.shape[1]):
            plt.plot(x, df.iloc[:, i], marker='o', linestyle='-', label=df.columns[i])

        plt.title('Data Column Fluctuations')
        plt.xlabel('Index')
        plt.ylabel('Value')
        plt.xticks(ticks=x[::10])
        plt.legend()
        plt.grid(True)
        plt.show()


    def get_rssi(self):
        """
        Get the real-time rssi of several wifi source
        """
        if glb_rssi_list != None:
            return glb_rssi_list
        else:
            return None

    def init_position(self, filename, dist_unit = 10):
        """
        Init the position index and location info according to the config file
        """
        try:
            self.__df_pos = pd.read_csv(filename)
        except Exception as e:
            print(f"Exception when read csv, path={filename}")
            return

        self.__df_pos['X'] = self.__df_pos['X'] * dist_unit
        self.__df_pos['Y'] = self.__df_pos['Y'] * dist_unit
        self.__df_pos.set_index('Pos', inplace=True)

        return self.__df_pos

    def _run_wifinoise(self):
        """
        Old program to collect wifi and noise
        """
        if self.__df_pos.empty == False and self.pos_index != 0:
            if self.pos_index in self.__df_pos.index:
                self.pos_x = self.__df_pos.loc[self.pos_index, 'X']
                self.pos_y = self.__df_pos.loc[self.pos_index, 'Y']
        print(f"testing index: {self.pos_index}, pos X: {self.pos_x}, pos Y: {self.pos_y}")
        
        self.do_task(self.pos_x, self.pos_y, 1)
        self.update_progress.emit(self.pos_index)
    
    def run(self):
        """
        !QThread function will be called by .start function. Only for data collection, 
        """
        rssi_table = self.repeat_multirssi(DEFAULT_DATA_AMOUNT, DEFAULT_INQUIRE_TIME_S, DEFAULT_DUPLICATE_ALLOW)
        self.save_rssi(self.save_folder, rssi_table, self.pos_index)
        # Emit a beep sound when the data collection is finished
        try:
            winsound.Beep(1000, 2000)  # Frequency: 1000 Hz, Duration: 2500 ms
            print(f"Play sound???")
        except Exception as e:
            print(f"Error playing sound: {e}")

    def repeat_multirssi(self, count, sleep_time=2, duplicate=True):
        """ repeat collect count times of wifi data
            @param sleep_time (second) gap time collect data again, scan time is 2s,so should > 2s
            @param duplicate True allow collect duplicate fingerprint
            @return rssi_table has count numbers row data
        """
        print(f"repeat rssi data collect, count={count}, slp_time={sleep_time}, allow duplicate={duplicate}")
        rssi_table = []
        seq = 0
        prev_rssi = None
        while seq < count:
            time.sleep(sleep_time)
            print("ttttt=", seq)
            glb_wifi_lock.acquire()
            if glb_rssi_list != None:
                if duplicate == False:
                    if glb_rssi_list != prev_rssi:
                        rssi_table.append(glb_rssi_list)
                        seq = seq + 1
                else:
                    rssi_table.append(glb_rssi_list)
                    seq = seq + 1
                prev_rssi = glb_rssi_list
            glb_wifi_lock.release()
        
        return rssi_table
    
    def save_rssi(self, foldpath, rssi_table, znum):
        """ 
        Rely on the UI input index, update while clicking
        """
        if not os.path.exists(foldpath):
            print(f"create folder path = {foldpath}")
            os.makedirs(foldpath)

        filename = f"{znum}.csv"
        filepath = os.path.join(foldpath, filename)
        print(f"File save in {filepath}")
        with open(filepath, 'w', newline='') as file:
            writer = csv.writer(file)
            writer.writerows(rssi_table)

if __name__ == '__main__':
    worker = WifiMonitor()
    worker.visualize_fluc(PATH_RSSI_FLUCTUATUION)
