%%%%%%%%%%%%%%%%%%%%%%%%%%%% 2020-10-05  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                         %
%  This script simulates the contraction of a healing burn wound. The     %
%  algorithm is as follows:                                               %
%                                                                         %
%  For each time step take the following steps.                           %
%  1) Assemble matrices and vectors and solve for the constituents.       %
%  2) Perform inner Picard iterations to account for the nonlinearity.    %
%  3) Assemble matrices and vectors and solve for the displacement velo-  %
%     city and the effective strain.                                      %
%  4) Perform inner Picard iterations to account for the non-linearity.   %
%  5) Compute the displacement, update the mesh and report RSAW.          %
%  6) Compute the strain energy density.                                  %
%                                                                         %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

addpath('MatricesAndVectors');
addpath('simps');
day = zeros(T/dt-1,1); 
rsaw = ones(T/dt,1); 
estrain = zeros(T/dt,1);
t = 1;
while day(t) < T
    
    % 1) Assemble matrices and vectors and solve for the constituents.
    AssembleMass; % Mass matrix (lumped)
    AssembleN; % Fibroblasts
    AssembleM; % Myofibroblasts
    AssembleC; % Signaling molecules
    AssembleR; % Collagen molecules
    
    % Beforehand, store the distribution/density (z^n) at previous timepoint.
    N_n = N; M_n = M; c_n = c; rho_n = rho;
    
    F_n = zeros(n,1);                %
    F_n(:) = F_n(:) - S_n(:,1)*N_eq; %
    S_n(:,1) = 0;                    %
    S_n(1,:) = 0;                    % BC for Fibroblasts.
    S_n(1,1) = 1;                    %
    F_n(1) = N_eq;                   %
    N = (Mass + S_n*dt)\(Mass*N + F_n*dt); % Solve for fibroblasts (N^{n+1}_0).
    
    S_m(:,1) = 0; %
    S_m(1,:) = 0; % BC for myofibroblasts.
    S_m(1,1) = 1; %
    F_m(1)   = 0; %
    M = (Mass + S_m*dt)\(Mass*M + F_m*dt); % Solve for myofibroblasts (M^{n+1}_0).
    
    S_c(:,1) = 0; %
    S_c(1,:) = 0; % BC for signaling molecules.
    S_c(1,1) = 1; %
    c = (Mass + S_c*dt)\(Mass*c); % Solve for signaling molecules (c^{n+1}_0).
    
    rho = (Mass + S_r*dt)\(Mass*rho + F_r*dt); % Sove for collagen molecules (rho^{n+1}_0).
    
    % 2) Perform inner Picard iterations to account for the nonlinearity.
    % In each iteration, we compute the relative 1-norm of the difference
    % between subsequent approximations per variable. The iterations stop if
    % the maximum of the relative 1-norms is less than 10^{-5}.
    
    max_norm = Inf;
    while max_norm > 1e-5
        AssembleN;
        F_n = zeros(n,1);
        F_n(:) = F_n(:) - S_n(:,1)*N_eq;
        S_n(:,1) = 0;
        S_n(1,:) = 0;
        S_n(1,1) = 1;
        F_n(1) = N_eq;
        N1 = (Mass + S_n*dt)\(Mass*N_n + F_n*dt); % N^{n+1}_{k+1}
        norm_N = norm(abs(N1-N)/abs(N1),1); % Relative 1-norm of N.
        
        AssembleM;
        S_m(:,1) = 0;
        S_m(1,:) = 0;
        S_m(1,1) = 1;
        F_m(1)   = 0;
        M1 = (Mass + S_m*dt)\(Mass*M_n + F_m*dt);
        norm_M = norm(abs(M1-M)/abs(M1+1e-40),1); % Relative 1-norm of M.
        
        AssembleC;
        S_c(:,1) = 0;
        S_c(1,:) = 0;
        S_c(1,1) = 1;
        c1 = (Mass + S_c*dt)\(Mass*c_n);
        norm_c = norm(abs(c1-c)/abs(c1+1e-40),1); % Relative 1-norm of c.
        
        AssembleR;
        rho1 = (Mass + S_r*dt)\(Mass*rho_n + F_r*dt);
        norm_rho = norm(abs(rho1-rho)/abs(rho1),1); % Relative 1-norm of rho.
        
        max_norm = max([norm_N,norm_M,norm_c,norm_rho]);
        
        % Update z^{n+1}
        N = N1; M = M1; c = c1; rho = rho1;
    end
    
    % 3) Assemble matrices and vectors and solve for v and epsilon.
    AssembleV;
    AssembleE;
    
    % Beforehand, store the distributions at previous timepoint.
    v_n = v; epsilon_n = epsilon;
    
    % System of matrices and vectors.
    M1 = [M_v(2:n-1,2:n-1)+S_vv(2:n-1,2:n-1)*dt  ,  S_ve(2:n-1,1:n)*dt; ...
        S_ev(1:n,2:n-1)*dt                     ,  M_e+S_ee*dt];
    ve = M1 \ ([M_v(2:n-1,2:n-1)*v(2:n-1)+f_v(2:n-1)*dt;M_e*epsilon]);
    v = vertcat(0,ve(1:n-2),0); epsilon = ve(n-1:length(ve));
    
    % 4) Perform inner Picard iterations to account for the non-linearity.
    % In each iteration, we compute the relative 1-norm of the difference
    % between subsequent approximations per variable. The iterations stop if
    % the maximum of the relative 1-norms is less than 10^{-5}.
    
    max_norm = Inf;
    while max_norm > 1e-5
        AssembleV;
        AssembleE;
        M1 = [M_v(2:n-1,2:n-1)+S_vv(2:n-1,2:n-1)*dt , S_ve(2:n-1,1:n)*dt; ...
            S_ev(1:n,2:n-1)*dt                    , M_e+S_ee*dt];
        ve = M1 \ ([M_v(2:n-1,2:n-1)*v_n(2:n-1) + f_v(2:n-1)*dt; ...
            M_e*epsilon_n]);
        v1 = vertcat(0,ve(1:n-2),0); % Retrieve v.
        epsilon1 = ve(n-1:length(ve)); % Retrieve epsilon.
        
        norm_v = norm(abs(v1-v)/abs(v1),1); % Relative 1-norm of v.
        norm_e = norm(abs(epsilon1-epsilon)/abs(epsilon1),1); % Relative 1-norm of epsilon.
        max_norm = max([norm_v,norm_e]);
        
        % Update v^{n+1} and epsilon^{n+1}
        v = v1; epsilon = epsilon1;
    end
    
    % 5) Compute the displacement, update the mesh and report RSAW.
    u = u + dt*v;
    x = x_init(1:n) + u';
    rsaw(t+1) = abs(x(lW))/Lw;

    plot(ax1,x,N); plot(ax2,x,M); plot(ax3,x,c); plot(ax4,x,rho); drawnow
    
    % 6) compute the strain energy density
    integrand = vertcat(mu*diff(v),0) + E*sqrt(rho).*epsilon.*epsilon;
    estrain(t+1) = simps(x,integrand);
    
    % Variables to record the day and time.
    t = t + 1;
    day(t) = day(t-1) + dt;
end