// Input from Raspi port1:pwm1:duration1, port2:pwm2:duration2, ..., port7:pwm7:duration7Serial1
const int actuatorPins[7] = {3, 4, 5, 6, 7, 8, 9}; // PWM pins for 7 actuators
// 3-5 are SMA instructions, connected to 5V power; 6 is the SMP cooling system instruction; 7-9 are SMP heating elements
int currentPWM[7] = {0};  // Stores the current PWM state for each actuator, range: 0-255
unsigned long endTime[7] = {0}; // Records the time for scheduled execution end
bool indefinite[7] = {false}; // Flags which actuators are running indefinitely
bool isRunning[7] = {false};  // Tracks the running status of actuators
bool systemRunning = true; // Determines whether the system should continue running

void setup() {
    Serial.begin(9600); // Initialize the serial communication for PC
    // Serial1.begin(9600); // Initialize serial communication for Raspberry Pi

    for (int i = 0; i < 7; i++) {
        pinMode(actuatorPins[i], OUTPUT); // Set each actuator pin as output
        analogWrite(actuatorPins[i], 0);  // Initially set to off
        isRunning[i] = false;  // Initial state is stopped
    }

    // Send "Arduino is ready" to Raspberry Pi
    Serial.println("Arduino is ready");
}

void loop() {
    if (systemRunning && Serial.available() > 0) {
        String command = Serial.readStringUntil('\n');  // Receive commands from Raspberry Pi
        if (command == "q") {
            // If 'q' is received, stop all actuators and exit
            Serial.println("Exit already!");
            quitProgram();
        } else if (command == "wake") {
            // If 'wake' command is received, send message to the host
            Serial.println("Hi Raspi, Arduino is ready!");
        } else {
            processCommand(command);
        }
    }

    if (systemRunning) {
        // Check execution time for each actuator
        for (int i = 0; i < 7; i++) {
            if (!indefinite[i] && millis() > endTime[i]) { // If end time has been reached
                if (isRunning[i]) { // If currently running, stop and send feedback
                    analogWrite(actuatorPins[i], 0);  // Stop PWM output
                    currentPWM[i] = 0;  // Set current PWM to 0
                    isRunning[i] = false;  // Set to stopped state
                    Serial.print("Port ");
                    Serial.print(i + 1);
                    Serial.println(" has stopped");
                }
            }
        }
    }
}

// Stop all ports and exit the program
void quitProgram() {
    for (int i = 0; i < 7; i++) {
        analogWrite(actuatorPins[i], 0);  // Set PWM to 0 for all ports
        currentPWM[i] = 0;
        isRunning[i] = false;
    }
    Serial.println("All ports have been stopped. Quitting the program.");
    systemRunning = false;  // Set system to stopped state
}

// Parse and process commands
void processCommand(String command) {
    // Split the command by ',' to get each actuator's control segment
    int currentIndex = 0;
    int port, pwm, duration;

    for (int i = 0; i < 7; i++) {
        int separatorIndex = command.indexOf(',', currentIndex);
        String segment = (separatorIndex != -1) ? command.substring(currentIndex, separatorIndex) : command.substring(currentIndex);
        currentIndex = separatorIndex + 1;

        // Parse each actuator's command segment
        if (parseSegment(segment, port, pwm, duration)) {
            port--;  // Convert 1-7 to array index 0-6
            analogWrite(actuatorPins[port], pwm);  // Set PWM for the corresponding port
            currentPWM[port] = pwm;

            // If duration is provided, set timer; otherwise, set to indefinite PWM state
            if (duration > 0) {
                endTime[port] = millis() + (duration * 1000);  // Record end time
                indefinite[port] = false;
                if (!isRunning[port]) { // If previously stopped, now start
                    isRunning[port] = true;  // Mark as running
                    Serial.print("Port ");
                    Serial.print(port + 1);
                    Serial.print(" started with PWM ");
                    Serial.print(pwm);
                    Serial.print(" for ");
                    Serial.print(duration);
                    Serial.println(" seconds");
                }
            } else {
                indefinite[port] = true;
                if (!isRunning[port]) { // If previously stopped, now start
                    isRunning[port] = true;  // Mark as running
                    Serial.print("Port ");
                    Serial.print(port + 1);
                    Serial.print(" started with PWM ");
                    Serial.print(pwm);
                    Serial.println(" indefinitely");
                }
            }
        }
    }
}

// Parse each actuator's command segment in the format 'port:pwm:duration'
bool parseSegment(String segment, int &port, int &pwm, int &duration) {
    int firstColon = segment.indexOf(':');
    int secondColon = segment.indexOf(':', firstColon + 1);

    if (firstColon == -1 || secondColon == -1) {
        // Return false if not enough colons are found, indicating parsing failure
        return false;
    }

    port = segment.substring(0, firstColon).toInt();
    pwm = segment.substring(firstColon + 1, secondColon).toInt();
    String durationStr = segment.substring(secondColon + 1);
    duration = durationStr.length() > 0 ? durationStr.toInt() : 0;  // If duration is missing, set to 0

    // If pwm value is 100, reduce by 1 to set it to 99
    if (pwm == 100) {
        pwm = 99;
    }

    // Map PWM value from 0-100 to 0-255
    pwm = map(pwm, 0, 100, 0, 255);
    
    return true;
}
