#!/bin/bash 
# created 16-12-2020
# tested 16-12-2020 unless otherwise commented

softfail () { 

	echo "$1" failed >> ${dirout}/fail.log
	#read -n1 -r -p "Press any key to continue..." key 
	#exit 1 
}

#########################################################
# DO PAY MAXIMUM ATTENTION TO THESE PARAMETERS
#########################################################

# simulation matters

# choose name of test case (suffix _Def.xml will be appended below)
export name=Meringolo

# choose resolution H/dx
resolution=0800;
label=$(date +%y%m%d-%H%M%S)

# hardware for GPU tasks
mygpu=1 # this is device 2 according to nvidia-smi
mygpu=0 # this is device 1 according to nvidia-smi

# multithreading for CPU tasks
nthreads=12

# what to do? pre and proper processing (set to 1 to activate)
setgencase=0
setdsph=0
setgauges=0
settimesteps=0

# what to do? postprocessing (set to 1 to activate)
setparticles=0

# what to do? compression tasks (set to 1 to activate)
setcgencase=0
setcdata=0
setcgauges=0
setctimesteps=0
setcparticles=0

#########################################################

# output matters

# set directory where to save data and create it
# the location where data are stored is set as an environment variable ${DATADIR}
export dirout=${DATADIR}/sph.dualsph.50/${name}_${label} # customize this path
echo $dirout
mkdir -p ${dirout}/gencase/
export diroutdata=${dirout}/data
sleep 2

# executable matters

export dirbin=${DSPH50g}  # the location of binary files is set as an environment variable
test ! -d "${dirbin}" && echo environment variable dirbin=${dirbin} not found && exit

export LD_LIBRARY_PATH=${LD_LIBRARY_PATH}:${dirbin}
export gencase="${dirbin}/GenCase_linux64"
export dualsphysicscpu="${dirbin}/DualSPHysics5.0CPU_linux64"
export dualsphysicsgpu="${dirbin}/DualSPHysics5.0_linux64"
export boundaryvtk="${dirbin}/BoundaryVTK_linux64"
export partvtk="${dirbin}/PartVTK_linux64"
export partvtkout="${dirbin}/PartVTKOut_linux64"
export measuretool="${dirbin}/MeasureTool_linux64"
export computeforces="${dirbin}/ComputeForces_linux64"
export isosurface="${dirbin}/IsoSurface_linux64"
export flowtool="${dirbin}/FlowTool_linux64"
export floatinginfo="${dirbin}/FloatingInfo_linux64"

#########################################################


# set interparticle spacing based on user-defined resolution
case ${resolution} in
	0400) dx=0.0025;;
	0800) dx=0.00125;; 
	1600) dx=0.000625;;
	3200) dx=0.0003125;;
	6400) dx=0.00015625;;
	*) echo "$0: abort: please specify H/dp 0400,0800,1600.3200,6400" && exit ;;
esac

echo "$0: info: H/dp resolution is ${resolution}; dp is ${dx}" 
sleep 2

#########################################################


# preprocessing (and setting the dp)
if [ ${setgencase} -eq 1 ]; then
	${gencase} ${name}_Def ${dirout}/gencase/${name} -dp:${dx} -save:all -threads:${nthreads}
	if [ $? -ne 0 ] ; then softfail ${gencase}; fi
fi


# sph solver
if [ ${setdsph} -eq 1 ]; then

	# guarantees against overriding: check if old log file exists 
	if [ -e ${dirout}/Run_${label}.out ]; then
		echo ${dualsphysicsgpu} aborted: risk of overriding existing simulation >> ${dirout}/fail.log
		exit 1 
	fi

 	# show and store information on commit of executable (if compiled after commit)
 	cat ${dirbin}/commit.log
 	rsync ${dirbin}/commit.log ${dirout}
 	sleep 2

	# -sv:none means no writing of particle files
	${dualsphysicsgpu} -gpu:${mygpu} ${dirout}/gencase/${name} ${dirout} -dirdataout data -stable -svres #-sv:none
	# guarantees against overriding: make a write-protected copy of the log file as a lock
	cp ${dirout}/Run.out ${dirout}/Run_${label}.out 
	chmod 444 ${dirout}/Run_${label}.out
	if [ $? -ne 0 ] ; then softfail ${dualsphysicsgpu}; fi
fi


# store gauge information in ad hoc directory
if [ ${setgauges} -eq 1 ]; then
	mkdir ${dirout}/gauges && mv ${dirout}/*Gauge* ${dirout}/gauges
fi


# store time-step information in ad hoc directory
if [ ${settimesteps} -eq 1 ]; then
	mkdir ${dirout}/timesteps && mv ${dirout}/*Dt* ${dirout}/timesteps
fi


## particles
if [ ${setparticles} -eq 1 ]; then
	export dirout2=${dirout}/particles

	time ${partvtk} -dirin ${diroutdata} -savevtk ${dirout2}/WaterParticles -onlytype:-all,+fluid -threads:${nthreads}
	if [ $? -ne 0 ] ; then softfail ${partvtk}; fi
fi


# compression gencase information
if [ ${setcgencase} -eq 1 ]; then

	outputfile=${dirout}/cgencase.xz.log
		xz --version > ${outputfile}

	for i in $(ls -1 ${dirout}/gencase/*); do
		echo $i >> ${outputfile}
		{ time xz -k -T${nthreads} -6 -C crc64 $i ;} |& tee -a ${outputfile}
		echo "--" >> ${outputfile}
	done

	mkdir ${dirout}/cgencase
	mv ${dirout}/gencase/*xz ${dirout}/cgencase
	chmod 444 ${dirout}/cgencase/*
fi


# compression gauge information
if [ ${setcgauges} -eq 1 ]; then

	outputfile=${dirout}/cgauges.xz.log
		xz --version > ${outputfile}

	for i in $(ls -1 ${dirout}/gauges/*); do
		echo $i >> ${outputfile}
		{ time xz -k -T${nthreads} -6 -C crc64 $i ;} |& tee -a ${outputfile}
		echo "--" >> ${outputfile}
	done

	mkdir ${dirout}/cgauges
	mv ${dirout}/gauges/*xz ${dirout}/cgauges
	chmod 444 ${dirout}/cgauges/*
fi


# compression time-step information
if [ ${setctimesteps} -eq 1 ]; then

	outputfile=${dirout}/ctimesteps.xz.log
		xz --version > ${outputfile}

	for i in $(ls -1 ${dirout}/timesteps/*); do
		echo $i >> ${outputfile}
		{ time xz -k -T${nthreads} -6 -C crc64 $i ;} |& tee -a ${outputfile}
		echo "--" >> ${outputfile}
	done

	mkdir ${dirout}/ctimesteps
	mv ${dirout}/timesteps/*xz ${dirout}/ctimesteps
	chmod 444 ${dirout}/ctimesteps/*
fi


# compression binaries
if [ ${setcdata} -eq 1 ]; then

	outputfile=${dirout}/cdata.xz.log
	xz --version > ${outputfile}

	for i in $(ls -1 ${diroutdata}/*); do
		echo $i >> ${outputfile}
		{ time xz -k -T${nthreads} -6 -C crc64 $i ;} |& tee -a ${outputfile}
		echo "--" >> ${outputfile}
	done

	mkdir ${dirout}/cdata 
	mv ${dirout}/data/*xz ${dirout}/cdata
	chmod 444 ${dirout}/cdata/*
fi


# compression vtks with particles
if [ ${setcparticles} -eq 1 ]; then

	outputfile=${dirout}/cparticles.xz.log
	xz --version > ${outputfile}

	for i in $(ls -1 ${dirout}/particles/*); do
		echo $i >> ${outputfile}
		{ time xz -k -T${nthreads} -6 -C crc64 $i ;} |& tee -a ${outputfile}
		echo "--" >> ${outputfile}
	done

	mkdir ${dirout}/cparticles 
	mv ${dirout}/particles/*xz ${dirout}/cparticles
	chmod 444 ${dirout}/cparticles/*
fi

echo $0: done $(date)
