# SMC supercoiling simulation
import numpy as np
import json, marshal, random
from scipy import stats


class SSS():
    def __init__(self, keep_seed=False, config_path='config.json') -> None:
        self.config_path = config_path
        self.load_config()
        self.compute_Ceff()

        # set seed
        if not keep_seed:
            np.random.seed(self.PARAMS["system"]["random_seed"])
            random.seed(self.PARAMS["system"]["random_seed"])
        self.L = self.PARAMS["system"]["L"]

        # initialize variables        
        self.numSegments = 1
        self.SMCdirection = [float(np.nan)] # 0 is left, 1 is right. The plasmid doesnt have a direction lifetime
        self.direction_lifetime = [float(np.nan)] # the plasmid doesnt have a direction lifetime
        self.SMC_lifetime = [float(np.nan)] # the plasmid doesnt have a lifetime
        self.SMC_steps = [0] # the steps each SMC has attempted
        self.SMC_steps_real = [0] # the steps each SMC has actaully done
        self.allowed_SMC_steps = [self.PARAMS["SMC"]["numSMCSteps"]] # the steps each SMC is allowed to do
        self.segment_borders = [[0, self.L]] # flexible, segments appear and merge
        self.segment_length = [self.L] # flexible, segments appear and merge
        self.segment_twist = [0] # flexible, segments appear and merge
        self.segment_relTwist = [0] # flexible, segments appear and merge
        self.segment_torque = [0] # flexible, segments appear and merge
        self.Zsegment_borders = [] # flexible, Zsegments appear and merge
        self.Zsegment_length = [] # flexible, Zsegments appear and merge
        self.Zsegment_twist = [] # flexible, Zsegments appear and merge
        self.Zsegment_relTwist = [] # flexible, Zsegments appear and merge
        self.Zsegment_torque = [] # flexible, Zsegments appear and merge
        self.Zsegment_parents = [] # flexible, Zsegments appear and merge
        self.topoActions = [0] # cumulative per time step
        self.accumulatedLk = [0] # cumulative per time step

    def load_config(self):
        with open(self.config_path) as f:
            self.PARAMS = json.load(f)

    def my_deecopy(self, var, dtype):
        return marshal.loads(marshal.dumps(var))
    
    def change_type(self, sub, dtype):
        sub_type = type(sub)
        if sub_type==list:
            return [self.change_type(ele, dtype) for ele in sub]
        else:
            if dtype=='int': return int(sub)
            if dtype=='float': return float(sub)

    def argmin(self, seq):
        return min(range(len(seq)), key=seq.__getitem__)

    def argmax(self, seq):
        return max(range(len(seq)), key=seq.__getitem__)
    
    def where(self, var):
            return [n for n,v in enumerate(var) if v]
    
    def compute_Ceff(self):
        if self.PARAMS["system"]["force"] == 0:
            self.Ceff = 100 # 1/ ( 1/100 ) from Lipfert, J., Kerssemakers, J. W. J., Jager, T., & Dekker, N. H. (2010). Magnetic torque tweezers: measuring torsional stiffness in DNA and RecA-DNA filaments. Nature Methods, 7(12), 977–980. doi:10.1038/nmeth.1520 
        else:
            self.Ceff = 1/ ( 1/100 + \
                1/(4*self.PARAMS["system"]["persistence_length"]) *\
                    np.sqrt(self.PARAMS["system"]["kbT"]/self.PARAMS["system"]["force"]\
                            *self.PARAMS["system"]["persistence_length"]) )
        
    def compute_torque(self, Tw, L):
        Ls = L * self.PARAMS["system"]["nm_per_bp"] # contour length of that piece of DNA
        if Ls == 0: return np.inf
        torque = 2 * np.pi * self.PARAMS["system"]["kbT"] * self.Ceff * Tw / Ls # e.g. Lipfert 2010 Nat Meth       
        return float(torque)

    def compute_Lkchange_topo(self, torque, maxLk=np.inf):
        torque_sign = np.sign(torque)
        if isinstance(torque, list):
            torque_sign[torque==0] = 1
        else:
            if torque == 0: torque_sign = 1
        DLk = torque_sign * self.PARAMS["topo"]["LK0"] * \
            np.exp(self.PARAMS["topo"]["deltaTheta"]*np.abs(torque)/self.PARAMS["system"]["kbT"])
        DLk_sign = np.sign(DLk)
        if isinstance(DLk_sign, list):
            DLk_sign[DLk_sign==0] = 1
            ind = np.abs(DLk)>np.abs(maxLk)
            DLk[ind] = np.array(maxLk)[ind]
            DLk = DLk.astype(int)
        else:
            if DLk_sign == 0: DLk_sign = 1
            if np.abs(DLk) > np.abs(maxLk):
                DLk = maxLk
            DLk = int(DLk)
        
        return DLk
    
    def sign(self, var):
        var_sign = np.sign(var)
        if isinstance(var, list):
            var_sign[var_sign==0] = 1
        else:
            if var_sign == 0: var_sign = 1
        return var_sign
    
    def MC_criterion(self, threshold):        
        return random.random() < threshold
    
    def generateNumber(self, params, mode, dtype=int):
        if mode == "deterministic":
            if isinstance(dtype, int):
                return int(params)
            else:
                return params
        elif mode == "exponential":
            return stats.expon(scale=params).rvs()
        else:
            raise Exception("Mode " + mode + " not implemented.")
    
    def find_nth_index(self, lst, condition, n):
        count = 0
        for index, item in enumerate(lst):
            if condition(item):
                count += 1
                if count == n:
                    return index
        return -1  # Return -1 if n'th element is not found

    def get_segmentID(self, pos, includeZ=True, return_all=False, \
                      evaluate_on_previous_step=False, checkZonly=False, \
                        onTmp=False):
        if onTmp and evaluate_on_previous_step:
            raise Exception('choose either tmp or previous step')
        
        # at the moment used in the SMC spwaning function, topo action function and 
        # evaluate_parents_of_moved_SMC_leg function to get all segments in which 
        # a SMC leg (a single position) could be in
        if evaluate_on_previous_step:
            Zsegment_borders = self.Zsegment_borders_previous
            segment_borders = self.segment_borders_previous
        elif onTmp:
            Zsegment_borders = self.Zsegment_borders_tmp
            segment_borders = self.segment_borders_tmp
        else:
            Zsegment_borders = self.Zsegment_borders
            segment_borders = self.segment_borders
        numZSegments = len(Zsegment_borders)

        # check Z loops first as they are smaller
        potential_IDsZ = []
        if includeZ and numZSegments>0:
            inside = []
            border_distance_selZ = []
            for boundaries in Zsegment_borders:
                
                if boundaries[0] < boundaries[1]: # segment doesnt cross the 0/L border
                    inside.append(pos>=boundaries[0] and pos<=boundaries[1])
                    border_distance_selZ.append(boundaries[1]-boundaries[0])
                elif boundaries[0] > boundaries[1]: # segment crosses the 0/L border
                    inside.append(pos>=boundaries[0] or pos<=boundaries[1])
                    dist = self.L-(boundaries[0]-boundaries[1])
                    border_distance_selZ.append(dist)
                else:
                    inside.append(False)
                    border_distance_selZ.append(0)

            if any(inside):
                potential_IDs = self.where(inside)
                if return_all: 
                    potential_IDsZ = potential_IDs
                else:
                    # return the one with the smallest distance between borders
                    receiver_found = False
                    borders = [Zsegment_borders[i] for i in potential_IDs]
                    table = self.determine_segment_segment_relationship(borders)
                    if 1 in table:
                        numEncompasses = [row.count(1) for row in table.tolist()]
                        maxEncompasses = max(numEncompasses)
                        if numEncompasses.count(maxEncompasses) == 1:
                            pick = numEncompasses.index(maxEncompasses)
                            receiver_found = True # will prevent entering looking for the smallest distance
                        if not receiver_found:
                            borders = [b for count,b in enumerate(borders) if numEncompasses[count]>0]
                            potential_IDs = [b for count,b in enumerate(potential_IDs) if numEncompasses[count]>0]
                    if not receiver_found: # this will also trigger if we have more than 1 complete encompass
                        # take the one with the closest boundary
                        dist_min = self.L
                        for n,border in enumerate(borders):
                            for b in border:
                                dist = self.compute_distance(pos,b)
                                if dist < dist_min:
                                    pick = n
                                    dist_min = dist
                                dist = self.compute_distance(b,pos)
                                if dist < dist_min:
                                    pick = n
                                    dist_min = dist
                    return potential_IDs[pick], 1

        if not checkZonly:
            inside = []
            for boundaries in segment_borders[1:]: # exclude the backbone
                if boundaries[0] < boundaries[1]: # segment doesnt cross the 0/L border
                    inside.append(pos>=boundaries[0] and pos<=boundaries[1])
                elif boundaries[0] > boundaries[1]: # segment crosses the 0/L border
                    inside.append(pos>=boundaries[0] or pos<=boundaries[1])
                    dist = self.L-(boundaries[0]-boundaries[1])
                else:
                    inside.append(False)
                    

            if not any(inside): 
                if return_all: return [0]+potential_IDsZ, [0]+[1]*len(potential_IDsZ)
                else: return 0, 0 # then it falls only into the backbone
        # the SMC might land in several segments becasue they are parents and children of each other
        # take the smallest segment because this is the child
        potential_IDs = self.where(inside)
        

        if return_all: 
            if checkZonly:
                return potential_IDsZ, [1]*len(potential_IDsZ)
            else:                
                return [x+1 for x in potential_IDs]+potential_IDsZ, [0]*len(potential_IDs)+[1]*len(potential_IDsZ)
        
        receiver_found = False
        borders = [segment_borders[i+1] for i in potential_IDs] # +1 because we removed the backbone
        table = self.determine_segment_segment_relationship(borders)
        if 1 in table:            
            numEncompasses = [row.count(1) for row in table.tolist()]
            maxEncompasses = max(numEncompasses)
            if numEncompasses.count(maxEncompasses) == 1:
                pick = numEncompasses.index(maxEncompasses)
                receiver_found = True # will prevent entering looking for the smallest distance
            if not receiver_found:
                borders = [b for count,b in enumerate(borders) if numEncompasses[count]>0]
                potential_IDs = [b for count,b in enumerate(potential_IDs) if numEncompasses[count]>0]
        if not receiver_found: # this will also trigger if we have more than 1 complete encompass
            # take the one with the closest boundary
            dist_min = self.L
            for n,border in enumerate(borders):
                for b in border:
                    dist = self.compute_distance(pos,b)
                    if dist < dist_min:
                        pick = n
                        dist_min = dist
                    dist = self.compute_distance(b,pos)
                    if dist < dist_min:
                        pick = n
                        dist_min = dist

        return potential_IDs[pick]+1, 0 # +1 because we removed the backbone

    def recompute(self, ID, isZ=False, onTmp=False):
        if onTmp:
            if isZ:
                self.Zsegment_relTwist_tmp[ID] = float( self.Zsegment_twist_tmp[ID] / self.Zsegment_length_tmp[ID] )
                self.Zsegment_torque_tmp[ID]   = self.compute_torque(self.Zsegment_twist_tmp[ID], self.Zsegment_length_tmp[ID])
            else:
                self.segment_relTwist_tmp[ID] = float( self.segment_twist_tmp[ID] / self.segment_length_tmp[ID] )
                self.segment_torque_tmp[ID]   = self.compute_torque(self.segment_twist_tmp[ID], self.segment_length_tmp[ID])
        else:
            if isZ:
                self.Zsegment_relTwist[ID] = float( self.Zsegment_twist[ID] / self.Zsegment_length[ID] )
                self.Zsegment_torque[ID]   = self.compute_torque(self.Zsegment_twist[ID], self.Zsegment_length[ID])

            else:
                self.segment_relTwist[ID] = float( self.segment_twist[ID] / self.segment_length[ID] )
                self.segment_torque[ID]   = self.compute_torque(self.segment_twist[ID], self.segment_length[ID])
    
    def SMCs_dissociate(self):
        while any([lifetime==0 for lifetime in self.SMC_lifetime]):
            self.SMC_dissociates() # this always takes away the first one to go

    def SMC_dissociates(self, ID=None):
        if ID is None:
            ID = self.SMC_lifetime.index(0) # get the next one that disappears


        # find all Z loops which are associated with this segment and remove those
        # the other parents of all of those get their twist and length
        left = [ID == parents[0] for parents in self.Zsegment_parents]
        right = [ID == parents[1] for parents in self.Zsegment_parents]
        associated_Zloops = [l or r for l,r in zip(left,right)]
        for IDZ in [n for n,v in enumerate(associated_Zloops) if v]:
            if self.Zsegment_parents[IDZ] == [ID, ID]: 
                receiver = 0
            else:
                receiver = list( set(self.Zsegment_parents[IDZ]) - set([ID]) )[0]
                if receiver == ID: 
                    receiver = 0
            # give the receiver the length and twist of    
            self.segment_length[receiver] += self.Zsegment_length[IDZ]
            self.segment_twist[receiver]  += self.Zsegment_twist[IDZ]
            # recompute relative twist and torque based on new length and twist of that segment
            self.recompute(receiver)
        for IDZ in sorted([n for n,v in enumerate(associated_Zloops) if v])[::-1]:
            self.remove_Zloop(IDZ)

        # give the content of the loop to be deleted to a segment which encompasses the loop
        # the backbone is excluded here. We take the backbone only if no other segment is 
        # a potential receiver
        all_borders = self.segment_borders[1:] + self.Zsegment_borders
        all_borders_isZ = [0]*(self.numSegments-1) + [1]*len(self.Zsegment_borders)
        relation = self.determine_interval_cases_wrap(self.segment_borders[ID], all_borders, self.L)
        relation[ID-1] = 0

        if any([r==1 for r in relation]):
            potential_receivers = [n for n,v in enumerate([r==1 for r in relation]) if v]
            if len(potential_receivers) == 1:
                receiver = potential_receivers[0]
                receiver_isZ = all_borders_isZ[receiver]
                receiver %= (self.numSegments-1)
                if not receiver_isZ: receiver += 1
            else:
                receivers = [r for r in potential_receivers if r < self.numSegments-1]
                receiversZ = [r-(self.numSegments-1) for r in potential_receivers if r >= self.numSegments-1] #ok
                if len(receiversZ) == 1:
                    receiver = receiversZ[0]
                    receiver_isZ = 1
                elif len(receiversZ) > 1:
                    # generate a segment-segment relationship and look for the innermost nested segment
                    receiver_found = False
                    borders = [b for n,b in enumerate(self.Zsegment_borders) if n in receiversZ]
                    table = self.determine_segment_segment_relationship(borders)
                    if 1 in table:
                        numEncompasses = [row.count(1) for row in table.tolist()]
                        maxEncompasses = max(numEncompasses)
                        if numEncompasses.count(maxEncompasses) == 1:
                            receiver = receiversZ[numEncompasses.index(maxEncompasses)]
                            receiver_isZ = 1
                            receiver_found = True # will prevent entering looking for the smallest distance
                        if not receiver_found:
                            borders = [b for count,b in enumerate(borders) if numEncompasses[count]>0]
                            receiversZ = [r for count,r in enumerate(receiversZ) if numEncompasses[count]>0]
                    if not receiver_found: # this will also trigger if we have more than 1 complete encompass
                        # take the one with the closest boundary
                        dist_min = self.L
                        for n,border in enumerate(borders):
                            for b in border:
                                for b_ID in self.segment_borders[ID]:
                                    dist = self.compute_distance(b_ID,b)
                                    if dist < dist_min:
                                        receiver = receiversZ[n]
                                        dist_min = dist
                                    dist = self.compute_distance(b,b_ID)
                                    if dist < dist_min:
                                        receiver = receiversZ[n]
                                        dist_min = dist
                        receiver_isZ = 1
                else:
                    if len(receivers) == 1:
                        receiver = receivers[0]+1
                        receiver_isZ = 0
                    elif len(receivers) > 1:
                        # generate a segment-segment relationship and look for the innermost nested segment
                        receiver_found = False
                        borders = [b for n,b in enumerate(self.segment_borders) if n-1 in receivers]
                        table = self.determine_segment_segment_relationship(borders)
                        if 1 in table:
                            numEncompasses = [row.count(1) for row in table.tolist()]
                            maxEncompasses = max(numEncompasses)
                            if numEncompasses.count(maxEncompasses) == 1:
                                receiver = receivers[numEncompasses.index(maxEncompasses)] + 1
                                receiver_isZ = 0
                                receiver_found = True # will prevent entering looking for the smallest distance
                            if not receiver_found:
                                borders = [b for count,b in enumerate(borders) if numEncompasses[count]>0]
                                receivers = [r for count,r in enumerate(receivers) if numEncompasses[count]>0]
                        
                        if not receiver_found: # this will also trigger if we have more than 1 complete encompass
                            # take the one with the closest boundary
                            dist_min = self.L
                            for n,border in enumerate(borders):
                                for b in border:
                                    for b_ID in self.segment_borders[ID]:
                                        dist = self.compute_distance(b_ID,b)
                                        if dist < dist_min:
                                            receiver = receivers[n]
                                            dist_min = dist
                                        dist = self.compute_distance(b,b_ID)
                                        if dist < dist_min:
                                            receiver = receivers[n]
                                            dist_min = dist

                            receiver += 1
                            receiver_isZ = 0


        else:
            receiver = 0
            receiver_isZ = 0

        # check receiver
        if receiver_isZ:
            self.Zsegment_length[receiver] += self.segment_length[ID]
            self.Zsegment_twist[receiver]  += self.segment_twist[ID]
        else:
            self.segment_length[receiver] += self.segment_length[ID]
            self.segment_twist[receiver]  += self.segment_twist[ID]
        self.recompute(receiver, isZ=receiver_isZ)

        # delete all entries regarding this SMC (which now disappears)
        self.numSegments -= 1
        del self.SMCdirection[ID]
        del self.direction_lifetime[ID]
        del self.SMC_lifetime[ID]
        del self.SMC_steps[ID]
        del self.segment_borders[ID]
        del self.allowed_SMC_steps[ID]
        del self.segment_length[ID]
        del self.segment_twist[ID]
        del self.segment_relTwist[ID]
        del self.segment_torque[ID]

        # update the list of Zloop parents. All entries larger than the deleted 
        # one will be decreased by 1
        for IDZ in range(len(self.Zsegment_parents)):
            for p in range(2):
                if self.Zsegment_parents[IDZ][p] > ID:
                    self.Zsegment_parents[IDZ][p] -= 1
                
    def SMC_spawns(self):    
        if self.step > 0 or self.PARAMS["SMC"]["k_on"] < 1:
            if not self.MC_criterion(self.PARAMS["SMC"]["k_on"]): return # only take action if the MC criterion says we really spawn an SMC        

        SMC_positions = self.keep_first_unique_and_set_to_nan([item for sublist in self.segment_borders[1:] for item in sublist])
        spawning_location_found = False
        spawning_trial_counter = 0
        while not spawning_location_found:
            if spawning_trial_counter > 300: return
            # check where it is supposed to spawn
            spawning_location = random.randint(0, self.L)
            # make sure we dont land on an SMC
            spawning_trial_counter += 1
            if any([(spawning_location+2)==p or (spawning_location+1)==p or (spawning_location)==p for p in SMC_positions]): continue
            # get sign of twist wherever we spawned
            parentID, isZ  = self.get_segmentID(spawning_location, includeZ=True)
            if isZ:
                twist_sign = self.sign(self.Zsegment_twist[parentID])
            else:
                twist_sign = self.sign(self.segment_twist[parentID])
            if twist_sign > 0:
                spawning_location_found = self.MC_criterion(self.PARAMS["SMC"]["pos_over_negative_loading_bias"])
            else:
                spawning_location_found = self.MC_criterion(1-self.PARAMS["SMC"]["pos_over_negative_loading_bias"])

        ID = self.numSegments
      
        if isZ:
            if self.Zsegment_length[parentID] < 4: return
            relTwist = self.Zsegment_relTwist[parentID]
            new_twist = relTwist * 3 # for now we are 3 bp long

            # update the Z loop
            self.Zsegment_length[parentID] -= 3
            self.Zsegment_twist[parentID] -= new_twist

        else: # if we spawn in a regular segment
            if self.segment_length[parentID] < 4: return
            relTwist = self.segment_relTwist[parentID]
            new_twist = relTwist * 3 # for now we are 3 bp long
            self.segment_length[parentID] -= 3
            self.segment_twist[parentID] -= new_twist
        self.recompute(parentID, isZ=isZ)
            
        # we found where the new SMC will spawn. Determine in which loop each leg of the SMC is        
        # make a new segment
        self.segment_borders.append( [spawning_location, spawning_location+2] )
        self.segment_length.append( 3 )
        self.segment_twist.append( new_twist )
        self.allowed_SMC_steps.append( self.PARAMS["SMC"]["numSMCSteps"] )
        self.segment_relTwist.append( 0 ) # will be recomputed
        self.segment_torque.append( 0 ) # will be recomputed
        self.numSegments += 1        
        self.recompute(ID)
                

        # choose direction at random and assign a "direction lifetime"
        self.SMCdirection.append( random.randint(0, 1) )
        direction_lifetime = self.generateNumber(1/self.PARAMS["SMC"]["k_direction_switch"], self.PARAMS["SMC"]["k_direction_switch_mode"])
        self.direction_lifetime.append( direction_lifetime )

        # count number of steps of the new SMC
        self.SMC_steps.append( 0 )

        # get a time when the SMC dissociates
        SMC_lifetime = self.generateNumber(1/self.PARAMS["SMC"]["k_off"], self.PARAMS["SMC"]["k_off_mode"])
        self.SMC_lifetime.append( SMC_lifetime )

    def assign_directions(self):
        for ID in range(1, self.numSegments):
            if self.direction_lifetime[ID] == 0:
                current_direction = self.SMCdirection[ID]
                other_direction = int(list(set([0,1]) - set([current_direction]))[0])
                self.SMCdirection[ID] = other_direction
                direction_lifetime = self.generateNumber(1/self.PARAMS["SMC"]["k_direction_switch"], self.PARAMS["SMC"]["k_direction_switch_mode"])
                self.direction_lifetime[ID] = direction_lifetime
    
    def determine_interval_cases_wrap(self, a, b, L):
        # 0 outside, 1 in, 2 partially, 3 encompasses
        modus = []
        for boundary in b:
            modus.append( self.determine_interval_cases(a, boundary, L) )
        return modus
    
    def determine_interval_cases(self, a, b, L):
        # find what a does wrt b. b should be a given interval and we want to know if a is inside, etc
        # 0: a not in b
        # 1: a fully in b
        # 2: a partially (one point) in b
        # 3: b fully in a
        a_full = False
        b_full = False
        if np.abs(a[1]-a[0]) == L: a_full = True
        if np.abs(b[1]-b[0]) == L: b_full = True
        if a_full and b_full: return 1
        if not a_full and b_full: return 1
        if a_full and not b_full: return 3
        a_wraps = False
        b_wraps = False
        if b[0]==b[1]: return None
        if a[1] < a[0]: a_wraps = True
        if b[1] < b[0]: b_wraps = True
        if not a_wraps and not b_wraps:
            if (a[0] < b[0] and a[1] < b[0]) \
                or (a[0] > b[1] and a[1] > b[1]): return 0 # out
            elif a[0] >= b[0] and a[1] <= b[1]: return 1 # in
            elif a[0] < b[0] and a[1] > b[1]: return 3 # a encompasses
            else: return 2 # partially
        elif a_wraps and not b_wraps:
            if a[0] > b[1] and a[1] < b[0]: return 0 # out
            elif (a[0]>b[1] and a[1]>b[1]) or (a[0]<b[0] and a[1]<b[0]): return 3 # a encompasses
            else: return 2 # partially
        elif a_wraps and b_wraps:        
            if a[0]>=b[0] and a[1]<=b[1]: return 1 # in
            elif a[0]<b[0] and a[1]>b[1]: return 3 # a encompasses
            else: return 2 # partially
        elif not a_wraps and b_wraps:
            if a[0]<b[0] and a[1]<b[0] \
            and a[0]>b[1] and a[1]>b[1]: return 0 # out
            elif (a[0]>=b[0] and a[1]>b[0]) \
            or (a[0]<b[1] and a[1]<=b[1]): return 1 # in
            else: return 2 # partially
        return 0 # if nothing else triggered, 

    def check_torque(self, Tw, L, checkL=False, ID=None):
        # if checkL = True, we also make sure that no segment length grow beyond the plasmid size
        if checkL: 
            if L > self.L: return True
            if L <= 0:
                return True
        torque  = self.compute_torque(Tw, L)
        # check whether we forbid the step because of positive or negative twist
        if ID is not None:
            if (torque/self.PARAMS["system"]["kbT"] > self.PARAMS["SMC"]["stalling_torque"]) and \
                (torque/self.PARAMS["system"]["kbT"] < -self.PARAMS["SMC"]["stalling_torque"]):
                self.step_forbidden[ID] = 2
                return True
            elif torque/self.PARAMS["system"]["kbT"] > self.PARAMS["SMC"]["stalling_torque"]:
                self.step_forbidden[ID] = 1
                return True
            elif torque/self.PARAMS["system"]["kbT"] < -self.PARAMS["SMC"]["stalling_torque"]:
                self.step_forbidden[ID] = -1
                return True
            else:
                return False
        else:
            return np.abs(torque/self.PARAMS["system"]["kbT"]) > self.PARAMS["SMC"]["stalling_torque"]


    def contains_only_specific_values(self, lst, values):
        for item in lst:
            if item not in values:
                return False
        return True
    
    def find_closest_border_behind_leg(self, SMCleg_pos, direction, relevant_borders, \
                                       border_isZ, find_leg_of_single_border=False, preferZ=False):
        
        if find_leg_of_single_border and len(relevant_borders)>1:
            raise Exception('Cannot search for specific leg if more than one border is specified.')
        
        # if preferZ is True, we only take Z loops into account in a first round. In case there are no hits, we take all segments into account
        L = self.L
        distances = []
        for count,rb in enumerate(relevant_borders):
            if np.isnan(rb[0]) or (preferZ and not border_isZ[count] and any([z if ~np.isnan(z) else False for z in border_isZ])): 
                distances.append(np.inf)
                continue
            b_wraps = rb[1] < rb[0] # find if the relevant borders wrap or not
            if b_wraps:
                if direction == 0:                    
                    if SMCleg_pos > max(rb):
                        distances.append( L - (SMCleg_pos-min(rb)) )
                        if find_leg_of_single_border: return self.argmin(rb)
                    elif SMCleg_pos < min(rb):
                        distances.append( min(rb)-SMCleg_pos )
                        if find_leg_of_single_border: return self.argmin(rb)
                    else:
                        distances.append( max(rb)-SMCleg_pos )
                        if find_leg_of_single_border: return self.argmax(rb)
                else:
                    
                    if SMCleg_pos > max(rb):
                        distances.append( SMCleg_pos - max(rb) )
                        if find_leg_of_single_border: return self.argmax(rb)
                    elif SMCleg_pos < min(rb):
                        distances.append( self.L - (max(rb)-SMCleg_pos) )
                        if find_leg_of_single_border: return self.argmax(rb)
                    else:
                        distances.append( SMCleg_pos-min(rb) )
                        if find_leg_of_single_border: return self.argmin(rb)
            
            else:                
                if direction == 0:
                    
                    if SMCleg_pos > max(rb):
                        distances.append( L - (SMCleg_pos - min(rb)) )
                        if find_leg_of_single_border: return self.argmin(rb)
                    elif SMCleg_pos < min(rb):
                        distances.append( min(rb) - SMCleg_pos )
                        if find_leg_of_single_border: return self.argmin(rb)
                    else:
                        distances.append( max(rb) - SMCleg_pos )
                        if find_leg_of_single_border: return self.argmax(rb)
                else:
                    
                    if SMCleg_pos > max(rb):
                        distances.append( SMCleg_pos - max(rb) )
                        if find_leg_of_single_border: return self.argmax(rb)
                    elif SMCleg_pos < min(rb):
                        distances.append( L - (max(rb)-SMCleg_pos) )
                        if find_leg_of_single_border: return self.argmax(rb)
                    else:
                        distances.append( SMCleg_pos - min(rb) )
                        if find_leg_of_single_border: return self.argmin(rb)


        # compute minimum distance
        Dmin = min(distances)
        if preferZ:
            if np.isnan(Dmin): 
                out = self.find_closest_border_behind_leg(SMCleg_pos, direction, relevant_borders, \
                                       border_isZ, find_leg_of_single_border=find_leg_of_single_border, preferZ=False)
                return out
            
        potential_IDs = self.where([d == Dmin for d in distances])
        potential_IDs_Z = [potential_IDs[n] for n in range(len(potential_IDs)) if border_isZ[potential_IDs[n]]==1]
        if len(potential_IDs_Z) > 0:
            if len(potential_IDs_Z) > 1:
                ID_selection = []
                for p in potential_IDs_Z:
                    Zborder = relevant_borders[p]
                    if any([np.isnan(x) for x in Zborder]):
                        ID_selection.append(np.nan)
                        continue
                    if Zborder not in self.Zsegment_borders:
                        ID_selection.append(np.nan)
                        continue
                    ID_selection.append( self.Zsegment_borders.index(Zborder) )
                # if several Z loops are the same distance away, 
                # take the one with the larger ID (the newer one)
                return potential_IDs_Z[self.argmax(ID_selection)] 
            else:
                return potential_IDs_Z[0] 
        
        elif len(potential_IDs_Z) == 0 and len(potential_IDs)>1:
            if len(potential_IDs) > 1:
                ID_selection = []
                for p in potential_IDs:
                    border = relevant_borders[p]
                    if any([np.isnan(x) for x in border]):
                        ID_selection.append(np.nan)
                        continue
                    if border not in self.segment_borders:
                        ID_selection.append(np.nan)
                        continue
                    ID_selection.append( self.segment_borders.index(border) )
                # if several segments are the same distance away, 
                # take the one with the larger ID (the newer one)
                return potential_IDs[self.argmax(ID_selection)]
        return self.argmin(distances)
    
    def find_replacement_of_removed_Zloop(self, parentID, parent_isZ, direction, border, stepsize):
        # parentID has to be a Zloop
        removed = [parentID==x[0] for x in self.remove_replace_Zloops_lastStep] # format is [removed, replaced] (the latter is a parent so it cannot be a Zloop)
        removed_IDs = [x[0] for x in self.remove_replace_Zloops_lastStep if parentID==x[0]]
        removed_count = removed.count(True)

        if removed_count == 1:
            replacement = self.remove_replace_Zloops_lastStep[removed.index(True)]
            parentID = replacement[1][direction]
            parent_isZ = False
            # check if the leg has a Z loop associated. If so, that Z loop is what we want
            associated_Zloops = [parentID in p for p in self.Zsegment_parents]
            for rID in removed_IDs:
                associated_Zloops[rID] = False
            associated_Zloops_count = associated_Zloops.count(True)
            if associated_Zloops_count == 1:
                associated_Zloop = associated_Zloops.index(True)
                # find if the position of that Z loop which has parentID as parent is crossed by our move
                # If yes, then we are after this associated_Zloop. If not, we keep using parentID (which is not a Zloop)
                parent_is_left_or_right_leg = self.Zsegment_parents[associated_Zloop].index(parentID)
                border_of_interest = self.Zsegment_borders[associated_Zloop][parent_is_left_or_right_leg]
                border_old = border[direction]
                if direction == 0:
                    border_new = border_old - stepsize
                else:
                    border_new = border_old + stepsize
                border_new_complete = self.my_deecopy(border, 'int')
                border_new_complete[direction] = border_new
                border_old_tmp = border_old
                border_new_tmp = border_new
                if np.abs(border_old_tmp-border_new_tmp) > stepsize: # then we happen to wrap
                    if   border_old_tmp > border_new_tmp: border_old_tmp -= self.L
                    elif border_old_tmp < border_new_tmp: border_old_tmp += self.L
                crossed = self.check_for_overstepping_myself(border_of_interest, border, \
                                                              border_new_complete, direction)
                if crossed:
                    parentID = associated_Zloop
                    parent_isZ = True
        return parentID, parent_isZ

    def evaluate_parents_of_moved_SMC_leg(self, border, ID, direction, \
                                          checkZonly=False, evaluate_on_previous_step=False, \
                                            stepsize=None, preferZ=False, excludeZ=None):
        
        # in case we evaluate on the previous step: it might be that we deleted Z loop that
        # still had an entry in the previous step. If that is the case, we artifically change
        # the output of this function to the segment/Zloop which replaced the removed Zloop


        # we look here concomittantly in normal and in Z segments (or only Z if asked for)
        if checkZonly:
            if evaluate_on_previous_step:
                all_borders = self.Zsegment_borders_previous
            else:
                all_borders = self.Zsegment_borders
            if excludeZ is not None: 
                numSegments = 0
                for exclude in excludeZ:
                    if exclude >= len(all_borders): continue
                    all_borders[exclude] = [float(np.nan), float(np.nan)]
        else:
            if evaluate_on_previous_step:
                all_borders = self.segment_borders_previous + self.Zsegment_borders_previous
                numSegments = len(self.segment_borders_previous)
            else:
                all_borders = self.segment_borders + self.Zsegment_borders
                numSegments = len(self.segment_borders)
            if excludeZ is not None:
                for exclude in excludeZ:

                    # all_borders[numSegments] gives the Z loop with exclude=0
                    if exclude+numSegments >= len(all_borders): continue
                    all_borders[exclude+numSegments] = [float(np.nan), float(np.nan)]

        # 0 outside, 1 in, 2 partially, 3 encompasses
        relation = self.determine_interval_cases_wrap(border, all_borders, self.L)

        if not checkZonly: relation[ID] = 0 # mask the segment we are working on atm
        # check inside only (dont care about outside). Take the one whose boudnaries are closest together
        # this one will be sitting on top of all other loops
        if self.contains_only_specific_values(relation, [0,1]):
            border_span = self.compute_span(all_borders)
            for n in range(len(all_borders)):
                if relation[n] == 0:
                    border_span[n] = np.inf
            if excludeZ is not None:
                remove = []
                for n in range(numSegments,len(border_span)):
                    if n-numSegments in excludeZ:
                        remove.append(n)
                for r in remove[::-1]:
                    border_span[r] = np.inf

            if all([x==np.inf for x in border_span]):
                return float(np.nan), float(np.nan)
            
            parentID = self.argmin(border_span)
            if checkZonly:
                parent_isZ = 1
            else:
                parent_isZ = 0
                if parentID >= self.numSegments: # then the parentID is a Zloop
                    parentID -= self.numSegments
                    parent_isZ = 1

        else:
            # ask for all segments in which the new position could be
            potID, potisZ = self.get_segmentID(border[direction], includeZ=True, \
                    return_all=True, evaluate_on_previous_step=evaluate_on_previous_step, \
                        checkZonly=checkZonly)
            if excludeZ is not None:
                remove = []
                for n in range(len(potID)):
                    if potisZ[n]:
                        if potID[n] in excludeZ:
                            remove.append(n)
                for r in remove[::-1]:

                    potID[r] = float(np.nan)
                    potisZ[r] = float(np.nan)
            # exclude ID itself, that is the trivial choice
            if ID in potID:
                remove = potID.index(ID)
                if not potisZ[remove]:
                    del potID[remove]
                    del potisZ[remove]
            if len(potID) == 0:
                    parentID = 0
                    parent_isZ = 0
            elif len(potID) == 1: # if there is only 1 option
                # check if the selected Zloop has been replaced   
                parentID = potID[0]
                parent_isZ = potisZ[0]   
            else:
                # evaluate which segment has the closest border behind the 
                # SMC (depends on direction of travel)
                relevant_borders = []
                for n in range(len(potID)):
                    if np.isnan(potID[n]): 
                        relevant_borders.append( [float(np.nan), float(np.nan)] )
                        continue
                    if potisZ[n]==1:
                        if evaluate_on_previous_step:
                            relevant_borders.append( self.Zsegment_borders_previous[potID[n]] )
                        else:
                            relevant_borders.append( self.Zsegment_borders[potID[n]] )
                    else:
                        if evaluate_on_previous_step:
                            relevant_borders.append( self.segment_borders_previous[potID[n]] )
                        else:
                            relevant_borders.append( self.segment_borders[potID[n]] )

                # find closest border that is _behind_ him. If Z and normal are present
                # at the same position, take Z. W can use the direction flag. it he moves
                # to the right (direction=1) we look left and vice versa
                SMCleg_pos = border[direction]
                closest_border = self.find_closest_border_behind_leg(SMCleg_pos, \
                    direction, relevant_borders, potisZ, preferZ=preferZ)
                
                parentID = potID[closest_border]
                parent_isZ = potisZ[closest_border]   

        # check if the selected Zloop has been replaced
        if parent_isZ and evaluate_on_previous_step:
            parentID, parent_isZ = self.find_replacement_of_removed_Zloop(parentID, \
                parent_isZ, direction, border, stepsize)
        if np.isnan(parentID):
            return float(np.nan), float(np.nan)
        else:
            return int(parentID), bool(parent_isZ)
            

    def determine_segment_segment_relationship(self, borders):
        numBorders = len(borders)
        table = np.zeros((numBorders, numBorders))
        for n,border in enumerate(borders):
            table[n,:] = self.determine_interval_cases_wrap(border, \
                            borders, self.L)
            table[n,n] = 0
        return table

    def keep_first_unique_and_set_to_nan(self, lst):
        unique_values = set()
        result = []

        for item in lst:
            if item not in unique_values:
                unique_values.add(item)
                result.append(item)
            else:
                result.append(float(np.nan))

        return result
    
    def check_if_Zloop_changes(self, border_old, onTmp=False):
        IDZ = [None]
        Z_changed = [None]
        if onTmp:
            condition = len(self.Zsegment_borders_tmp)>0
        else:
            condition = len(self.Zsegment_borders)>0
        if condition:
            # check if we are enlarging any Zloop border
            # If we move a Z loop border, then the segment itself only gets its borders moved
            # it doesnt gain length, because the length goes to the Zloop
            # it also doest gain twist (also goes to Zloop) since there is an SMC separating Zloop and segment
            if onTmp:
                Z_changed = [border_old == Zbord for Zborder in self.Zsegment_borders_tmp for Zbord in Zborder]
            else:
                Z_changed = [border_old == Zbord for Zborder in self.Zsegment_borders for Zbord in Zborder]
            Z_changed = [[Z_changed[i], Z_changed[i+1]] for i in range(0, len(Z_changed), 2)]
            Z_changed_segment = [any(Z) for Z in Z_changed]
            IDZ = self.where(Z_changed_segment)#[0]
            IDZ = [idz for idz in IDZ if idz not in self.remove_Zloops]
            if len(IDZ) == 0:
                IDZ = [None]
        return IDZ

    def check_Zloop_crossings(self):
        # check for Z loop crossings
        plasmid = np.zeros((self.L))
        for b in self.Zsegment_borders_tmp:
            if b[0] < b[1]:
                plasmid[b[0]+1:b[1]] += 1
            elif b[1] < b[0]:
                plasmid[b[0]+1:self.L] += 1
                plasmid[0:b[1]] += 1
            if any(plasmid>1):
                return 1
        return 0
    
    def check_specific_Zloop_crossing(self, specific_border):
        plasmid = np.zeros((self.L))
        if specific_border[0] < specific_border[1]:
            plasmid[specific_border[0]:specific_border[1]+1] += 1
        else:
            plasmid[specific_border[0]:self.L] += 1
            plasmid[0:specific_border[1]+1] += 1

        overlap = []
        for count,b in enumerate(self.Zsegment_borders_tmp):
            if count in self.remove_Zloops: continue
            plasmid_copy = np.array(self.my_deecopy(plasmid.tolist(), None))
            if b[0] < b[1]:
                plasmid_copy[b[0]:b[1]+1] += 1
            elif b[0] > b[1]:
                plasmid_copy[b[0]:self.L] += 1
                plasmid_copy[0:b[1]+1] += 1
            if (plasmid_copy>1).tolist().count(True)>1:
                overlap.append(count)
        return overlap

    
    def make_resolve_Zloop(self, border_old_complete, border_new_complete, ID, \
                             step_into_ID, step_into_isZ, come_from_ID, come_from_isZ):

        # we do not need to make a copy here. the last call to update_step() did that and we asked
        # that this copy is not being transcribed. So we continue working on the copy. Only if everything
        # is successful, we transcribe it at the end of this function

        # keys: 0 outside, 2 partial, 1 inside, 3 encompassing              
        moveID_before = self.determine_interval_cases_wrap(border_old_complete, self.segment_borders_previous, self.L)
        moveID_after  = self.determine_interval_cases_wrap(border_new_complete, self.segment_borders_previous, self.L)

        # remove the backbone for this purpose and the segment that is being enlarged itself
        moveID_before[0] = 0
        moveID_before[ID] = 0
        moveID_after[0] = 0
        moveID_after[ID] = 0

        # look for changes
        changes = [a!=b for a,b in zip(moveID_before, moveID_after)]

        # look for the last crossed SMC
        direction = [a!=b for a,b in zip(border_old_complete, border_new_complete)].index(True)
        other_direction = [a==b for a,b in zip(border_old_complete, border_new_complete)].index(True)

        segment_borders = self.my_deecopy(self.segment_borders_previous, 'int')
        segment_borders[ID] = [float(np.nan), float(np.nan)]
        closest_segment_behind = self.find_closest_border_behind_leg(border_new_complete[direction], direction, \
                                                segment_borders, [0]*self.numSegments)
        closest_SMC_behind_move = self.find_closest_border_behind_leg(border_new_complete[direction], direction, \
                                                [self.segment_borders_previous[closest_segment_behind]], [0], \
                                                find_leg_of_single_border=True)
        crossed_SMCpos_behind_move = self.segment_borders_previous[closest_segment_behind][closest_SMC_behind_move]

        closest_segment_front = self.find_closest_border_behind_leg(border_old_complete[direction], other_direction, \
                                                segment_borders, [0]*self.numSegments)
        closest_SMC_infrontof_move = self.find_closest_border_behind_leg(border_old_complete[direction], other_direction, \
                                                [self.segment_borders_previous[closest_segment_front]], [0], \
                                                find_leg_of_single_border=True)
        crossed_SMCpos_infrontof_move = self.segment_borders_previous[closest_segment_front][closest_SMC_infrontof_move]

        change_boundaries = [n for n,v in enumerate(changes) if v]
        change_boundaries = list(set(change_boundaries)-set(self.change_boundaries_treated))
        
        
        # only do the changes on the first one. For any more borders, we recursively enter the function again, reevaluating 
        # if we really have to take any action after the last round of changes
        for change_boundary in change_boundaries[0:1]:

            before = moveID_before[change_boundary]
            after = moveID_after[change_boundary]
            if [before, after] in self.implemented_changes:
                self.change_boundaries_treated.append(change_boundary)
                step_forbidden = self.make_resolve_Zloop(border_old_complete, border_new_complete, ID, \
                                step_into_ID, step_into_isZ, come_from_ID, come_from_isZ)
                if step_forbidden: return 1
            else:
                self.implemented_changes.append( [before, after] )            

            # outside to partial
            # if we step into a Z loop, we break that Z loop up into 2
            if before==0 and after==2:
                # the new Zloop
                if direction == 0: 
                    new_borders = [border_new_complete[direction], crossed_SMCpos_behind_move]
                    new_parents = [ID, closest_segment_behind]                    
                else: 
                    new_borders = [crossed_SMCpos_behind_move, border_new_complete[direction]]
                    new_parents = [closest_segment_behind, ID]
                length = self.compute_span(new_borders)
                if step_into_isZ:
                    twist = length * self.Zsegment_relTwist_tmp[step_into_ID]
                else:
                    twist = length * self.segment_relTwist_tmp[step_into_ID]                
                if self.check_torque(twist, length, checkL=True, ID=ID): 
                    return 1
                
                # deal with Z loops
                take_from = step_into_ID
                take_from_isZ = step_into_isZ
                stepsize = self.compute_span([border_old_complete[direction], border_new_complete[direction]])
                if stepsize > self.L/2: stepsize = self.L - stepsize
                come_from_IDZ, come_from_IDZ_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                                        ID, direction, stepsize=stepsize, \
                                        checkZonly=True, evaluate_on_previous_step=True, \
                                        excludeZ=sorted(set(self.remove_Zloops))[::-1]) # exclude the Z loops that were removed in the previous execution of update_step()
                if np.isnan(come_from_IDZ): return 1

                if take_from_isZ:
                    if not (take_from == come_from_IDZ and take_from_isZ == come_from_IDZ_isZ):
                        if direction == 0:
                            take_from_parents_tmp = [self.Zsegment_parents_previous[take_from][0], ID]
                            take_from_borders_tmp = [self.Zsegment_borders_previous[take_from][0], self.segment_borders_tmp[ID][0]]
                        else:
                            take_from_parents_tmp = [ID, self.Zsegment_parents_previous[take_from][1]]
                            take_from_borders_tmp = [self.segment_borders_tmp[ID][1], self.Zsegment_borders_previous[take_from][1]]
                        self.Zsegment_parents_tmp[take_from] = take_from_parents_tmp
                        self.Zsegment_borders_tmp[take_from] = take_from_borders_tmp
                    take_from_length_tmp = self.Zsegment_length_tmp[take_from] - length
                    take_from_twist_tmp = self.Zsegment_twist_tmp[take_from] - twist
                    self.Zsegment_length_tmp[take_from] = take_from_length_tmp
                    self.Zsegment_twist_tmp[take_from] = take_from_twist_tmp
                else:
                    take_from_length_tmp = self.segment_length_tmp[take_from] - length
                    take_from_twist_tmp = self.segment_twist_tmp[take_from] - twist
                    
                    self.segment_length_tmp[take_from] = take_from_length_tmp
                    self.segment_twist_tmp[take_from] = take_from_twist_tmp
                
                if self.check_torque(take_from_twist_tmp, take_from_length_tmp, checkL=True, ID=ID): 
                    return 1                

                # check if we came from any Zloops
                if len(self.Zsegment_borders_previous) > 0:                    
                    if not (take_from == come_from_IDZ and take_from_isZ == come_from_IDZ_isZ):                        
                        if ~np.isnan(come_from_IDZ) and not (come_from_IDZ==0 and come_from_IDZ_isZ==0):
                            # switch parent and borders of that Z loop
                            if direction == 0:
                                Zparents = [closest_segment_front, self.Zsegment_parents_tmp[come_from_IDZ][1]]
                                Zborders = [crossed_SMCpos_infrontof_move, self.Zsegment_borders_tmp[come_from_IDZ][1]]
                            else:
                                Zparents = [self.Zsegment_parents_tmp[come_from_IDZ][0], closest_segment_front]
                                Zborders = [self.Zsegment_borders_tmp[come_from_IDZ][0], crossed_SMCpos_infrontof_move]
                            # implement
                            self.Zsegment_parents_tmp[come_from_IDZ] = Zparents
                            self.Zsegment_borders_tmp[come_from_IDZ] = Zborders
                            if Zparents[0] == Zparents[1]: # dissolve the Z loop
                                self.segment_twist_tmp[Zparents[0]] += self.Zsegment_twist_tmp[come_from_IDZ]
                                self.segment_length_tmp[Zparents[0]] += self.Zsegment_length_tmp[come_from_IDZ]
                                self.recompute(Zparents[0], onTmp=True)
                                self.remove_Zloops.append(come_from_IDZ)

                # make a new Z loop with that
                self.make_new_Zloop(new_borders, new_parents, length, twist, onTmp=True)
                self.recompute(take_from, isZ=take_from_isZ, onTmp=True)

                # finally, check if the new Z loop overlaps with an existing one at one of its legs. If so, we take the length and twist from the 
                # new Z loop and subtract it from the overlapping one. 
                # if it's not at a leg, it's an internal additional Z loop which is ok. We leave this one unaltered
                overlap = self.check_specific_Zloop_crossing(new_borders)
                IDZ_last = len(self.Zsegment_borders_tmp)-1
                if IDZ_last in overlap:
                    del overlap[overlap.index(IDZ_last)]
                for ID_OL in overlap:   
                    # check if legs overlap. If not at least one leg overlaps, the 2nd Z loop is entirely internal to the first one and we leave it alone
                    legs_overlap = [b in self.Zsegment_borders_tmp[ID_OL] for b in self.Zsegment_borders_tmp[IDZ_last]]
                    if all(legs_overlap): # dissolve ID_OL
                        self.Zsegment_twist_tmp[IDZ_last] += self.Zsegment_twist_tmp[ID_OL] # give it to the newest Z loop
                        self.Zsegment_length_tmp[IDZ_last] += self.Zsegment_length_tmp[ID_OL] # give it to the newest Z loop
                        self.remove_Zloops.append(ID_OL)
                        self.recompute(IDZ_last, isZ=True, onTmp=True)
                        # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                        if step_into_ID in self.remove_Zloops and step_into_isZ:
                            step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                                ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
                            if np.isnan(step_into_ID): return 1
                        if come_from_ID in self.remove_Zloops and come_from_isZ:
                            come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                                ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
                            if np.isnan(come_from_ID): return 1
                        continue
                    if not any(legs_overlap): continue # do nothing
                    if self.Zsegment_length_tmp[ID_OL] > 1:
                        self.Zsegment_twist_tmp[ID_OL] -= self.Zsegment_twist_tmp[IDZ_last]
                        self.Zsegment_length_tmp[ID_OL] -= self.Zsegment_length_tmp[IDZ_last]
                        self.Zsegment_twist_tmp[IDZ_last] += self.Zsegment_twist_tmp[IDZ_last]
                        self.Zsegment_length_tmp[IDZ_last] += self.Zsegment_length_tmp[IDZ_last]
                        if self.check_torque(self.Zsegment_twist_tmp[ID_OL], self.Zsegment_length_tmp[ID_OL], checkL=True, ID=ID): 
                            return 1
                        
                    
                    # update its parents and borders
                    overlap_side = int(self.Zsegment_borders_tmp[ID_OL][1] == self.Zsegment_borders_tmp[IDZ_last][1])
                    other_side = int(not bool(overlap_side))
                    self.Zsegment_borders_tmp[ID_OL][overlap_side] = self.Zsegment_borders_tmp[IDZ_last][other_side]
                    self.Zsegment_parents_tmp[ID_OL][overlap_side] = self.Zsegment_parents_tmp[IDZ_last][other_side]
                    self.recompute(ID_OL, isZ=True, onTmp=True)
                    
                    break # if we delete IDZ_last, we do not check for any overlaps anymore

                # check if we have Zsgements left over whose parents moved away so their border is now not associated anymore
                # with any segment. We remove those and give their content to their remaining parent
                for IDZ2 in range(len(self.Zsegment_borders_tmp)):
                    for parent in self.Zsegment_parents_tmp[IDZ2]:
                        if self.segment_borders_tmp[parent][0] not in self.Zsegment_borders_tmp[IDZ2] \
                        and self.segment_borders_tmp[parent][1] not in self.Zsegment_borders_tmp[IDZ2] \
                        and IDZ2 not in self.remove_Zloops:
                            if self.Zsegment_parents_tmp[IDZ2] == [parent, parent]:
                                target_segment = 0
                            else:
                                target_segment = list( set(self.Zsegment_parents_tmp[IDZ2]) - set([parent]) )[0]
                            self.segment_length_tmp[target_segment] += self.Zsegment_length_tmp[IDZ2]
                            self.segment_twist_tmp[target_segment]  += self.Zsegment_twist_tmp[IDZ2]
                            self.remove_Zloops.append(IDZ2) 
                            # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                            if step_into_ID in self.remove_Zloops and step_into_isZ:
                                step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                        excludeZ=sorted(set(self.remove_Zloops))[::-1])
                                if np.isnan(step_into_ID): return 1
                            if come_from_ID in self.remove_Zloops and come_from_isZ:
                                come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                        excludeZ=sorted(set(self.remove_Zloops))[::-1])
                                if np.isnan(come_from_ID): return 1


            # inside to partial
            # we make a new Z loop which remains inside the segment we are exiting
            # If we step into a Zloop, we also make a new Z loop outside the segment we are exiting, between
            # the last SMCposition and the new segment border (the one we are moving)
            # and we move the existing one past the moved segment border
            elif before==1 and after==2: 
                if direction == 0: 
                    new_borders = [crossed_SMCpos_infrontof_move, border_new_complete[other_direction]]
                    new_parents = [closest_segment_front, ID]
                else: 
                    new_borders = [border_new_complete[other_direction], crossed_SMCpos_infrontof_move]
                    new_parents = [ID, closest_segment_front]
                # check if any intervening Zloop borders are in the new borders. If so, make the new Z loop only
                # between this border and crossed_SMCpos_infrontof_move
                segment_behind = self.find_closest_border_behind_leg(new_borders[direction], direction, self.segment_borders_tmp, \
                                       [0]*self.numSegments, find_leg_of_single_border=False, preferZ=False)
                if segment_behind != ID:
                    SMC_behind = self.find_closest_border_behind_leg(new_borders[direction], direction, \
                                                    [self.segment_borders_previous[segment_behind]], [0], \
                                                    find_leg_of_single_border=True)
                    crossed_SMC_behind = self.segment_borders_previous[segment_behind][SMC_behind]
                    new_borders[other_direction] = crossed_SMC_behind
                    new_parents[other_direction] = segment_behind

                length = self.compute_span(new_borders)
                if come_from_isZ:
                    twist = length * self.Zsegment_relTwist_tmp[come_from_ID]
                else:
                    twist = length * self.segment_relTwist_tmp[come_from_ID]
                make_new_Zloop = True
                if new_borders[0]==new_borders[1] or new_parents[0]==new_parents[1]:
                    twist = 0
                    length = 0
                    make_new_Zloop = False
                else:
                    if self.check_torque(twist, length, checkL=True, ID=ID): 
                        return 1

                # what is taken into the Z loop has to be removed from the segment we are coming from
                if come_from_isZ:
                    self.Zsegment_twist_tmp[come_from_ID] -= twist
                    self.Zsegment_length_tmp[come_from_ID] -= length
                    if self.check_torque(self.Zsegment_twist_tmp[come_from_ID], self.Zsegment_length_tmp[come_from_ID], checkL=True, ID=ID): 
                        return 1
                    # if we come from a Zloop, then the one we are coming from changes borders and parents such that the new one
                    # doesnt overlap with the one we are coming from
                    segment_behind = self.find_closest_border_behind_leg(border_new_complete[direction], direction, self.segment_borders_tmp, \
                                       [0]*self.numSegments, find_leg_of_single_border=False, preferZ=False)
                    if segment_behind != ID:
                        SMC_behind = self.find_closest_border_behind_leg(border_new_complete[direction], direction, \
                                                    [self.segment_borders_tmp[segment_behind]], [0], \
                                                    find_leg_of_single_border=True)
                        # then we have likely a row of segment borders
                        if segment_behind == self.Zsegment_parents_tmp[come_from_ID][other_direction]:
                            segment_borders_tmp = self.my_deecopy(self.segment_borders_tmp, 'int')
                            segment_borders_tmp[segment_behind] = [float(np.nan), float(np.nan)]
                            segment_behind = self.find_closest_border_behind_leg(border_new_complete[direction], direction, segment_borders_tmp, \
                                       [0]*self.numSegments, find_leg_of_single_border=False, preferZ=False)
                            SMC_behind = self.find_closest_border_behind_leg(border_new_complete[direction], direction, \
                                                    [self.segment_borders_tmp[segment_behind]], [0], \
                                                    find_leg_of_single_border=True)
                        self.Zsegment_parents_tmp[come_from_ID][direction] = segment_behind
                        self.Zsegment_borders_tmp[come_from_ID][direction] = self.segment_borders_tmp[segment_behind][SMC_behind]
                    else:                 
                        self.Zsegment_parents_tmp[come_from_ID][direction] = ID
                        self.Zsegment_borders_tmp[come_from_ID][direction] = self.segment_borders_tmp[ID][direction]
                else:
                    self.segment_twist_tmp[come_from_ID] -= twist
                    self.segment_length_tmp[come_from_ID] -= length
                    if self.check_torque(self.segment_twist_tmp[come_from_ID], self.segment_length_tmp[come_from_ID], checkL=True, ID=ID): 
                        return 1
                
                take_from = step_into_ID
                take_from_isZ = step_into_isZ
                if take_from_isZ:
                    
                    # make a new Z loop between the crossing point and the segment border
                    if direction == 0:
                        middle_parents_tmp = [ID, closest_segment_behind]
                        middle_borders_tmp = [border_new_complete[0], crossed_SMCpos_behind_move]
                    else:
                        middle_parents_tmp = [closest_segment_behind, ID]
                        middle_borders_tmp = [crossed_SMCpos_behind_move, border_new_complete[1]]
                    middle_length_tmp = self.compute_span(middle_borders_tmp)
                    middle_twist_tmp = middle_length_tmp * self.Zsegment_relTwist_tmp[take_from]
                    if self.check_torque(middle_twist_tmp, middle_length_tmp, checkL=True, ID=ID): 
                        return 1
                    
                    # shrink the one we stepped into
                    if direction == 0:
                        take_from_parents_tmp = [self.Zsegment_parents_tmp[take_from][0], ID]
                        take_from_borders_tmp = [self.Zsegment_borders_tmp[take_from][0], self.segment_borders_tmp[ID][0]]
                    else:
                        take_from_parents_tmp = [ID, self.Zsegment_parents_tmp[take_from][1]]
                        take_from_borders_tmp = [self.segment_borders_tmp[ID][1], self.Zsegment_borders_tmp[take_from][1]]
                    
                    twist_take_from = middle_length_tmp * self.Zsegment_relTwist_tmp[take_from]
                    take_from_length_tmp = self.Zsegment_length_tmp[take_from] - middle_length_tmp
                    take_from_twist_tmp = self.Zsegment_twist_tmp[take_from] - twist_take_from
                    if self.check_torque(take_from_twist_tmp, take_from_length_tmp, checkL=True, ID=ID): 
                        return 1

                    # update the one that shrunk
                    self.Zsegment_parents_tmp[take_from] = take_from_parents_tmp
                    self.Zsegment_borders_tmp[take_from] = take_from_borders_tmp
                    self.Zsegment_length_tmp[take_from] = take_from_length_tmp
                    self.Zsegment_twist_tmp[take_from] = take_from_twist_tmp
                    self.recompute(take_from, isZ=take_from_isZ, onTmp=True)                    
                    # make the middle one
                    self.make_new_Zloop(middle_borders_tmp, middle_parents_tmp, middle_length_tmp, middle_twist_tmp, onTmp=True)
                    
                # make the new Z loops with that - that is the one inside
                if make_new_Zloop:
                    self.make_new_Zloop(new_borders, new_parents, length, twist, onTmp=True)
                _, _ = self.remove_duplicate_Zloops(ignore=self.remove_Zloops) # this doesnt delete but adds it to self.remove_Zloops
                # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                stepsize = self.compute_span([border_old_complete[direction], border_new_complete[direction]])
                if stepsize > self.L/2: stepsize = self.L - stepsize
                if step_into_ID in self.remove_Zloops and step_into_isZ:
                    step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                        ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                            excludeZ=list(set(self.remove_Zloops))[::-1])
                    if np.isnan(step_into_ID): return 1
                if come_from_ID in self.remove_Zloops and come_from_isZ:
                    come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                        ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                            excludeZ=list(set(self.remove_Zloops))[::-1])
                    if np.isnan(come_from_ID): return 1
                
            # partial to encompassing            
            elif before==2 and after==3: 
                
                # get all the Zloops that are affected by this move
                IDZ_lst = self.check_if_Zloop_changes(border_new_complete[direction], onTmp=True)
                if IDZ_lst[0] is None:
                    continue

                # check if update_step() left some Zloops whose borders are not at regular segments
                # If so, we dissolve them
                stepsize = self.compute_span([border_old_complete[direction], border_new_complete[direction]])
                if stepsize > self.L/2: stepsize = self.L - stepsize
                for IDZ2 in range(len(self.Zsegment_borders_tmp)):
                    if IDZ2 in IDZ_lst: continue
                    for parent in self.Zsegment_parents_tmp[IDZ2]:
                        if self.segment_borders_tmp[parent][0] not in self.Zsegment_borders_tmp[IDZ2] \
                        and self.segment_borders_tmp[parent][1] not in self.Zsegment_borders_tmp[IDZ2] \
                            and IDZ2 not in self.remove_Zloops:
                            if self.Zsegment_parents_tmp[IDZ2] == [parent, parent]:
                                target_segment = 0
                            else:
                                target_segment = list( set(self.Zsegment_parents_tmp[IDZ2]) - set([parent]) )[0]
                            self.segment_length_tmp[target_segment] += self.Zsegment_length_tmp[IDZ2]
                            self.segment_twist_tmp[target_segment]  += self.Zsegment_twist_tmp[IDZ2]
                            self.remove_Zloops.append(IDZ2) # was without self.

                            # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                            if step_into_ID in self.remove_Zloops and step_into_isZ:
                                step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                        excludeZ=sorted(set(self.remove_Zloops))[::-1])
                                if np.isnan(step_into_ID): return 1
                            if come_from_ID in self.remove_Zloops and come_from_isZ:
                                come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                        excludeZ=sorted(set(self.remove_Zloops))[::-1])
                                if np.isnan(come_from_ID): return 1

                        
                # reevaluate this after the moves
                if direction == 0:
                    stepsize = [self.compute_span([a,b]) for a,b in zip(border_new_complete,border_old_complete) if a!=b][0]
                else:
                    stepsize = [self.compute_span([b,a]) for a,b in zip(border_new_complete,border_old_complete) if a!=b][0]
                segmentID_crossedSMC, crossed_SMCs, _, SMC_positions = \
                    self.get_crossed_SMCs(border_old_complete, border_new_complete, direction, stepsize)

                # get the segments which are affected by this Z loop 
                # (must be 2, one for each Zloop border)
                SMC_positions = [(SMC_positions[i], SMC_positions[i + 1]) for i in range(0, len(SMC_positions), 2)]
                crossed_SMCs = [(crossed_SMCs[i], crossed_SMCs[i + 1]) for i in range(0, len(crossed_SMCs), 2)]
                crossed_positions = [[p for p,ctmp in zip(pos,c) if ctmp] for pos,c in zip(SMC_positions, crossed_SMCs)]
                crossed_positions = [x for x in crossed_positions if len(x)>0]  
                for IDZ_ind in range(len(IDZ_lst)):
                    
                    # the following complicated line is necessary because we delete
                    # Zloops as we go through the loop
                    IDZ = IDZ_lst[IDZ_ind]

                    segments_with_crossed_SMCs = [any(x) for x in segmentID_crossedSMC]
                    if segments_with_crossed_SMCs.count(True) != 1:                                          
                        
                        border_new = border_new_complete[direction]
                        if direction == 0:
                            dist = [min([self.compute_distance(p, border_new) for p in pos]) for pos in crossed_positions]
                        else:
                            dist = [min([self.compute_distance(border_new, p) for p in pos]) for pos in crossed_positions] 
                        crossed_segment = [i for i,x in enumerate(segments_with_crossed_SMCs) if x][dist.index(min(dist))] # gets the segment ID of the SMC position with the min distance to the new border
                        ID_receiver = crossed_segment
                    else:
                        # the segment receiving the Zloop's length and twist
                        ID_receiver = segments_with_crossed_SMCs.index(True)
                    isZ_receiver = False
                    
                    # check if the receiver segment has an associated Z loop. If so, this Z loop
                    # is the actual receiver
                    Zloop_receivers = [(ID_receiver in p) if n not in self.remove_Zloops else False for n,p in enumerate(self.Zsegment_parents_tmp)]
                    # obviously, the parents of the Z loop which is to be dissolved will show up as parents
                    # with an associated Z loop. Misleading since we are in the process of deleting said Z loop
                    Zloop_receivers[IDZ] = False
                    Zloop_receivers_count = Zloop_receivers.count(True)
                    if Zloop_receivers_count > 1:
                        # check which border(s) we crossed. If there are mulitple, we give it to the one closest to the position we came from
                        if come_from_isZ:
                            if IDZ == come_from_ID:
                                ID_receiver = self.Zsegment_parents_tmp[come_from_ID][direction]
                                isZ_receiver = False
                            else:
                                ID_receiver = come_from_ID
                                isZ_receiver = True
                        else:
                            ID_receiver = come_from_ID
                            isZ_receiver = False

                        # shouldnt happen since every segment can only by parent to 2 Zloops at every leg
                    elif Zloop_receivers_count == 1:
                        ID_receiver = Zloop_receivers.index(True)
                        isZ_receiver = True

                    # give the Z-loop's length and twist to the receiver and recompute receiver
                    if isZ_receiver:
                        self.Zsegment_length_tmp[ID_receiver] += self.Zsegment_length_tmp[IDZ]
                        self.Zsegment_twist_tmp[ID_receiver] += self.Zsegment_twist_tmp[IDZ]
                    else:
                        self.segment_length_tmp[ID_receiver] += self.Zsegment_length_tmp[IDZ]
                        self.segment_twist_tmp[ID_receiver] += self.Zsegment_twist_tmp[IDZ]
                    self.recompute(ID_receiver, isZ=isZ_receiver, onTmp=True)

                    # if this Z loop was parent to another segment, the segment receiving
                    # the Z-loop's length and twist will also be parent to the Z-loop's child

                    # remove Z loop                        
                    self.remove_Zloops.append(IDZ)
                    
                    # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                    if step_into_ID in self.remove_Zloops and step_into_isZ:
                        step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=sorted(set(self.remove_Zloops))[::-1])
                        if np.isnan(step_into_ID): return 1
                    if come_from_ID in self.remove_Zloops and come_from_isZ:
                        come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=sorted(set(self.remove_Zloops))[::-1])
                        if np.isnan(come_from_ID): return 1
                
                    
                    _, _ = self.remove_duplicate_Zloops(ignore=self.remove_Zloops) # this doesnt delete but adds it to self.remove_Zloops
                    # reevaluate the come_from and step_into segments if the removed Z loop was one of them
                    if step_into_ID in self.remove_Zloops and step_into_isZ:
                        step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=list(set(self.remove_Zloops))[::-1])
                        if np.isnan(step_into_ID): return 1
                    if come_from_ID in self.remove_Zloops and come_from_isZ:
                        come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=list(set(self.remove_Zloops))[::-1])
                        if np.isnan(come_from_ID): return 1

            
            # it might be that we identified several modifications which yielded the same boundaries
            # the result is that we have several identical Zloops. We remove those
            _, _ = self.remove_duplicate_Zloops(ignore=self.remove_Zloops) # this doesnt delete but adds it to self.remove_Zloops
            # reevaluate the come_from and step_into segments if the removed Z loop was one of them
            if step_into_ID in self.remove_Zloops and step_into_isZ:
                step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                        excludeZ=list(set(self.remove_Zloops))[::-1])
                if np.isnan(step_into_ID): return 1
            if come_from_ID in self.remove_Zloops and come_from_isZ:
                come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                        excludeZ=list(set(self.remove_Zloops))[::-1])
                if np.isnan(come_from_ID): return 1
            

            # we now did one round of changes. Recursively enter the function again to re-evaluate if any more changes need to be made
            self.change_boundaries_treated.append(change_boundary)
            step_forbidden = self.make_resolve_Zloop(border_old_complete, border_new_complete, ID, \
                            step_into_ID, step_into_isZ, come_from_ID, come_from_isZ)
            if step_forbidden: return 1
        
        return 0
    
    def remove_duplicate_Zloops(self, ignore=[], update_list=None):
        # make concatemers of all properties
        seen = []
        seen_first = []
        seen_in = []
        remove = []
        for IDZ in range(len(self.Zsegment_borders_tmp)):
            if IDZ in ignore: continue
            concat = [self.Zsegment_borders_tmp[IDZ], self.Zsegment_parents_tmp[IDZ]]
            if concat in seen:
                remove.append(IDZ)
                ind_in_seen = seen.index(concat)
                seen_in.append(seen_first[ind_in_seen])
            else:
                seen.append(concat)
                seen_first.append(IDZ)


        # add the twist that is being deleted to the entry where it first appeared
        for r in range(len(remove)):
            self.Zsegment_twist_tmp[seen_in[r]] += self.Zsegment_twist_tmp[remove[r]]
            self.Zsegment_length_tmp[seen_in[r]] += self.Zsegment_length_tmp[remove[r]]

        self.remove_Zloops += remove
        return remove, update_list        

    def make_new_borders(self, ID, stepsize, direction, isZ=False):
        if direction == 0: # towards the left
            if isZ:
                border_old = self.Zsegment_borders[ID][0]
            else:
                border_old = self.segment_borders[ID][0]
            border_new = border_old - stepsize
            if border_new < 0: # account for circularity of plasmid
                difference = -border_new
                border_new = self.L - difference
            if isZ:
                border_old_complete = [border_old, self.Zsegment_borders[ID][1]]
                border_new_complete = [border_new, self.Zsegment_borders[ID][1]]
            else:
                border_old_complete = [border_old, self.segment_borders[ID][1]]
                border_new_complete = [border_new, self.segment_borders[ID][1]]

        if direction == 1: # towards the right
            if isZ:
                border_old = self.Zsegment_borders[ID][1]
            else:
                border_old = self.segment_borders[ID][1]
            border_new = border_old + stepsize
            if border_new > self.L:
                difference = border_new - self.L
                border_new = difference

            if isZ:
                border_old_complete = [self.Zsegment_borders[ID][0], border_old]
                border_new_complete = [self.Zsegment_borders[ID][0], border_new]
            else:
                border_old_complete = [self.segment_borders[ID][0], border_old]
                border_new_complete = [self.segment_borders[ID][0], border_new]
        return int(border_old), int(border_new), self.change_type(border_old_complete, 'int'), self.change_type(border_new_complete, 'int')

    def make_copy(self, previous=False):
        if previous:
            self.segment_borders_previous = self.my_deecopy(self.segment_borders, 'int')
            self.segment_length_previous = self.my_deecopy(self.segment_length, 'int')
            self.segment_twist_previous = self.my_deecopy(self.segment_twist, 'float')
            self.segment_relTwist_previous = self.my_deecopy(self.segment_relTwist, 'float')
            self.segment_torque_previous = self.my_deecopy(self.segment_torque, 'float')

            self.Zsegment_borders_previous = self.my_deecopy(self.Zsegment_borders, 'int')
            self.Zsegment_length_previous = self.my_deecopy(self.Zsegment_length, 'int')
            self.Zsegment_twist_previous = self.my_deecopy(self.Zsegment_twist, 'float')
            self.Zsegment_relTwist_previous = self.my_deecopy(self.Zsegment_relTwist, 'float')
            self.Zsegment_torque_previous = self.my_deecopy(self.Zsegment_torque, 'float')
            self.Zsegment_parents_previous = self.my_deecopy(self.Zsegment_parents, 'int')
        else:
            self.segment_borders_tmp = self.my_deecopy(self.segment_borders, 'int')
            self.segment_length_tmp = self.my_deecopy(self.segment_length, 'int')
            self.segment_twist_tmp = self.my_deecopy(self.segment_twist, 'float')
            self.segment_relTwist_tmp = self.my_deecopy(self.segment_relTwist, 'float')
            self.segment_torque_tmp = self.my_deecopy(self.segment_torque, 'float')

            self.Zsegment_borders_tmp = self.my_deecopy(self.Zsegment_borders, 'int')
            self.Zsegment_length_tmp = self.my_deecopy(self.Zsegment_length, 'int')
            self.Zsegment_twist_tmp = self.my_deecopy(self.Zsegment_twist, 'float')
            self.Zsegment_relTwist_tmp = self.my_deecopy(self.Zsegment_relTwist, 'float')
            self.Zsegment_torque_tmp = self.my_deecopy(self.Zsegment_torque, 'float')
            self.Zsegment_parents_tmp = self.my_deecopy(self.Zsegment_parents, 'int')

    def transcribe_copy(self):
        self.segment_borders = self.segment_borders_tmp[:]
        self.segment_length = self.segment_length_tmp[:]
        self.segment_twist = self.segment_twist_tmp[:]
        self.segment_relTwist = self.segment_relTwist_tmp[:]
        self.segment_torque = self.segment_torque_tmp[:]

        self.Zsegment_borders = self.Zsegment_borders_tmp[:]
        self.Zsegment_length = self.Zsegment_length_tmp[:]
        self.Zsegment_twist = self.Zsegment_twist_tmp[:]
        self.Zsegment_relTwist = self.Zsegment_relTwist_tmp[:]
        self.Zsegment_torque = self.Zsegment_torque_tmp[:]
        self.Zsegment_parents = self.Zsegment_parents_tmp[:]

    def compute_distance(self, a, b):
        dist = a-b
        if dist < 0:
            dist = self.L + dist
        return dist

    def compute_span(self, border):
        if len(border) == 0: return []
        if isinstance(border[0], list):
            spans = []
            for b in border:
                spans.append( self.compute_span(b) )
            return spans 
        
        border_wraps = False
        if border[0]>border[1]:
            border_wraps = True
        if border_wraps:
            span = self.L + (border[1]-border[0])
        else:
            span = border[1]-border[0]
        return int(span)


    def update_step(self, ID, come_from_ID, come_from_isZ, \
                    border_old_complete, border_new_complete, stepsize, \
                        direction, work_on_premade_copy=False, do_not_transcribe=False, \
                            add_to_Zloop_removal=False):

        # we first need to check if the move is allowed for all potential IDZ's. 
        # so we make a copy of the current segments and Zsegments, then evaluate on these
        # if all moves are allowed. If so, we copy the tmp ones over to the real ones.
        # if not, just exit and move on
        other_direction = list(set([0,1])-set([direction]))[0]
        border_old_complete_backup = self.my_deecopy(border_old_complete, 'int')
        if not work_on_premade_copy: 
            self.make_copy()
            self.treated_IDZ = []
        if not add_to_Zloop_removal:
            self.remove_Zloops = []
            self.remove_replace_Zloops_lastStep = []
                
        # check if there are any Z loop we have to handle. The output tells us
        # that the IDZ's Zloop is changing. if None is returned, we are working on the 
        # segment itself, i.e. ID
        IDZ_lst = self.check_if_Zloop_changes(border_old_complete[direction])

        # evaluate the changes on the copy (same naming but everything ends on _tmp)
        for IDZ in IDZ_lst:
            if IDZ in self.treated_IDZ:
                continue
            else:
                self.treated_IDZ.append( IDZ )

            step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(\
                    border_new_complete, ID, direction, checkZonly=False, preferZ=True, \
                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
            if np.isnan(step_into_ID): return 1
            come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(\
                    border_old_complete, ID, direction, checkZonly=False, preferZ=True, \
                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
            if np.isnan(come_from_ID): return 1

            # update Zloop if we are working on one. 
            # Note that we are enlarging the segment border no matter what
            if IDZ is not None:

                # check if we cross legs. In this case we resolve the Z loop
                other_leg = self.Zsegment_borders[IDZ][direction]
                border_old_tmp = border_old_complete_backup[direction]
                border_new_tmp = border_new_complete[direction]
                if np.abs(border_old_tmp-border_new_tmp) > stepsize: # then we happen to wrap
                    if   border_old_tmp > border_new_tmp: border_old_tmp -= self.L
                    elif border_old_tmp < border_new_tmp: border_old_tmp += self.L
                crossed_own_leg = self.check_for_overstepping_myself(other_leg, \
                    border_old_complete_backup, border_new_complete, direction)
                # if we crossed our own leg, this Z loop no longer exists. We give its length
                # and twist to the segment which encompasses it (not its parents)
                # first we need to make the move until the two SMC arms are next to each other
                # left = 0, right = 1
                moving_leg_is_left_right_Zleg = self.Zsegment_borders[IDZ].index(border_old_complete_backup[direction])
                if crossed_own_leg:                    

                    # make the move as far as possible and transfer length and twist to IDZ
                    # until the SMC arms are next to each other
                    if direction == 0:
                        intermediate_step = self.compute_span([other_leg, border_old_complete[direction]])-1
                    else:
                        intermediate_step = self.compute_span([border_old_complete[direction], other_leg])-1

                    if intermediate_step == 0: continue
                    border_new_intermediate = self.my_deecopy(border_old_complete, 'int')
                    if direction == 0:
                        border_new_intermediate[direction] = border_old_complete[direction]-intermediate_step
                    else:
                        border_new_intermediate[direction] = border_old_complete[direction]+intermediate_step
                    if border_new_intermediate[direction] > self.L:
                        border_new_intermediate[direction] -= self.L
                    if border_new_intermediate[direction] < 0:
                        border_new_intermediate[direction] += self.L
                    # check if the new intermediate border is permitted or if we would land on another SMC
                    SMC_positions = self.keep_first_unique_and_set_to_nan([item for sublist in self.segment_borders_tmp for item in sublist])
                    SMC_positions[0] = float(np.nan) # the first segment is always the backbone without SMCs
                    SMC_positions[1] = float(np.nan) # the first segment is always the backbone without SMCs
                    SMC_positions[ID*2] = float(np.nan)
                    SMC_positions[ID*2+1] = float(np.nan)
                    if border_new_intermediate[direction] in SMC_positions:
                        return 1
                    # the z loop under consideration will get length and twist before we remove it
                    
                    # find the moving leg and its parent. If the parent also has another
                    # Z loop asscoiated, then the IDZ's content (twist and length)
                    # will go to it. If not, then it'll go to the parent segment
                    # so first check if we have another Z loop                    
                    fixed_leg_is_left_right_Zleg = list(set([0,1])-set([moving_leg_is_left_right_Zleg]))[0]
                    other_Zloop = [parent[fixed_leg_is_left_right_Zleg]==ID and \
                                   border[fixed_leg_is_left_right_Zleg]!=border_old_complete_backup[other_direction]\
                                    for parent,border in zip(self.Zsegment_parents, self.Zsegment_borders)]
                    other_Zloop_count = other_Zloop.count(True)
                    if other_Zloop_count > 1:
                        # get the moving leg of the parent (that is the direction) and see which of the Z loops
                        # are associated with that position
                        moving_pos = border_old_complete_backup[direction]
                        other_Zloop_narrowed = [(moving_pos in b and i) for b,i in zip(self.Zsegment_borders, other_Zloop)]

                        target_segment = other_Zloop_narrowed.index(True)
                        target_segment_isZ = 1
                    elif other_Zloop_count == 0:
                        target_segment = ID
                        target_segment_isZ = 0
                    else: # one Z loop found
                        target_segment = other_Zloop.index(True)
                        target_segment_isZ = 1

                    # give the receiver the length and twist of
                    if target_segment_isZ:
                        self.Zsegment_length_tmp[target_segment] += self.Zsegment_length_tmp[IDZ]
                        self.Zsegment_twist_tmp[target_segment]  += self.Zsegment_twist_tmp[IDZ]
                    else:
                        self.segment_length_tmp[target_segment] += self.Zsegment_length_tmp[IDZ]
                        self.segment_twist_tmp[target_segment]  += self.Zsegment_twist_tmp[IDZ]
                    self.recompute(target_segment, isZ=target_segment_isZ, onTmp=True)
                
                    # save which Zloop was dissolved and what is its replacement. With replacement I mean here
                    # not the target segment, but the parent in the direction 'direction' of the Zloop
                    # which is being dissolved. Reason: This will be one parent of a new Zloop if we form one
                    self.remove_replace_Zloops_lastStep.append( [IDZ, self.Zsegment_parents[IDZ]] )

                    # with IDZ poised for deletion, reevaluate come_from and step_into
                    self.remove_Zloops.append( IDZ ) # poise for removal
                    if step_into_ID in self.remove_Zloops and step_into_isZ:
                        step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=sorted(set(self.remove_Zloops))[::-1])
                        if np.isnan(step_into_ID): return 1
                    if come_from_ID in self.remove_Zloops and come_from_isZ:
                        come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                            ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                excludeZ=sorted(set(self.remove_Zloops))[::-1])
                        if np.isnan(come_from_ID): return 1
                        

                    # move by the intermediate step
                    step_forbidden = self.update_step(ID, come_from_ID, come_from_isZ, \
                        border_old_complete, border_new_intermediate, intermediate_step, \
                            direction, work_on_premade_copy=True, add_to_Zloop_removal=True, do_not_transcribe=True) # we make changes here only on tmp
                    if step_forbidden: return 1
                    
                    border_old_complete = self.my_deecopy(border_new_intermediate, 'int')
                    # at this stage we moved the SMC arms right before a new Z loop is to be made
                    # we can then enter self.make_Zloop() to do that

                    # we later want to make the rest of the step for which we will look up
                    # the segment length and twist in IDZ. However, IDZ will be deleted. We 
                    # should instead look in target_segment. -> call IDZ = target_segment if
                    # target segment is a Zloop. If not, fall back to ID by assigning IDZ = None
                    if target_segment_isZ:
                        IDZ = target_segment
                    else:
                        IDZ = None

                ## update borders of IDZ as well if IDZ still exists
                if IDZ is not None:
                    moving_leg_is_left_right_Zleg_inv = int(not bool(moving_leg_is_left_right_Zleg))
                    if border_new_complete[direction] == self.Zsegment_borders_tmp[IDZ][moving_leg_is_left_right_Zleg_inv] and IDZ not in self.remove_Zloops:
                        
                        if self.Zsegment_parents_tmp[IDZ] == [ID, ID]:
                            parent = 0
                        else:
                            parent = list( set(self.Zsegment_parents_tmp[IDZ]) - set([ID]) )[0]
                        self.segment_length_tmp[parent] += self.Zsegment_length_tmp[IDZ]
                        self.segment_twist_tmp[parent]  += self.Zsegment_twist_tmp[IDZ]
                        self.recompute(parent, onTmp=True)
                        self.remove_Zloops.append(IDZ)

                    tmp = self.my_deecopy(self.Zsegment_borders_tmp[IDZ], 'int')
                    tmp_after = self.my_deecopy(tmp, 'int')
                    tmp_after[moving_leg_is_left_right_Zleg] = border_new_complete[direction]
                    if self.compute_span(tmp) < self.L*2/4 and self.compute_span(tmp_after) > self.L*3/4 and IDZ not in self.remove_Zloops:

                        if self.Zsegment_parents_tmp[IDZ] == [ID, ID]:
                            parent = 0
                        else:
                            parent = list( set(self.Zsegment_parents_tmp[IDZ]) - set([ID]) )[0]
                        self.segment_length_tmp[parent] += self.Zsegment_length_tmp[IDZ]
                        self.segment_twist_tmp[parent]  += self.Zsegment_twist_tmp[IDZ]
                        self.recompute(parent, onTmp=True)
                        self.remove_Zloops.append(IDZ)
                    self.Zsegment_borders_tmp[IDZ][moving_leg_is_left_right_Zleg] = border_new_complete[direction]


                    if IDZ in self.remove_Zloops: 
                        IDZ = None
                        # with IDZ poised for deletion, reevaluate come_from and step_into
                        if step_into_ID in self.remove_Zloops and step_into_isZ:
                            step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                                ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
                            if np.isnan(step_into_ID): return 1
                        if come_from_ID in self.remove_Zloops and come_from_isZ:
                            come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                                ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, \
                                    excludeZ=sorted(set(self.remove_Zloops))[::-1])
                            if np.isnan(come_from_ID): return 1
                
            self.segment_borders_tmp[ID] = border_new_complete

            # find out if the Zloop IDZ expands (concomittant with ID) or contracts
            # (then IDZ is adjacent to ID)
            # we have to determine if IDZ is getting larger (then we add the stepsize)
            # or smaller (then we take it away). The change in border has already been
            # made, so we can compare Zsegment_borders[IDZ] (before the step)
            # to Zsegment_borders_tmp[IDZ] (after the step)
            Zloop_contracts = False
            if IDZ is not None:
                if direction == 0:
                    stepsize = self.compute_distance(border_old_complete[direction], \
                                                    border_new_complete[direction])
                else:
                    stepsize = self.compute_distance(border_new_complete[direction], \
                                                    border_old_complete[direction])
                # we might have done the intermediate step. 
                # if we cannot do more than that, leave it at that. Just continue
                if stepsize == 0: continue 
                
                span_before = self.compute_span(self.Zsegment_borders[IDZ])
                span_after = self.compute_span(self.Zsegment_borders_tmp[IDZ])                
                if span_before > span_after:
                    Zloop_contracts = True                

            if IDZ is None or Zloop_contracts:
                segment_length = self.segment_length_tmp[ID] + stepsize
            else: # the segment is concomittant with a Zloop
                segment_length = self.Zsegment_length_tmp[IDZ] + stepsize

            if come_from_isZ:
                step_into_length = self.Zsegment_length_tmp[come_from_ID] - stepsize
            else:
                step_into_length = self.segment_length_tmp[come_from_ID] - stepsize
                
            # if we do not allow crossing, then a segment length of 0 is forbidden if there
            # is a non-zero twist, otherwise there would be infinite torque
            if step_into_length <= 0:
                return 1                

            ## compute new segment twist
            # first for the piece we came from
            if IDZ is None or Zloop_contracts: # no Z loop is being changed by our move. just get the segment under consideration
                segment_twist = self.segment_twist_tmp[ID]
            else: # the segment is concomittant with a Zloop
                segment_twist = self.Zsegment_twist_tmp[IDZ]                
            # relative twist from the loop we are currently in, as far as we can travel
            if come_from_isZ:
                step_into_relTwist = self.Zsegment_relTwist_tmp[come_from_ID]
            else:

                step_into_relTwist = self.segment_relTwist_tmp[come_from_ID]
            added_twist_segment    = -self.PARAMS["SMC"]["Tw"] + stepsize*step_into_relTwist
            added_twist_step_into  =  self.PARAMS["SMC"]["Tw"] - stepsize*step_into_relTwist
            segment_twist += added_twist_segment
            if come_from_isZ:
                step_into_twist   = self.Zsegment_twist_tmp[come_from_ID] + added_twist_step_into
            else:
                step_into_twist   = self.segment_twist_tmp[come_from_ID] + added_twist_step_into
            # check if making that move is allowed according to the torques 
            if self.check_torque(segment_twist, segment_length, checkL=True, ID=ID): 
                return 1
            if self.check_torque(step_into_twist, step_into_length, checkL=True, ID=ID): 
                return 1

            # if we enlarge a Zloop, update its length and twist etc and borders. 
            # ALSO update the corresponding segment border
            # if we just update a segment, do it all on the segment
            # update the segment we are coming from as well
            
            # update segment we are coming from
            if come_from_isZ:
                self.Zsegment_length_tmp[come_from_ID] = step_into_length
                self.Zsegment_twist_tmp[come_from_ID] = step_into_twist                    
            else:
                self.segment_length_tmp[come_from_ID] = step_into_length
                self.segment_twist_tmp[come_from_ID] = step_into_twist
            self.recompute(come_from_ID, isZ=come_from_isZ, onTmp=True)

            # update segment length and twist of the moving segment            
            if IDZ is None or Zloop_contracts:
                self.segment_length_tmp[ID] = segment_length
                self.segment_twist_tmp[ID] = segment_twist
                self.recompute(ID, onTmp=True)
            else:
                self.Zsegment_length_tmp[IDZ] = segment_length
                self.Zsegment_twist_tmp[IDZ] = segment_twist
                self.recompute(IDZ, isZ=True, onTmp=True)

        # if we made it until here, all changes were possible. We adapt them
        if (not work_on_premade_copy) and (not do_not_transcribe): 
            self.transcribe_copy()
        # remove Z loops which were poised for removal
        if not add_to_Zloop_removal:
            if do_not_transcribe:
                self.clean_selfcontained_Zloops(onTmp=True) # adds Zsegments to self.remove_Zloops                
            else:
                self.clean_selfcontained_Zloops() # adds Zsegments to self.remove_Zloops
                [self.remove_Zloop(i) for i in sorted(set(self.remove_Zloops))[::-1]]; self.remove_Zloops = [] # sorting and inverting the order to ensure that we first delete larger indices
      
        return 0 # return flag 0 to signal that step was allowed and was made. flag is 1 if step was not allowed
    
    def clean_selfcontained_Zloops(self, onTmp=False):
        # if a Zloop completey contains one of its parents
        # this function doesnt delete them, it just adds them to self.remove_Zloops (it is taking care of length/twist transfer though)

        if onTmp:
            for IDZ in range(len(self.Zsegment_parents_tmp)):
                parents = self.Zsegment_parents_tmp[IDZ]
                borders = [self.Zsegment_borders_tmp[IDZ][0]-1, self.Zsegment_borders_tmp[IDZ][1]+1]
                relation = [self.determine_interval_cases(borders, \
                        self.segment_borders_tmp[p], self.L) for p in parents]
                if 3 in relation:
                    receiver = parents[relation.index(3)]
                    self.segment_length_tmp[receiver] += self.Zsegment_length_tmp[IDZ]
                    self.segment_twist_tmp[receiver] += self.Zsegment_twist_tmp[IDZ]
                    self.remove_Zloops.append( IDZ )
        else:
            for IDZ in range(len(self.Zsegment_parents)):
                parents = self.Zsegment_parents[IDZ]
                borders = [self.Zsegment_borders[IDZ][0]-1, self.Zsegment_borders[IDZ][1]+1]
                relation = [self.determine_interval_cases(borders, \
                        self.segment_borders[p], self.L) for p in parents]
                if 3 in relation:
                    receiver = parents[relation.index(3)]
                    self.segment_length[receiver] += self.Zsegment_length[IDZ]
                    self.segment_twist[receiver] += self.Zsegment_twist[IDZ]
                    self.remove_Zloops.append( IDZ )
        
        
    def make_new_Zloop(self, Zsegment_border, Zsegment_parents, Zsegment_length, Zsegment_twist, onTmp=False):
        if onTmp:
            self.Zsegment_borders_tmp.append( [int(x) for x in Zsegment_border] )
            self.Zsegment_parents_tmp.append( [int(x) for x in Zsegment_parents] )
            self.Zsegment_length_tmp.append( Zsegment_length )
            self.Zsegment_twist_tmp.append( Zsegment_twist )
            self.Zsegment_relTwist_tmp.append( 0 ) # recompute
            self.Zsegment_torque_tmp.append( 0 ) # recompute        
            self.recompute(len(self.Zsegment_borders_tmp)-1, isZ=True, onTmp=onTmp)
        else:
            self.Zsegment_borders.append( [int(x) for x in Zsegment_border] )
            self.Zsegment_parents.append( [int(x) for x in Zsegment_parents] )
            self.Zsegment_length.append( Zsegment_length )
            self.Zsegment_twist.append( Zsegment_twist )
            self.Zsegment_relTwist.append( 0 ) # recompute
            self.Zsegment_torque.append( 0 ) # recompute        
            self.recompute(len(self.Zsegment_borders)-1, isZ=True)

    def remove_Zloop(self, ID, update_list=None, onTmp=False):
        # delete the entries relating to this Z loop
        if onTmp:
            del self.Zsegment_borders_tmp[ID]
            del self.Zsegment_length_tmp[ID]
            del self.Zsegment_twist_tmp[ID]
            del self.Zsegment_relTwist_tmp[ID]
            del self.Zsegment_torque_tmp[ID]
            del self.Zsegment_parents_tmp[ID]
        else:
            del self.Zsegment_borders[ID]
            del self.Zsegment_length[ID]
            del self.Zsegment_twist[ID]
            del self.Zsegment_relTwist[ID]
            del self.Zsegment_torque[ID]
            del self.Zsegment_parents[ID]

        if update_list is not None:
            if isinstance(update_list[0], list): # update_list can also be a list of lists
                for n in range(len(update_list)):
                    update_list[n] = [x-1 if (x>ID) else x for x in update_list[n]]
            else:
                update_list = [x-1 if (x>ID) else x for x in update_list]
        return update_list


    def get_crossed_SMCs(self, border_old_complete, border_new_complete, direction, stepsize, onTmp=False):
        if onTmp:
            segment_borders = self.segment_borders_tmp
        else:
            segment_borders = self.segment_borders
        segment_borders_clean = [s if s[0]!=s[1] else [float(np.nan),float(np.nan)] for s in segment_borders]
        SMC_positions = self.keep_first_unique_and_set_to_nan([item for sublist in segment_borders_clean for item in sublist])
        SMC_positions[0] = float(np.nan) # the first segment is always the backbone without SMCs
        SMC_positions[1] = float(np.nan) # the first segment is always the backbone without SMCs

        border_old_tmp = border_old_complete[direction]
        border_new_tmp = border_new_complete[direction]
        if np.abs(border_old_tmp-border_new_tmp) > stepsize: # then we happen to wrap
            if border_old_tmp > border_new_tmp: border_old_tmp -= self.L
            elif border_old_tmp < border_new_tmp: border_old_tmp += self.L
        crossed_SMCs = [self.check_for_overstepping_myself(pos, border_old_complete, \
                        border_new_complete, direction) for pos in SMC_positions]
        segmentID_crossedSMC = [(crossed_SMCs[i], crossed_SMCs[i + 1]) for i in range(0, len(crossed_SMCs), 2)]
        return segmentID_crossedSMC, crossed_SMCs, border_old_tmp, SMC_positions

    def check_if_move_sits_on_existing_SMC_position(self, border_new, stepsize, direction, other_leg):
        step_forbidden = False
        original_stepsize = self.my_deecopy(stepsize, 'int')
        while any([border_new in b for b in self.segment_borders]):
            if direction == 0:
                border_new -= 1                    
            else:
                border_new += 1
            if border_new < 0:
                border_new = self.L + border_new
            if border_new > self.L:
                border_new -= self.L
            stepsize += 1
            if stepsize > 1.5*original_stepsize or border_new == other_leg: 
                step_forbidden = True
                break # break just the while loop
        return int(border_new), int(stepsize), bool(step_forbidden)
    
    def check_for_overstepping_myself(self, stable_leg, border_old_complete, border_new_complete, direction):
        move_old = border_old_complete[direction]
        move_new = border_new_complete[direction]
        move_wraps = False
        if direction == 0 and move_old < move_new: move_wraps = True
        if direction == 1 and move_old > move_new: move_wraps = True
        overstepping = False
        if move_wraps and \
            (\
                (stable_leg > move_old and stable_leg > move_new) or \
                (stable_leg < move_old and stable_leg < move_new) \
            ): overstepping = True
        if (not move_wraps) and \
            (\
                (move_old < stable_leg < move_new) or \
                (move_new < stable_leg < move_old) \
            ): overstepping = True
        return overstepping
    
    def make_SMC_step(self):
        # we update the SMCs in random order such that we do not prioritize any
        # we dont take the first one, i.e the unlooped plasmid, into account
        # However, we do prioritize newly spawned SMCs so we dont end up with
        # an unphysical loop length of 0 (and equal sement borders)
        spawned_IDs = self.where([x[0]==x[1]-2 for x in self.segment_borders])#[0]
        other_IDs = list( set( np.arange(1, self.numSegments) ) - set( spawned_IDs ) )
        segment_IDs = np.random.permutation( other_IDs )
        segment_IDs = np.random.permutation( spawned_IDs ).tolist() + segment_IDs.tolist()

        steps_forbidden = []
        self.step_forbidden = [0] * self.numSegments
                      
        for ID in segment_IDs: 
            
            # we take away from the SMC lifetime no matter what
            self.SMC_lifetime[ID] -= 1
            # update direction counter
            self.direction_lifetime[ID] -= 1

            # we also count a step - no matter what
            self.SMC_steps[ID] += 1

            # check if we exhausted the number of SMC steps (useful to simulate an EQ mutant)
            if self.SMC_steps[ID] > self.allowed_SMC_steps[ID]: continue

            # get direction and step size
            direction = self.SMCdirection[ID]
            stepsize  = self.generateNumber(self.PARAMS["SMC"]["stepsize"], self.PARAMS["SMC"]["stepsize_mode"], dtype=float)

            # check if the loop is larger than the entire plasmid. If so, stall
            if self.segment_length[ID] + stepsize > self.L: continue

            # compute the new boundaries if the step is successful
            _, border_new, border_old_complete, border_new_complete = \
                self.make_new_borders(ID, stepsize, direction)
            if border_new_complete[0]==border_new_complete[1]: continue # we can't move onto ourselves
                
            # check if we stepped over ourselves. That is also forbidden
            other_direction = list(set([0,1]) - set([direction]))[0]
            if self.check_for_overstepping_myself(border_old_complete[other_direction], border_old_complete, border_new_complete, direction):
                continue

            other_leg = border_new_complete[other_direction]
            border_new, stepsize, step_forbidden = \
                self.check_if_move_sits_on_existing_SMC_position(border_new, stepsize, direction, other_leg)
            if step_forbidden: continue # move to the next step
            border_new_complete[direction] = border_new

            # evaluate if there are any, one, or multiple crossings happening along the way.
            # we discretize in 1 bp steps. So along the stepsize we want to take, we only execute 
            # the step until the next SMC is encountered. Then we just barely jump over it, ignoring 
            # twist exchanges for this 1 bp step. In the next step, we can go on as usual
            # as for the SMCs we might skip, we do not take into account those that did not move yet
            # i.e. the ones that have equal left and right boundary
            come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, ID, direction, preferZ=True)
            if np.isnan(come_from_ID): continue

            _, crossed_SMCs, border_old_tmp, SMC_positions = \
                self.get_crossed_SMCs(border_old_complete, border_new_complete, direction, stepsize)
            
            crossing = False
            if any(crossed_SMCs): 
                crossing = True  
                if (not self.PARAMS["SMC"]["allow_Zloops"]):
                    continue # we do not allow Zloops, thus crossing SMCs is forbidden       

            # if there is no crossing, i.e. nothing about Z loops
            if not crossing:
                step_forbidden = self.update_step(ID, come_from_ID, come_from_isZ, \
                    border_old_complete, border_new_complete, stepsize, direction)
                steps_forbidden.append(step_forbidden)
                if step_forbidden:                     
                    continue
                self.remove_Zloops = []
                


            ## DEAL WITH CROSSINGS -> Z-loops
            # we have a crossing, meaning that we cannot take the full stepsize
            if crossing:
                # determine the step size: how far can we travel before bumping into another SMC
                crossed_SMCpos = [pos for pos,crossed in zip(SMC_positions, crossed_SMCs) if crossed]
                stepsizes = [int(np.abs(self.compute_span([border_old_tmp,pos]))) for pos in crossed_SMCpos]
                stepsizes = [s-self.L if s > self.L else s for s in stepsizes]
                stepsizes = [self.L-s if s > self.L/2 else s for s in stepsizes]
                crossed_SMC_ID = self.argmin(stepsizes)
                stepsize = stepsizes[crossed_SMC_ID] # this is the step size to land exactly on top of the crossed SMC

                crossed_SMCpos = crossed_SMCpos[crossed_SMC_ID]
                # redefine the new segment border. We make this one bp longer than it actually in 
                # order to get past the SMC for the next round
                stepsize += 1 # now we land exactly after the SMC to cross
                _, border_new, border_old_complete, border_new_complete = \
                    self.make_new_borders(ID, stepsize, direction)
                if border_new_complete[0] == border_new_complete[1]: # if we land onto ourselves, we cannot make the move
                    steps_forbidden.append(True)
                    continue
                
                # check if we would move onto an already existing SMC
                other_direction = list(set([0,1])-set([direction]))[0]
                other_leg = border_new_complete[other_direction]
                border_new, stepsize, step_forbidden = \
                    self.check_if_move_sits_on_existing_SMC_position(border_new, stepsize, \
                    direction, other_leg) # this function makes larger steps in case we land on an SMC
                steps_forbidden.append(step_forbidden)
                if step_forbidden:                     
                    continue # move to the next step
                border_new_complete[direction] = border_new
                
                # before we update the step until the crossing, we make a copy of the current
                # quantities to find out where we are stepping into if we make a new Z loop
                self.make_copy(previous=True)
                # let us update the step until the crossing. We do this on a copy, as usual.
                # we also do not transcribe this in the end even if successful because we first have
                # to evaluate if we can make/resolve the necessary Z loops                
                step_forbidden = self.update_step(ID, come_from_ID, come_from_isZ, \
                    border_old_complete, border_new_complete, stepsize, direction, do_not_transcribe=True)
                steps_forbidden.append(step_forbidden)
                if step_forbidden:                     
                    continue


                # the last step might have changed our borders because we could have crossed
                # a Z loop position which led to its dissolution. Therefore, evaluate 
                # at the current stage where we are coming from and where we step into
                come_from_ID, come_from_isZ = self.evaluate_parents_of_moved_SMC_leg(border_old_complete, \
                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, excludeZ=sorted(set(self.remove_Zloops))[::-1])
                if np.isnan(come_from_ID): continue
                step_into_ID, step_into_isZ = self.evaluate_parents_of_moved_SMC_leg(border_new_complete, \
                    ID, direction, evaluate_on_previous_step=True, stepsize=stepsize, preferZ=True, excludeZ=sorted(set(self.remove_Zloops))[::-1])
                if np.isnan(step_into_ID): continue

                # check if we make or dissolve any Z loops
                if [come_from_ID, come_from_isZ] != [step_into_ID, step_into_isZ]:
                    self.change_boundaries_treated = []
                    self.implemented_changes = []
                    step_forbidden = self.make_resolve_Zloop(border_old_complete, border_new_complete, ID, \
                                step_into_ID, step_into_isZ, come_from_ID, come_from_isZ)
                    steps_forbidden.append(step_forbidden)
                    if step_forbidden:                     
                        continue
                    [self.remove_Zloop(i, onTmp=True) for i in sorted(set(self.remove_Zloops))[::-1]]; self.remove_Zloops = []

                else:
                    steps_forbidden.append(True)
                    continue
                self.remove_Zloops = []
                # if we came until here, then everything is good. We can transcribe the copy
                self.transcribe_copy()

    def topo_action(self):
        if self.PARAMS["topo"]["k_topo"] < 1:
            if not self.MC_criterion(self.PARAMS["topo"]["k_topo"]): return # only take action if the MC criterion says so
     
        # find where it spawns
        spawning_location = spawning_location = random.randint(0, self.L)
        # get the segment where it spawned
        ID, isZ = self.get_segmentID(spawning_location, includeZ=True)
        # get twist and torque in this segment
        if isZ:
            Tw = self.Zsegment_twist[ID]
            torque = self.Zsegment_torque[ID]
            length = self.Zsegment_length[ID]
        else:
            Tw = self.segment_twist[ID]
            torque = self.segment_torque[ID]
            length = self.segment_length[ID]       

        DLk = int( self.compute_Lkchange_topo(torque, maxLk=Tw) ) # we cannot remove more than the twist in the segment        
        if DLk == 0: return

        self.current_topo_actions += 1
        self.current_accumulatedLk += DLk
        # remove the linking number change from the segment and update related quantities
        if isZ:
            self.Zsegment_twist[ID] -= DLk
        else:
            self.segment_twist[ID] -= DLk
        self.recompute(ID, isZ=isZ)
            
    def get_repetitions(self, k):
        if 0 < k <= 1: return 1
        elif k == 0: return 0
        else: return np.random.poisson(lam=k)
        
    def run(self):

        ### loop over simulation steps
        for self.step in range(self.PARAMS["system"]["numSteps"]):

            self.remove_Zloops = []                

            ## check SMC dissociation
            self.SMCs_dissociate()            

            ## check if an SMC spawns
            [self.SMC_spawns() for _ in range(self.get_repetitions(self.PARAMS["SMC"]["k_on"]))]

            ## check if we need to define new directions
            self.assign_directions()

            ## make SMC action
            self.make_SMC_step()

            ## check topo action
            self.current_topo_actions = self.my_deecopy(self.topoActions[-1], dtype='int')
            self.current_accumulatedLk = self.my_deecopy(self.accumulatedLk[-1], dtype='int')
            [self.topo_action() for _ in range(self.get_repetitions(self.PARAMS["topo"]["k_topo"]))]

            # save the history            
            if len(self.topoActions) <= self.step+1:
                self.topoActions.append( self.current_topo_actions )
            if len(self.accumulatedLk) <= self.step+1:
                self.accumulatedLk.append( self.current_accumulatedLk )


if __name__ == '__main__':
    module = SSS()
    module.run()
    # get the topo-removed Lk's as module.accumulatedLk
    print('RAN')