function [p, t, be] = ...
    remesh_global(fd, fh, h0, bbox, pfix, varargin)
%2D-mesh generator using signed distance & size functions.
%Adapted from J. Koko.
%--------------------------------------------------------------------
% (c) 2009, Koko J., ISIMA, koko@isima.fr
% Koko J., A Matlab mesh generator for the two-dimensional finite element method,
% Applied Mathematics and Computation 250, p. 650-664 (2015)
%--------------------------------------------------------------------
% Scale factor and time step
Fscale=1.2; dt=.1;
% Convergence tolerances
epsp=.005; epsr=.5; deps=sqrt(eps)*h0; geps=.001*h0;
iterMax=5000; mp=5;
% Minimum triangle quality
qmin=.5;

pgon = varargin{1}{1};

% Initialize the fix points set
npf=size(pfix,1);

% Extract mesh size function name and compare strings
fhn=func2str(fh);

% Compute the approximate medial axis
if strcmpi(fhn,'hgeom')
    hh0=h0/2;
    ex=(bbox(1,1):hh0:bbox(2,1)); ey=(bbox(1,2):hh0:bbox(2,2));
    [x,y]=meshgrid(ex,ey);
    z=feval(fd,[x(:) y(:)],varargin{:});
    [mx,nx]=size(x); z=reshape(z,mx,nx);
    [zx,zy]=gradient(z,hh0);  zz=sqrt(zx.*zx+zy.*zy);
    imax=find(zz<.99 & z<=0); pmax=[x(imax) y(imax)]; pmax=[pmax; pfix];
    clear ex ey x y z zx zy zz imax
end

% Initial grid
[x,y]=meshgrid(bbox(1,1):h0:bbox(2,1),bbox(1,2):h0*sqrt(3)/2:bbox(2,2));
x(2:2:end,:)=x(2:2:end,:)+h0/2;
p = [x(:),y(:)];
clear x y

% Remove points outside the region, apply the rejection method
p=p(feval(fd,p,varargin{:})<geps,:);

if strcmpi(fhn,'hgeom')
    r0=feval(fh,p,fd,pmax,varargin{:});
else
    r0=feval(fh,p,varargin{:});
end

r0=1./r0.^2;
p=[pfix; p(rand(size(p,1),1)<r0./max(r0),:)];
clear r0

% Remove nodes outside the domain
np0=size(p,1);
dp=feval(fd,p,varargin{:});
ii=find(dp<geps); q=p(ii,:);

% Add fixed nodes
if (npf>0), p=setdiff(q,pfix,'rows'); p=[pfix; p];
else, p=q; end
clear q dp

iter=0; ifix=[];  tolp=1; tolr=10^2;
while iter<iterMax && tolp>epsp
    iter=iter+1;
    
    % Delaunay triangluation
    if tolr>epsr
        np=size(p,1);
        p0=p;
        t=delaunayn(p);
        % Reject triangles with centroid outside the domain
        pm=(p(t(:,1),:)+p(t(:,2),:)+p(t(:,3),:))/3;
        t=t(feval(fd,pm,varargin{:})<-geps,:);
        % Reorder (locally) triangle vertices counter clockwise
        ar=tarea(p,t); it=find(ar<0);
        itt=t(it,2); t(it,2)=t(it,3); t(it,3)=itt;
        % Form all edges without duplication & extract boundary nodes
        [e,ib]=kmg2dedg(t);
        be=e(ib,:);
        bn=unique(be);
        ii=setdiff(1:np,bn);
    end
    
    % Compute edge lengths & forces
    evec=p(e(:,1),:)-p(e(:,2),:);
    Le=sqrt(sum(evec.^2,2));
    if (strcmpi(fhn,'hgeom'))
        He=feval(fh,(p(e(:,1),:)+p(e(:,2),:))/2,fd,pmax,varargin{:});
    else
        He=feval(fh,(p(e(:,1),:)+p(e(:,2),:))/2,varargin{:});
    end
    L0=He*Fscale*sqrt(sum(Le.^2)/sum(He.^2));
    L=Le./L0;
    
    % split too long edges
    if (iter>np)
        il=find(L>1.5);
        if (~isempty(il))
            p=[p; (p(e(il,1),:)+p(e(il,2),:))/2];
            tolp=1; tolr=1.2;
            continue
        end
    end
    
    F=(1-L.^4).*exp(-L.^4)./L;
    Fvec=F*[1,1].*evec;
    
    % Assemble edge forces on nodes
    Fe=full(sparse(e(:,[1,1,2,2]),ones(size(F))*[1,2,1,2],[Fvec,-Fvec],np,2));
    Fe(1:npf,:)=0;
    if (iter>mp*np), Fe(ifix,:)=0; end
    
    % Move nodes
    p(1:size(p0,1),:)=p(1:size(p0,1),:)+dt*Fe;
    p0=p;
    
    % Project external nodes onto the boundary
    while any(~isinterior(pgon, p(:,1), p(:,2)))
        [X,Y] = pgon.boundary;
        
        I = find(~isinterior(pgon, p(:,1), p(:,2)));
        if isempty(I)
            break;
        end
        p_out = p(I,:);
        d = BoundarySegment(X, Y, p_out);
        
        xy1 = [X(d),Y(d)];
        xy2 = [X(d+1), Y(d+1)];
        m = (xy1(:,2)-xy2(:,2))./(xy1(:,1)-xy2(:,1));
        b = xy1(:,2)-m.*xy1(:,1);
        x = p_out(:,1);
        y = p_out(:,2);
        perpSlope = -1./m;
        yInt = -perpSlope .* x + y;
        xIntersection = (yInt - b) ./ (m - perpSlope);
        yIntersection = perpSlope .* xIntersection + yInt;
        p_out = [xIntersection, yIntersection];
        for i=1:length(I)
            if xy1(i,1)==0 && xy2(i,1)==0 % Symm Vert Bnd
                p_out(i,:) = [0, p(I(i),2)];
                if p_out(i,2)<0, p_out(i,2)=0; end
                if p_out(i,2)>10, p_out(i,2)=10; end
            elseif xy1(i,2)==0 && xy2(i,2)==0 % Symm Horz Bnd
                p_out(i,:) = [p(I(i),1), 0];
                if p_out(i,1)<0, p_out(i,1)=0; end
                if p_out(i,1)>10, p_out(i,1)=10; end
            elseif xy1(i,2)==10 && xy2(i,2)==10 % Ess Horz Bnd
                p_out(i,:) = [p(I(i),1), 10];
                if p_out(i,1)<0, p_out(i,1)=0; end
                if p_out(i,1)>10, p_out(i,1)=10; end
            elseif xy1(i,1)==10 && xy2(i,1)==10 % Ess Vert Bnd
                p_out(i,:) = [10, p(I(i),2)];
                if p_out(i,2)<0, p_out(i,2)=0; end
                if p_out(i,2)>10, p_out(i,2)=10; end
            end
        end
        p(I,:) = p_out;
    end
    
    % Stopping criteria
    dp=dt*sqrt(sum(Fe.^2,2));
    tolp=max(dp(ii))/h0;
    tolr=max(sqrt(sum((p(1:size(p0,1),:)-p0).^2,2))/h0);
    
    % Check the nodes speed if iter>mp*np
    if (iter>mp*np), ifix=find(sqrt(sum(dp.^2,2))<dt*epsp); end
    
    % check the triangle orientation & quality if tolp<epsp
    if tolp<epsp
        [ar,qt,~]=tarea(p,t); [qtmin,itmin]=min(qt);
        if (min(ar)<0 || qtmin < qmin)
            tolp=1; tolr=1.2; ifix=[];
            if (qtmin < qmin)
                it=t(itmin,:);
                it=setdiff(it,union((1:npf)',ifix)); pt=p(it,:);
                p(it,:)=[];
                if (length(it)==3)
                    p=[p; (pt(1,:)+pt(2,:)+pt(3,:))/3];
                elseif (length(it)==2)
                    p=[p; (pt(1,:)+pt(2,:))/2];
                end
            end
        end
    end
end

[ar,~] = tarea(p,t); t(ar<1e-5,:)=[];
[e,ib] = kmg2dedg(t);
be = e(ib,:);
end