function [p,t,be,bn] = mesh(fd,fh,h0,bbox,pfix,varargin)
%KMG2D 2D-mesh generator using signed distance & size functions.
%--------------------------------------------------------------------
% (c) 2009, Koko J., ISIMA, koko@isima.fr
% Koko J., A Matlab mesh generator for the two-dimensional finite element method,
% Applied Mathematics and Computation 250, p. 650-664 (2015)
%--------------------------------------------------------------------
% Scale factor and time step
Fscale=1.2; dt=.1;
% Convergence tolerances
epsp=.005; epsr=.5; deps=sqrt(eps)*h0; geps=.001*h0;
iterMax=5000; mp=5;
% Minimum triangle quality
qmin=.5;

% Initialize the fix points set
npf=size(pfix,1);

% Initial grid
[x,y]=meshgrid(bbox(1,1):h0:bbox(2,1),bbox(1,2):h0*sqrt(3)/2:bbox(2,2));
x(2:2:end,:)=x(2:2:end,:)+h0/2;                      
p=[x(:),y(:)];   
clear x y

% Remove points outside the region, apply the rejection method
p=p(feval(fd,p,varargin{:})<geps,:); 
% p=p(feval(fd,p,{[],WoundBnd,s,geo})<geps,:); 

r0=feval(fh,p,varargin{:});
%r0=feval(fh,p,{[],WoundBnd,s,geo});

r0=1./r0.^2;                    
p=[pfix; p(rand(size(p,1),1)<r0./max(r0),:)];
clear r0

% Remove nodes outside the domain
np0=size(p,1); 
dp=feval(fd,p,varargin{:});
% dp=feval(fd,p,{[],WoundBnd,s,geo});
ii=find(dp<geps); q=p(ii,:);

% Add fixed nodes
if (npf>0), p=setdiff(q,pfix,'rows'); p=[pfix; p];
else, p=q; end
clear q dp

% Initial distribution
close
plot(p(:,1),p(:,2),'.')
fprintf('\n Initial number of nodes : %5d \n\n',size(p,1))
pause(3)

% Polygon for projecting nodes on boundary
pgon = polyshape(pfix);

iter=0; ifix=[];  tolp=1; tolr=10^2;
while (iter<iterMax && tolp>epsp)
  iter=iter+1;

  % Delaunay triangluation
  if (tolr>epsr)
    np=size(p,1);
    p0=p;
    t=delaunayn(p);
    % Reject triangles with centroid outside the domain
    pm=(p(t(:,1),:)+p(t(:,2),:)+p(t(:,3),:))/3;
    t=t(feval(fd,pm,varargin{:})<-geps,:);
%     t=t(feval(fd,pm,{[],WoundBnd,s,geo})<-geps,:);
    % Reorder (locally) triangle vertices counter clockwise
    ar=tarea(p,t); it=find(ar<0);
    itt=t(it,2); t(it,2)=t(it,3); t(it,3)=itt;
    % Form all edges without duplication & extract boundary nodes 
    [e,ib]=kmg2dedg(t);
    be=e(ib,:);
    bn=unique(be);
    ii=setdiff(1:np,bn);
    % Graphical output of the current mesh
    triplot(t,p(:,1),p(:,2)), axis equal, axis off, drawnow, hold off
  end
  
  % Compute edge lengths & forces
  evec=p(e(:,1),:)-p(e(:,2),:);
  Le=sqrt(sum(evec.^2,2));
  He=feval(fh,(p(e(:,1),:)+p(e(:,2),:))/2,varargin{:});
  %      He=feval(fh,(p(e(:,1),:)+p(e(:,2),:))/2,{[],WoundBnd,s,geo});
  L0=He*Fscale*sqrt(sum(Le.^2)/sum(He.^2));
  L=Le./L0;
  
  % split too long edges
  if (iter>np)
      il=find(L>1.5); 
      if (~isempty(il))
          p=[p; (p(e(il,1),:)+p(e(il,2),:))/2];
          tolp=1; tolr=1.2; 
          fprintf('Number of edges split : %3d \n',length(il))
          continue
      end
  end
  
  F=(1-L.^4).*exp(-L.^4)./L;
  Fvec=F*[1,1].*evec;
 
  % Assemble edge forces on nodes
  Fe=full(sparse(e(:,[1,1,2,2]),ones(size(F))*[1,2,1,2],[Fvec,-Fvec],np,2));
  Fe(1:npf,:)=0;
  if (iter>mp*np), Fe(ifix,:)=0; end
 
  % Move nodes
  p=p+dt*Fe;

  % Project external nodes onto the boundary
  while any(~isinterior(pgon, p(:,1), p(:,2)))
      [X,Y] = pgon.boundary;

      I = find(~isinterior(pgon, p(:,1), p(:,2)));
      if isempty(I)
          break;
      end
      p_out = p(I,:);
      d = BoundarySegment(X, Y, p_out);

      xy1 = [X(d),Y(d)];
      xy2 = [X(d+1), Y(d+1)];
      m = (xy1(:,2)-xy2(:,2))./(xy1(:,1)-xy2(:,1));
      b = xy1(:,2)-m.*xy1(:,1);
      x = p_out(:,1);
      y = p_out(:,2);
      perpSlope = -1./m;
      yInt = -perpSlope .* x + y;
      xIntersection = (yInt - b) ./ (m - perpSlope);
      yIntersection = perpSlope .* xIntersection + yInt;
      p_out = [xIntersection, yIntersection];
      for i=1:length(I)
          if xy1(i,1)==0 && xy2(i,1)==0 %VertBnd
              p_out(i,:) = [0, p(I(i),2)];
          elseif xy1(i,2)==0 && xy2(i,2)==0 %HorzBnd
              p_out(i,:) = [p(I(i),1), 0];
          elseif xy1(i,2)==10 && xy2(i,2)==10 %EssBnd Horz
              p_out(i,:) = [p(I(i),1), 10];
          elseif xy1(i,1)==10 && xy2(i,1)==10 %EssBnd Vert
              p_out(i,:) = [10, p(I(i),2)];
          end
      end
      p(I,:) = p_out;
  end
  
  % Stopping criteria
  dp=dt*sqrt(sum(Fe.^2,2));
  tolp=max(dp(ii))/h0;
  tolr=max(sqrt(sum((p-p0).^2,2))/h0);
  
  % Check the nodes speed if iter>mp*np
  if (iter>mp*np), ifix=find(sqrt(sum(dp.^2,2))<dt*epsp); end
  
  % check the triangle orientation & quality if tolp<epsp 
  if (tolp<epsp)
     [ar,qt,~]=tarea(p,t); [qtmin,itmin]=min(qt);
     if (min(ar)<0 || qtmin < qmin)
         tolp=1; tolr=1.2; ifix=[];
         if (qtmin < qmin)
            it=t(itmin,:); 
            it=setdiff(it,union([1:npf]',ifix)); pt=p(it,:);
            p(it,:)=[];
            if (length(it)==3)
                p=[p; (pt(1,:)+pt(2,:)+pt(3,:))/3];
            elseif (length(it)==2)
                p=[p; (pt(1,:)+pt(2,:))/2];
            end
         end
     end
     if (min(qt)<qmin)
        fprintf('Low quality triangle qt=%12.6f \n',min(qt))
     end
  end
end

% Plot the final mesh
close
triplot(t,p(:,1),p(:,2)), axis equal, axis off, drawnow, hold off
fprintf('\nNumber of nodes ---------: %5d \n',size(p,1))
fprintf('Number of triangles--------: %5d \n',size(t,1))
fprintf('Triangle quality measure---> %5.3f \n',min(qt))
fprintf('Number of iterations-------: %4d \n',iter)
end
 

%
%---- Triangles area & quality --------------------------------------------
%
function [ar,qt,te]=tarea(p,t)
%Compute triangle area and quality
%
it1=t(:,1); it2=t(:,2); it3=t(:,3);
x21=p(it2,1)-p(it1,1); y21=p(it2,2)-p(it1,2); 
x31=p(it3,1)-p(it1,1); y31=p(it3,2)-p(it1,2);
x32=p(it3,1)-p(it2,1); y32=p(it3,2)-p(it2,2);
ar=(x21.*y31-y21.*x31)/2;
if (nargout==1), return, end
a1=sqrt(x21.^2+y21.^2); a2=sqrt(x31.^2+y31.^2); a3=sqrt(x32.^2+y32.^2);
qt=(a2+a3-a1).*(a3+a1-a2).*(a1+a2-a3)./(a1.*a2.*a3);
if (nargout == 2) , return, end
te=[a1,a2,a3];
end
