clc; clear; close all
LEFT = 1; RIGHT = 2; LEFT_RIGHT = 3;
VISCOUS = 1; HYSTERETIC = 0;
DampType = VISCOUS;

%% Material properties
EI = 2*6.044e6;
m = 2*60;
DampRail = 0.0001;
% Sleepers
Width = 0.3;
Spacing = 0.6;
Ms = 315;
Js = Ms/6*Width^2;
% Rail pads
Kv = 100e6;
Kt = Kv*Width^2/4;
DampRP = 0.001;
% Under sleeper pads
DampUSP = 0.001;
Ku = 45e9;
Kuh = Ku*1e2*0;
%% Ballast and soil - stiffness properties for full track (2m cross section)
Ballast.Damp = 0.001;
Ballast.D = 0.03;
Ballast.H = 0.3;
Ballast.Rho = 1800;
Ballast.Kn = 1.4222e+07;
Ballast.Ks = 1.7778e+06;
Ballast.Kd = 6.2222e+06;

Soil.Cs = 140*[1;1];
Soil.Cp = 262*[1;1];
Soil.Rho = 18952*[1;1];
Soil.Damp = 1.5859e-3*[1;1];
Soil.H = 0.6*[1;1];
Soil.KvBot = 1.4348e8;
Soil.KhBot = 3.0572e9;
Soil.CvBot = 2.0328e4;
Soil.ChBot = 3.8462e4;
nLayer = length(Soil.H);
Soil.Kn = zeros(nLayer,1);
Soil.Ks = zeros(nLayer,1);
Soil.Kd = zeros(nLayer,1);
Soil.M = zeros(nLayer,1);
Soil.N = zeros(nLayer,1);
for iLayer = 1: 1: nLayer
    Cs = Soil.Cs(iLayer);
    Cp = Soil.Cp(iLayer);
    Rho = Soil.Rho(iLayer);
    G = Cs^2*Rho;
    Lambda = Cp^2*Rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    Soil.Kn(iLayer) = B(1);
    Soil.Ks(iLayer) = B(2);
    Soil.Kd(iLayer) = (Soil.Kn(iLayer) - Soil.Ks(iLayer)) / 2;
    Soil.M(iLayer) = Rho*Ballast.D^2;
end

% Ballast and soil
Dlat = Ballast.D;
Hlat = [Ballast.H; Soil.H];
Rho = [Ballast.Rho; Soil.Rho];
Knlat = [Ballast.Kn;Soil.Kn];
Kslat = [Ballast.Ks;Soil.Ks];
Kdlat = [Ballast.Kd;Soil.Kd];
Damp = [Ballast.Damp;Soil.Damp];
Mlat = Rho*Dlat^2;
Knlat = Knlat.*(1+1i*Damp);
Kslat = Kslat.*(1+1i*Damp);
Kdlat = Kdlat.*(1+1i*Damp);
Nlat = round(Hlat/Dlat);
Hlat = Dlat*Nlat;
if ~isempty(Soil.KvBot)
    Knlat = [Knlat;(Soil.KvBot+1i*Soil.CvBot)*Dlat];
    Kslat = [Kslat;(Soil.KhBot+1i*Soil.ChBot)*Dlat];
    Kdlat = [Kdlat;0];
    Mlat = [Mlat;0];
    Nlat = [Nlat;0];
end

%% Calculate geometrical quantities
N = round(Width/Dlat)+1;
M = round(Spacing/Dlat); M = M - N;
Width = (N-1)*Dlat;
Spacing = (M+N)*Dlat;
% Under-sleeper pads divided by each grain of ballast
Ku  = Ku/N;
Kuh = Kuh/N;
%% Frequency of interest, loading locations and output locations
freq = 50;
w = 2*pi*freq;
% Damped constants
EI = EI*(1+1i*DampRail);
Kv = Kv*(1+1i*DampRP);
Kt = Kt*(1+1i*DampRP);
Ku = Ku*(1+1i*DampUSP);
Kuh = Kuh*(1+1i*DampUSP);

%% Calculate transmitting boundary
start = clock;
if DampType == VISCOUS
    DF = w;
else
    DF = 2;
end
OUTPUT_FEM = TransmittingBoundaryFEM(EI,m,Kv,Kt,Ms,Js,...
                                     Ku,Kuh,Dlat,Mlat,...
                                     Knlat,Kslat,Kdlat,Nlat,N,M,w,DampType);
fprintf(1, ['Time spent with FEM TB: ' num2str(etime(clock,start)) ' s.\n']);
start = clock;
OUTPUT = TransmittingBoundary(real(EI)+1i*DF*imag(EI),m,...
                              real(Kv)+1i*DF*imag(Kv),...
                              real(Kt)+1i*DF*imag(Kt),Ms,Js,...
                              real(Ku)+1i*DF*imag(Ku),...
                              real(Kuh)+1i*DF*imag(Kuh),Dlat,Mlat,...
                              real(Knlat)+1i*DF*imag(Knlat),...
                              real(Kslat)+1i*DF*imag(Kslat),Nlat,N,M,...
                              w,LEFT_RIGHT,1e-12,100);
fprintf(1, ['Time spent with analytical TB: ' num2str(etime(clock,start)) ' s.\n']);

%% Relative errors
fprintf(1, '\n');
fprintf(1, 'Errors: column, Kright, Kleft, Fright, Fleft:\n');
OUTPUT_FEM.FlexRight = inv(OUTPUT_FEM.KRight);
OUTPUT_FEM.FlexLeft = inv(OUTPUT_FEM.KLeft);
for iCol = 1: 1: length(OUTPUT.KRight)
    if OUTPUT.KRight(iCol,iCol) == inf
        fErrorR = 0;
        fErrorR_F = 0;
    else
        fErrorR = max(abs(OUTPUT.KRight(:,iCol)-OUTPUT_FEM.KRight(:,iCol))./(abs(OUTPUT_FEM.KRight(:,iCol))));
        fErrorR_F = max(abs(OUTPUT.FlexRight(:,iCol)-OUTPUT_FEM.FlexRight(:,iCol))./(abs(OUTPUT_FEM.FlexRight(:,iCol))));
    end
    if OUTPUT.KLeft(iCol,iCol) == inf
        fErrorL = 0;
        fErrorL_F = 0;
    else
        fErrorL = max(abs(OUTPUT.KLeft(:,iCol)-OUTPUT_FEM.KLeft(:,iCol))./abs(OUTPUT_FEM.KLeft(:,iCol)));
        fErrorL_F = max(abs(OUTPUT.FlexLeft(:,iCol)-OUTPUT_FEM.FlexLeft(:,iCol))./abs(OUTPUT_FEM.FlexLeft(:,iCol)));
    end
    fprintf(1, '   Col %d:    %e,    %e,    %e,    %e\n', iCol, fErrorR, fErrorL, fErrorR_F, fErrorL_F);
end

%% Relative errors - with respect to maximum of column
fprintf(1, '\n');
fprintf(1, 'Errors: column, Kright, Kleft, Fright, Fleft:\n');
for iCol = 1: 1: length(OUTPUT.KRight)
    if OUTPUT.KRight(iCol,iCol) == inf
        fErrorR = 0;
        fErrorR_F = 0;
    else
        fErrorR = max(abs(OUTPUT.KRight(:,iCol)-OUTPUT_FEM.KRight(:,iCol)))/max(abs(OUTPUT_FEM.KRight(:,iCol)));
        fErrorR_F = max(abs(OUTPUT.FlexRight(:,iCol)-OUTPUT_FEM.FlexRight(:,iCol)))/max(abs(OUTPUT_FEM.FlexRight(:,iCol)));
    end
    if OUTPUT.KLeft(iCol,iCol) == inf
        fErrorL = 0;
        fErrorL_F = 0;
    else
        fErrorL = max(abs(OUTPUT.KLeft(:,iCol)-OUTPUT_FEM.KLeft(:,iCol)))/max(abs(OUTPUT_FEM.KLeft(:,iCol)));
        fErrorL_F = max(abs(OUTPUT.FlexLeft(:,iCol)-OUTPUT_FEM.FlexLeft(:,iCol)))/max(abs(OUTPUT_FEM.FlexLeft(:,iCol)));
    end
    fprintf(1, '   Col %d:    %e,    %e,    %e,    %e\n', iCol, fErrorR, fErrorL, fErrorR_F, fErrorL_F);
end
