function OUTPUT = MovingLoadTransition(Inputs,dt,Tmax,V,W,ADJUST,dtOut,PLOT)
if nargin < 5, W = []; end
if nargin < 6, ADJUST = 0; end
if nargin < 7, dtOut = dt; end
if nargin < 8, PLOT = 0; end
OUTPUT = [];
clc
close all
%% Constants
LEFT = 1; RIGHT = 2; LEFT_RIGHT = 3;
VISCOUS = 1;
DampType = VISCOUS;

%% Track properties
Track = Inputs.Track;
DampEI = Track.DampRail;
EI = Track.EI; EI = EI*(1+1i*DampEI);
m = Track.m;
a = Track.a;
% Sleepers
Width = Track.Width;
Spacing = Track.Spacing;
Ms = Track.Ms;
Js = Track.Js;
% Rail pads
DampRP = Track.DampRP;
Kv = Track.Kv; Kv = Kv*(1+1i*DampRP);
Kt = Track.Kt; Kt = Kt*(1+1i*DampRP);
% Under sleeper pads
DampUSP = Track.DampUSP;
Ku = Track.Ku; Ku = Ku*(1+1i*DampUSP);
Kuh = Track.Kuh; Kuh = Kuh*(1+1i*DampUSP);

%% Ballast properties
Ballast = Inputs.Ballast;
Dampb = Ballast.Damp;
D = Ballast.D;
Hb = Ballast.H;
Rhob = Ballast.Rho;
Knb = Ballast.Kn*(1+1i*Dampb);
Ksb = Ballast.Ks*(1+1i*Dampb);
Kdb = Ballast.Kd*(1+1i*Dampb);
Mb  = Rhob*D^2*a;
Nb = round(Hb/D);
%% Foundation properties - ballast and soil - Left side
Soil = Inputs.Soil;
DampLeft = Soil.Left.Damp;
HLeft = Soil.Left.H;
RhoLeft = Soil.Left.Rho;
CsLeft = Soil.Left.Cs;
CpLeft = Soil.Left.Cp;
KvBotLeft = Soil.Left.KvBot;
KhBotLeft = Soil.Left.KhBot;
CvBotLeft = Soil.Left.CvBot;
ChBotLeft = Soil.Left.ChBot;
% Generate left lattice vectors
nLayerLeft = length(HLeft);
NLeft = zeros(nLayerLeft,1);
MLeft = zeros(nLayerLeft,1);
KnLeft = zeros(nLayerLeft,1);
KsLeft = zeros(nLayerLeft,1);
KdLeft = zeros(nLayerLeft,1);
for iLayer = 1: 1: nLayerLeft
    Cs = CsLeft(iLayer);
    Cp = CpLeft(iLayer);
    Rho = RhoLeft(iLayer);
    G = Cs^2*Rho;
    Lambda = Cp^2*Rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    KnLeft(iLayer) = B(1);
    KsLeft(iLayer) = B(2);
    KdLeft(iLayer) = (KnLeft(iLayer) - KsLeft(iLayer)) / 2;
    MLeft(iLayer) = Rho*D^2;
    NLeft(iLayer) = round(HLeft(iLayer)/D);
end
KnLeft = KnLeft.*(1+1i*DampLeft);
KsLeft = KsLeft.*(1+1i*DampLeft);
KdLeft = KdLeft.*(1+1i*DampLeft);
% Add ballast layer
KnLeft = [Knb;KnLeft];
KsLeft = [Ksb;KsLeft];
KdLeft = [Kdb;KdLeft];
MLeft = [Mb;MLeft];
NLeft = [Nb;NLeft];
% Add bottom layer
if ~isempty(KvBotLeft)
    KnLeft = [KnLeft;(KvBotLeft+1i*CvBotLeft)*D];
    KsLeft = [KsLeft;(KhBotLeft+1i*ChBotLeft)*D];
    KdLeft = [KdLeft;0];
    MLeft = [MLeft;0];
    NLeft = [NLeft;0];
end

%% Foundation properties - ballast and soil - Right side
DampRight = Soil.Right.Damp;
HRight = Soil.Right.H;
RhoRight = Soil.Right.Rho;
CsRight = Soil.Right.Cs;
CpRight = Soil.Right.Cp;
KvBotRight = Soil.Right.KvBot;
KhBotRight = Soil.Right.KhBot;
CvBotRight = Soil.Right.CvBot;
ChBotRight = Soil.Right.ChBot;
% Generate left lattice vectors
nLayerRight = length(HRight);
NRight = zeros(nLayerRight,1);
MRight = zeros(nLayerRight,1);
KnRight = zeros(nLayerRight,1);
KsRight = zeros(nLayerRight,1);
KdRight = zeros(nLayerRight,1);
for iLayer = 1: 1: nLayerRight
    Cs = CsRight(iLayer);
    Cp = CpRight(iLayer);
    Rho = RhoRight(iLayer);
    G = Cs^2*Rho;
    Lambda = Cp^2*Rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    KnRight(iLayer) = B(1);
    KsRight(iLayer) = B(2);
    KdRight(iLayer) = (KnRight(iLayer) - KsRight(iLayer)) / 2;
    MRight(iLayer) = Rho*D^2;
    NRight(iLayer) = round(HRight(iLayer)/D);
end
KnRight = KnRight.*(1+1i*DampRight);
KsRight = KsRight.*(1+1i*DampRight);
KdRight = KdRight.*(1+1i*DampRight);
% Add ballast layer
KnRight = [Knb;KnRight];
KsRight = [Ksb;KsRight];
KdRight = [Kdb;KdRight];
MRight = [Mb;MRight];
NRight = [Nb;NRight];
% Add bottom layer
if ~isempty(KvBotRight)
    KnRight = [KnRight;(KvBotRight+1i*CvBotRight)*D];
    KsRight = [KsRight;(KhBotRight+1i*ChBotRight)*D];
    KdRight = [KdRight;0];
    MRight = [MRight;0];
    NRight = [NRight;0];
end
%% Foundation properties - ballast and soil - Middle
DampMid = Soil.Mid.Damp;
HMid = Soil.Mid.H;
RhoMid = Soil.Mid.Rho;
CsMid = Soil.Mid.Cs;
CpMid = Soil.Mid.Cp;
KvBotMid = Soil.Mid.KvBot;
KhBotMid = Soil.Mid.KhBot;
CvBotMid = Soil.Mid.CvBot;
ChBotMid = Soil.Mid.ChBot;
% Generate left lattice vectors
nLayerMid = length(HMid);
NMid = zeros(nLayerMid,1);
MMid = zeros(nLayerMid,1);
KnMid = zeros(nLayerMid,1);
KsMid = zeros(nLayerMid,1);
KdMid = zeros(nLayerMid,1);
for iLayer = 1: 1: nLayerMid
    Cs = CsMid(iLayer);
    Cp = CpMid(iLayer);
    Rho = RhoMid(iLayer);
    if Cp == inf || Cs == inf
        KnMid(iLayer) = inf;
        KsMid(iLayer) = inf;
        KdMid(iLayer) = inf;
    else
        G = Cs^2*Rho;
        Lambda = Cp^2*Rho - 2*G;
        Mat = [1/2 -3/2; 1/2 1/2];
        B = Mat^-1*[Lambda;G];
        KnMid(iLayer) = B(1);
        KsMid(iLayer) = B(2);
        KdMid(iLayer) = (KnMid(iLayer) - KsMid(iLayer)) / 2;
    end
    MMid(iLayer) = Rho*D^2;
    NMid(iLayer) = round(HMid(iLayer)/D);
end
KnMid = KnMid.*(1+1i*DampMid);
KsMid = KsMid.*(1+1i*DampMid);
KdMid = KdMid.*(1+1i*DampMid);
% Add ballast layer
KnMid = [Knb;KnMid];
KsMid = [Ksb;KsMid];
KdMid = [Kdb;KdMid];
MMid = [Mb;MMid];
NMid = [Nb;NMid];
% Add bottom layer
if ~isempty(KvBotMid)
    KnMid = [KnMid;(KvBotMid+1i*CvBotMid)*D];
    KsMid = [KsMid;(KhBotMid+1i*ChBotMid)*D];
    KdMid = [KdMid;0];
    MMid = [MMid;0];
    NMid = [NMid;0];
end

%% Check compatibility of foundations
if any(NLeft ~= NRight) || any(NLeft ~= NMid) || length(HLeft) ~= length(HRight) || length(HLeft) ~= length(HMid)
    fprintf(1, '\n');
    fprintf(1, 'ERROR: left and right foundations must have the same layering depths.\n');
end

Knlat = KnLeft;
Kslat = KsLeft;
Kdlat = KdLeft;
Mlat = MLeft;
Nlat = NLeft;
if any(Knlat~=KnRight) || any(Kslat~=KsRight) || any(Kdlat~=KdRight) || any(Mlat~=MRight)
    Knlat = [Knlat KnRight];
    Kslat = [Kslat KsRight];
    Kdlat = [Kdlat KdRight];
    Mlat  = [Mlat  MRight ];
end
%% Mid sections
MidSections = Soil.MidSections;

%% Approach Slabs
ApproachSlab = Inputs.ApproachSlab;

%% Geometrcal approximations
N = round(Width/D)+1;
M = round(Spacing/D); M = M - N;
Ku  = Ku/N;
Kuh = Kuh/N;
%% Inputs
if Kuh == inf, Kuh = 100*Ku; end

%% Calculate transmitting boundaries and incident displacements for left side
[Fleft,Fright,Tdm,OUTPUT_Freq] = DuhamelMatrices(...
                            EI,m,Kv,Kt,Ms,Js,Ku,Kuh,...
                            D,Mlat(:,1),Knlat(:,1),Kslat(:,1),Kdlat(:,1),Nlat(:,1),...
                            N,M,DampType,dt,Tmax,Soil.Left.FilenameDM,W);
[Tinc,Uleft,Uright] = IncidentDisplacements(...
                            EI,m,Kv,Kt,Ms,Js,Ku,Kuh,...
                            D,Mlat(:,1),Knlat(:,1),Kslat(:,1),Kdlat(:,1),Nlat(:,1),...
                            N,M,DampType,dt,Tmax,V,Soil.Left.FilenameID,W,OUTPUT_Freq);
clear OUTPUT_Freq;
TincL = Tinc;
%% Calculate transmitting boundaries and incident displacements for right side
if size(Knlat,2) > 1
    Mlat(1,2) = Mlat(1,1);
    Knlat(1,2) = Knlat(1,1);
    Kslat(1,2) = Kslat(1,1);
    Nlat(1,2) = Nlat(1,1);
    [Fleft2,Fright2,Tdm2,OUTPUT_Freq] = DuhamelMatrices(...
                                EI,m,Kv,Kt,Ms,Js,Ku,Kuh,...
                                D,Mlat(:,2),Knlat(:,2),Kslat(:,2),Kdlat(:,2),Nlat(:,2),...
                                N,M,DampType,dt,Tmax,Soil.Right.FilenameDM,W);
    [Tinc2, Uleft2,  Uright2] = IncidentDisplacements(...
                                EI,m,Kv,Kt,Ms,Js,Ku,Kuh,...
                                D,Mlat(:,2),Knlat(:,2),Kslat(:,2),Kdlat(:,2),Nlat(:,2),...
                                N,M,DampType,dt,Tmax,V,Soil.Right.FilenameID,W,OUTPUT_Freq);
    clear OUTPUT_Freq;
    Fleft = Fleft2;
    Uright = Uright2;
else
    Mlat  = [Mlat  Mlat];
    Knlat = [Knlat Knlat];
    Kslat = [Kslat Kslat];
    Kdlat = [Kdlat Kdlat];
    Nlat  = [Nlat  Nlat];
end
Mlat  = [Mlat  MMid];
Knlat = [Knlat KnMid];
Kslat = [Kslat KsMid];
Kdlat = [Kdlat KdMid];
Nlat  = [Nlat  NMid];
TincR = Tinc + (N+M)*D*sum(MidSections(:,1))/V;

%% Assemble FEM matrices for mid section
nNodePerColLat = sum(Nlat(:,1)) + 1;
nNodePerCell = nNodePerColLat*(N+M) + 2*(N+M);
nNodeTotal = nNodePerCell * sum(MidSections(:,1));
NODE_TYPE = zeros(nNodeTotal,5); % [TYPE X Y Mass J]
RAIL = 1; SLEEPER = 2; BALLAST = 3; SOIL = 4; FIXED = 5; SLAB = 6;
ad2Connection = zeros(10000, 9); % [RAIL Node1 Node2 EI CEI m] or [SPRING Node1 Node2 Dofs1 Dofs2 K C]
kCon = 0;
BEAM = 1;
SPRING = 2;
kCell = 0;
knlatPrev = Knlat(:,1);
kslatPrev = Kslat(:,1);
kdlatPrev = (knlatPrev-kslatPrev)/2;
fprintf(1, '\n');
fprintf(1, 'Assembling element matrices for %d cells...\n', sum(MidSections(:,1)));
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;
FIXED_BALLAST_NODES = [];
for iCell = 1: 1: size(MidSections,1)
    mlat = zeros(size(Mlat,1),1);
    knlat = mlat;
    kslat = mlat;
    kdlat = mlat;
    % Left
    if MidSections(iCell,2) ~= 0
        mlat =  mlat  + Mlat(:,1) *MidSections(iCell,2);
        knlat = knlat + Knlat(:,1)*MidSections(iCell,2);
        kslat = kslat + Kslat(:,1)*MidSections(iCell,2);
        kdlat = kdlat + Kdlat(:,1)*MidSections(iCell,2);
    end
    % Mid
    if MidSections(iCell,3) ~= 0
        mlat =  mlat  + Mlat(:,3) *MidSections(iCell,3);
        knlat = knlat + Knlat(:,3)*MidSections(iCell,3);
        kslat = kslat + Kslat(:,3)*MidSections(iCell,3);
        kdlat = kdlat + Kdlat(:,3)*MidSections(iCell,3);
    end
    % Right
    if MidSections(iCell,4) ~= 0
        mlat =  mlat  + Mlat(:,2) *MidSections(iCell,4);
        knlat = knlat + Knlat(:,2)*MidSections(iCell,4);
        kslat = kslat + Kslat(:,2)*MidSections(iCell,4);
        kdlat = kdlat + Kdlat(:,2)*MidSections(iCell,4);
    end
    for jCell = 1: 1: MidSections(iCell,1)
        if mod(kCell+1,round(sum(MidSections(:,1)/50)))==0
            star = star + 1;
            if star <= 50 && STARS(star) == 0
                fprintf(1, '*');
                STARS(star) = 1;
            end
        end
        kNode_RR = kCell*nNodePerCell + 1;
        % Column by column assemble the elements
        SurfaceNodes = zeros(1,N+M);
        for iN = 0: 1: N+M-2
            if kCon + 1000 > size(ad2Connection,1)
                ad2Connection = [ad2Connection; zeros(10000,size(ad2Connection,2))];
            end
            %% Define rail nodes, ballast nodes and sleeper node
            kNode_RS = [];
            kNode_RL = kNode_RR;
            kNode_RM = kNode_RL + nNodePerColLat + 1;
            kNode_RR = kNode_RM + 1;
            kNode_BL = kNode_RL + 1;
            kNode_BR = kNode_RR + 1;
            if mod(N,2)==1 && iN == (N-1)/2
                kNode_RS = kNode_RL;
                kNode_S = kNode_RL + 1;
                kNode_RM = kNode_RM + 1;
                kNode_RR = kNode_RR + 1;
                kNode_BL = kNode_BL + 1;
                kNode_BR = kNode_BR + 1;
            elseif mod(N,2)==1 && iN == (N-1)/2-1
                kNode_BR = kNode_BR + 1;
            elseif mod(N,2)==0 && iN == N/2-1
                kNode_RS = kNode_RM;
                kNode_S = kNode_RM + 1;
                kNode_RR = kNode_RR + 1;
                kNode_BR = kNode_BR + 1;
            end
            if iN == 0
                kNode_RLS = kNode_RL;
                kNode_BLS = kNode_BL;
            end
            SurfaceNodes(iN+1) = kNode_BL;
            if iN == N+M-2, SurfaceNodes(iN+2) = kNode_BR; end
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],1) = RAIL;
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],2) = (kCell*(N+M)+iN-1+(1:0.5:2))*D;
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],3) = N*D;
            NODE_TYPE([kNode_BL:(kNode_RM-1) kNode_BR:(kNode_BR+nNodePerColLat-1)],1) = BALLAST;
            NODE_TYPE(kNode_BL:(kNode_RM-1),2) = (kCell*(N+M)+iN)*D;
            NODE_TYPE(kNode_BL:(kNode_RM-1),3) = -(0:1:(nNodePerColLat-1))*D;
            NODE_TYPE(kNode_BR:(kNode_BR+nNodePerColLat-1),2) = (kCell*(N+M)+iN+1)*D;
            NODE_TYPE(kNode_BR:(kNode_BR+nNodePerColLat-1),3) = -(0:1:(nNodePerColLat-1))*D;
            if ~isempty(kNode_RS)
                NODE_TYPE(kNode_S,1) = SLEEPER;
                NODE_TYPE(kNode_S,2) = NODE_TYPE(kNode_RS,2);
                NODE_TYPE(kNode_S,3) = N*D/2;
                NODE_TYPE(kNode_S,4) = Ms;
                NODE_TYPE(kNode_S,5) = Js;
            end
                
            %% Assemble rails
            kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RL kNode_RM real(EI) imag(EI) m];
            kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RM kNode_RR real(EI) imag(EI) m];
            
            %% Assemble rail pads
            if ~isempty(kNode_RS)
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_RS kNode_S [1 0 -1  0] real(Kv) imag(Kv)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_RS kNode_S [0 1  0 -1] real(Kt) imag(Kt)];
            end
            
            %% Assemble ballast springs
            kDepth = 0;
            z = 0;
            for iLayer = 1: 1: size(Nlat,1)
            for iDiv = 1: 1: Nlat(iLayer,1)
                % Depth bellow ballast
                if  iLayer == 1, z = -D; end
                z = z + D;
                mm = mlat(iLayer);
                kn = knlat(iLayer);
                ks = kslat(iLayer);
                kd = kdlat(iLayer);
                % Check if properties must be corrected (due to presence of
                %                                           approach slabs)
                IS_SLAB = 0;
                for iSlab = 1: 1: length(ApproachSlab)
                    if ~(kCell+1 >= ApproachSlab(iSlab).Cells(1) && ...
                            kCell+1 <= ApproachSlab(iSlab).Cells(2))
                        continue;
                    end
                    if ~(z >= ApproachSlab(iSlab).Depth && ...
                        z+D <= ApproachSlab(iSlab).Depth+ApproachSlab(iSlab).Thick)
                        continue;
                    end
                    mm = ApproachSlab(iSlab).Rho*D^2*a;
                    E = ApproachSlab(iSlab).E;
                    G = E/2.4;
                    Lambda = 2*G/3;
                    Mat = [1/2 -3/2; 1/2 1/2];
                    B = Mat^-1*[Lambda;G];
                    kn = B(1)*a;
                    ks = B(2)*a;
                    kd = (kn-ks)/2;
                    IS_SLAB = 1;
                    % Add supports of approach slab if it is the first or
                    % last cell
                    if (kCell+1 == ApproachSlab(iSlab).Cells(1) && iN == 0)
                        Krot = ApproachSlab(iSlab).RotSpringStart;
                        Kvert = ApproachSlab(iSlab).VertSpringStart;
                        if real(Krot)>0 && real(Kvert)>0
                            FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BL+kDepth+(0:1)'];
                        elseif real(Kvert)>0
                            if (z+2*D > ApproachSlab(iSlab).Depth+ApproachSlab(iSlab).Thick)
                                FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BL+kDepth+1];
                            end
                        end
                    end
                    if (kCell+1 == ApproachSlab(iSlab).Cells(2) && iN == N+M-2)
                        Krot = ApproachSlab(iSlab).RotSpringEnd;
                        Kvert = ApproachSlab(iSlab).VertSpringEnd;
                        if real(Krot)>0 && real(Kvert)>0
                            FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BR+kDepth+(0:1)'];
                        elseif real(Kvert)>0
                            if (z+2*D > ApproachSlab(iSlab).Depth+ApproachSlab(iSlab).Thick)
                                FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BR+kDepth+1];
                            end
                        end
                    end
                end
                % Add mass to the nodes
                NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],4) = ...
                    NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],4) + 0.25*mm;
                if iLayer > 1
                    NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],1) = SOIL;
                end
                if IS_SLAB
                    NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],1) = SLAB;
                end
                if iN == 0
                    NODE_TYPE(kNode_BL+kDepth+(0:1),4) = ...
                        NODE_TYPE(kNode_BL+kDepth+(0:1),4) + 0.25*mm;
                elseif iN == N+M-2
                    NODE_TYPE(kNode_BR+kDepth+(0:1),4) = ...
                        NODE_TYPE(kNode_BR+kDepth+(0:1),4) + 0.25*mm;
                end
                if isinf(real(kn)) || isinf(real(ks)) || isinf(real(kd))
                    FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BL+kDepth+(0:1)';kNode_BR+kDepth+(0:1)'];
                end
                % Add springs Left
                if iN == 0
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [0 1 0 -1] real(kn) imag(kn)];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [1 0 -1 0] real(ks) imag(ks)];
                else
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [0 1 0 -1] 0.5*real(kn) 0.5*imag(kn)];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [1 0 -1 0] 0.5*real(ks) 0.5*imag(ks)];
                end
                % Add springs right
                if iN == N+M-2
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [0 1 0 -1] real(kn) imag(kn)];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [1 0 -1 0] real(ks) imag(ks)];
                else
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [0 1 0 -1] 0.5*real(kn) 0.5*imag(kn)];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [1 0 -1 0] 0.5*real(ks) 0.5*imag(ks)];
                end
                % Add top and bottom springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth [1 0 -1 0] 0.5*real(kn) 0.5*imag(kn)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth [0 1 0 -1] 0.5*real(ks) 0.5*imag(ks)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth+1 [1 0 -1 0] 0.5*real(kn) 0.5*imag(kn)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth+1 [0 1 0 -1] 0.5*real(ks) 0.5*imag(ks)];
                % Add diagonal springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(kd) imag(kd)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth sqrt(2)/2*[-1 -1 1 1] real(kd) imag(kd)];
                kDepth = kDepth + 1;
            end
            if Nlat(iLayer,1) == 0
                if isinf(real(knlat(iLayer))) || isinf(real(kslat(iLayer))) || isinf(real(kdlat(iLayer)))
                    FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES;kNode_BL+kDepth;kNode_BR+kDepth];
                end
                if iLayer > 1
                    NODE_TYPE([kNode_BL+kDepth;kNode_BR+kDepth],1) = SOIL;
                end
                % Add top and bottom springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                if iN == 0
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                elseif iN == N+M-2
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                end
            end
            end
        end
        % Add springs between sleeper and ballast
        SurfaceNodes = SurfaceNodes(1:N);
        for iNode = 1: 1: N
            kNode = SurfaceNodes(iNode);
            % Vertical connection
            dist = (-(N-1)/2+iNode-1)*D;
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode kNode_S [0 -1 1 dist] real(Ku) imag(Ku)];
            % Horizontal connection
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode -1 [1 0 0 0] real(Kuh) imag(Kuh)];
        end
        kCell = kCell + 1;
        if kCell == 1
            % save this
            knlatPrev = knlat;
            kslatPrev = kslat;
            kdlatPrev = kdlat;
            kNode_BR_Prev = kNode_BR;
            kNode_RR_Prev = kNode_RR;
            continue;
        end
        % Assemble connection with Previous cell
        kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RR_Prev kNode_RLS real(EI) imag(EI) m];
        kDepth = 0;
        knn = (knlat+knlatPrev)/2;
        kss = (kslat+kslatPrev)/2;
        kdd = (kdlat+kdlatPrev)/2;
        for iLayer = 1: 1: size(Nlat,1)
            if isinf(real(knlat(iLayer))), knn(iLayer) = knlatPrev(iLayer); end
            if isinf(real(knlatPrev(iLayer))), knn(iLayer) = knlat(iLayer); end
            if isinf(real(kslat(iLayer))), kss(iLayer) = kslatPrev(iLayer); end
            if isinf(real(kslatPrev(iLayer))), kss(iLayer) = kslat(iLayer); end
            if isinf(real(kdlat(iLayer))), kdd(iLayer) = kdlatPrev(iLayer); end
            if isinf(real(kdlatPrev(iLayer))), kdd(iLayer) = kdlat(iLayer); end
        end
        for iLayer = 1: 1: size(Nlat,1)
        for iDiv = 1: 1: Nlat(iLayer,1)
            kn = knn(iLayer);
            ks = kss(iLayer);
            kd = kdd(iLayer);
            
            % Depth bellow ballast
            if  iLayer == 1, z = -D; end
            z = z + D;
            % Check if properties must be corrected (due to presence of
            %                                           approach slabs)
            for iSlab = 1: 1: length(ApproachSlab)
                if ~(kCell+1 >= ApproachSlab(iSlab).Cells(1) && ...
                        kCell <= ApproachSlab(iSlab).Cells(2))
                    continue;
                end
                if ~(z >= ApproachSlab(iSlab).Depth && ...
                    z+D <= ApproachSlab(iSlab).Depth+ApproachSlab(iSlab).Thick)
                    continue;
                end
                if (kCell+1 > ApproachSlab(iSlab).Cells(1) && ...
                        kCell+1 <= ApproachSlab(iSlab).Cells(2))
                    E = ApproachSlab(iSlab).E;
                    G = E/2.4;
                    Lambda = 2*G/3;
                    Mat = [1/2 -3/2; 1/2 1/2];
                    B = Mat^-1*[Lambda;G];
                    kn = B(1)*a;
                    ks = B(2)*a;
                    kd = (kn-ks)/2;
                end
                
                MAKE_IT_ZERO = 0;
                if isinf(real(knlat(iLayer))), MAKE_IT_ZERO = 1; end
                if isinf(real(knlatPrev(iLayer))), MAKE_IT_ZERO = 1; end
                if isinf(real(kslat(iLayer))), MAKE_IT_ZERO = 1; end
                if isinf(real(kslatPrev(iLayer))), MAKE_IT_ZERO = 1; end
                if isinf(real(kdlat(iLayer))), MAKE_IT_ZERO = 1; end
                if isinf(real(kdlatPrev(iLayer))), MAKE_IT_ZERO = 1; end
                if MAKE_IT_ZERO
                    kn = 0;
                    ks = 0;
                    kd = 0;
                end
            end
            
            % Add top and bottom springs
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth [1 0 -1 0] 0.5*real(kn) 0.5*imag(kn)];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth [0 1 0 -1] 0.5*real(ks) 0.5*imag(ks)];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth+1 [1 0 -1 0] 0.5*real(kn) 0.5*imag(kn)];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth+1 [0 1 0 -1] 0.5*real(ks) 0.5*imag(ks)];
            % Add diagonal springs
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(kd) imag(kd)];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth sqrt(2)/2*[-1 -1 1 1] real(kd) imag(kd)];
            kDepth = kDepth + 1;
        end
        end
        
        % save this
        knlatPrev = knlat;
        kslatPrev = kslat;
        kdlatPrev = kdlat;
        kNode_BR_Prev = kNode_BR;
        kNode_RR_Prev = kNode_RR;
    end
end
%% Add nodes at end and beginning of model, where transmitting boundaries
%  will be coupled
nNodeBeg = 1+nNodePerColLat;
NODE_BEG = zeros(nNodeBeg,5);
NODE_BEG(1,1:3) = [RAIL -D N*D];
NODE_BEG(2:end,1) = BALLAST;
NODE_BEG(2:end,2) = -D;
NODE_BEG(2:end,3) = -(0:1:(nNodePerColLat-1))*D;

NODE_END = NODE_BEG;
NODE_END(:,2) = NODE_TYPE(end,2) + D;

NODE_TYPE = [NODE_BEG;NODE_TYPE;NODE_END];
FIXED_BALLAST_NODES = FIXED_BALLAST_NODES + nNodeBeg;

ad2Connection(ad2Connection(:,2)>0,2) = ad2Connection(ad2Connection(:,2)>0,2) + nNodeBeg;
ad2Connection(ad2Connection(:,3)>0,3) = ad2Connection(ad2Connection(:,3)>0,3) + nNodeBeg;
%% Connect middle region to left boundary
kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM 1 nNodeBeg+1 real(EI) imag(EI) m];
kDepth = 0;
for iLayer = 1: 1: size(Nlat,1)
for iDiv = 1: 1: Nlat(iLayer,1)
    if isinf(real(Knlat(iLayer,1))) || isinf(real(Kslat(iLayer,1))) || isinf(real(Kdlat(iLayer,1)))
        FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES; 2+kDepth; 2+kDepth+1; nNodeBeg+2+kDepth; nNodeBeg+2+kDepth+1];
    end
    if iLayer > 1
        NODE_TYPE([2+kDepth; 2+kDepth+1; nNodeBeg+2+kDepth; nNodeBeg+2+kDepth+1],1) = SOIL;
    end
    % Add top and bottom springs
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth nNodeBeg+2+kDepth [1 0 -1 0] 0.5*real(Knlat(iLayer,1)) 0.5*imag(Knlat(iLayer,1))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth nNodeBeg+2+kDepth [0 1 0 -1] 0.5*real(Kslat(iLayer,1)) 0.5*imag(Kslat(iLayer,1))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth+1 nNodeBeg+2+kDepth+1 [1 0 -1 0] 0.5*real(Knlat(iLayer,1)) 0.5*imag(Knlat(iLayer,1))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth+1 nNodeBeg+2+kDepth+1 [0 1 0 -1] 0.5*real(Kslat(iLayer,1)) 0.5*imag(Kslat(iLayer,1))];
    % Add diagonal springs
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth nNodeBeg+2+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(Kdlat(iLayer,1)) imag(Kdlat(iLayer,1))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING 2+kDepth+1 nNodeBeg+2+kDepth sqrt(2)/2*[-1 -1 1 1] real(Kdlat(iLayer,1)) imag(Kdlat(iLayer,1))];
    kDepth = kDepth + 1;
end
end
%% Connect middle region to right boundary
nNode = size(NODE_TYPE,1);
kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM nNode-2*nNodeBeg+1 nNode-nNodeBeg+1 real(EI) imag(EI) m];
kDepth = 0;
kNodeLeft = nNode-2*nNodeBeg+2;
kNodeRight = nNode-nNodeBeg+2;
for iLayer = 1: 1: size(Nlat,1)
for iDiv = 1: 1: Nlat(iLayer,1)
    if isinf(real(Knlat(iLayer,2))) || isinf(real(Kslat(iLayer,2))) || isinf(real(Kdlat(iLayer,2)))
        FIXED_BALLAST_NODES = [FIXED_BALLAST_NODES; kNodeLeft+kDepth; kNodeLeft+kDepth; kNodeRight+kDepth; kNodeRight+kDepth+1];
    end
    if iLayer > 1
        NODE_TYPE([kNodeLeft+kDepth; kNodeLeft+kDepth; kNodeRight+kDepth; kNodeRight+kDepth+1],1) = SOIL;
    end
    % Add top and bottom springs
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth kNodeRight+kDepth [1 0 -1 0] 0.5*real(Knlat(iLayer,2)) 0.5*imag(Knlat(iLayer,2))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth kNodeRight+kDepth [0 1 0 -1] 0.5*real(Kslat(iLayer,2)) 0.5*imag(Kslat(iLayer,2))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth+1 kNodeRight+kDepth+1 [1 0 -1 0] 0.5*real(Knlat(iLayer,2)) 0.5*imag(Knlat(iLayer,2))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth+1 kNodeRight+kDepth+1 [0 1 0 -1] 0.5*real(Kslat(iLayer,2)) 0.5*imag(Kslat(iLayer,2))];
    % Add diagonal springs
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth kNodeRight+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(Kdlat(iLayer,2)) imag(Kdlat(iLayer,2))];
    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNodeLeft+kDepth+1 kNodeRight+kDepth sqrt(2)/2*[-1 -1 1 1] real(Kdlat(iLayer,2)) imag(Kdlat(iLayer,2))];
    kDepth = kDepth + 1;
end
end
FIXED_BALLAST_NODES = unique(sort(FIXED_BALLAST_NODES,'ascend'));
%% Split the rail connections which are longer than D/2 into two,
%  for coherence purposes
nCon = kCon;
for iCon = 1: 1: nCon
    if ad2Connection(iCon,1) ~= BEAM, continue; end
    Node1 = ad2Connection(iCon,2);
    Node2 = ad2Connection(iCon,3);
    dX = NODE_TYPE(Node2,2) - NODE_TYPE(Node1,2);
    if abs(dX) < D/2*(1+1e-5), continue; end
    NODE_TYPE = [NODE_TYPE(1:Node2-1,:); (NODE_TYPE(Node2,:)+NODE_TYPE(Node1,:))/2; NODE_TYPE(Node2:end,:)];
    Node3 = Node2;
    ad2Connection(ad2Connection(:,2)>=Node3,2) = ad2Connection(ad2Connection(:,2)>=Node3,2) + 1;
    ad2Connection(ad2Connection(:,3)>=Node3,3) = ad2Connection(ad2Connection(:,3)>=Node3,3) + 1;
    FIXED_BALLAST_NODES(FIXED_BALLAST_NODES>=Node3) = FIXED_BALLAST_NODES(FIXED_BALLAST_NODES>=Node3) + 1;
    kCon = kCon + 1;  ad2Connection(kCon,1:6) = [BEAM Node3 Node3+1 real(EI) imag(EI) m];
    ad2Connection(iCon,3) = Node3;
end
ad2Connection = ad2Connection(1:kCon,:);
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');
NODE_TYPE(FIXED_BALLAST_NODES,1) = FIXED;
%% Shift the x coordinate such that x=0 corresponds first Sleeper
SLEEPER_NODES = find(NODE_TYPE(:,1)==SLEEPER);
NODE_TYPE(:,2) = NODE_TYPE(:,2) - min(NODE_TYPE(SLEEPER_NODES,2));

%% Make sure that there is one node at each loaded position (moving load)
if ADJUST
    fprintf(1, '\n');
    fprintf(1, 'Adjunting rail discretization to time steps...\n');
    fprintf(1, '[0%%                                            100%%]\n');
    fprintf(1, '[');
    STARS = zeros(50,1); star = 0;
    RAIL_NODES = find(NODE_TYPE(:,1) == RAIL);
    X_Rail = NODE_TYPE(RAIL_NODES,2);
    xMin = min(X_Rail);
    xMax = max(X_Rail);
    T = -Tmax:dt:((N+M)*D*sum(MidSections(:,1))/V+Tmax);
    X = V*T;
    for iX = 1: 1: length(X)
        if mod(iX+1,round(length(X)/50))==0
            star = star + 1;
            if star <= 50 && STARS(star) == 0
                fprintf(1, '*');
                STARS(star) = 1;
            end
        end
        x = X(iX);
        if x<= xMin || x >= xMax, continue; end
        % Add node
        kNodeLeft = find((NODE_TYPE(:,1)==RAIL).*(NODE_TYPE(:,2)<=x));
        kNodeLeft = kNodeLeft(end);
        if abs(x-NODE_TYPE(kNodeLeft,2)) < V*dt/1000, continue; end
        kNodeRight = find((NODE_TYPE(:,1)==RAIL).*(NODE_TYPE(:,2)>x));
        kNodeRight = kNodeRight(1);
        if abs(x-NODE_TYPE(kNodeRight,2)) < V*dt/1000, continue; end

        % Find the element with these nodes
        jCon = find((ad2Connection(:,1)==BEAM).*...
                    (ad2Connection(:,2)==kNodeLeft).*...
                    (ad2Connection(:,3)==kNodeRight));
        if isempty(jCon), continue; end

        % Add node and dhift all nodes by greater than NodeRight by 1
        NODE_TYPE = [NODE_TYPE(1:kNodeRight-1,:);
                     RAIL x NODE_TYPE(kNodeLeft,3:end);
                     NODE_TYPE(kNodeRight:end,:)];
        ad2Connection(ad2Connection(:,2)>=kNodeRight,2) = ad2Connection(ad2Connection(:,2)>=kNodeRight,2) + 1;
        ad2Connection(ad2Connection(:,3)>=kNodeRight,3) = ad2Connection(ad2Connection(:,3)>=kNodeRight,3) + 1;
        FIXED_BALLAST_NODES(FIXED_BALLAST_NODES>=kNodeRight) = FIXED_BALLAST_NODES(FIXED_BALLAST_NODES>=kNodeRight) + 1;
        % Add new connection
        ad2Connection(jCon,3) = kNodeRight;
        kCon = kCon + 1;  ad2Connection(kCon,1:6) = [BEAM kNodeRight kNodeRight+1 real(EI) imag(EI) m];
    end
    for iStar = star+1:1:50
        fprintf(1, '*');
    end
    fprintf(1, ']\n');
end
ad2Connection = ad2Connection(1:kCon,:);

%% Plot modeled section
if PLOT
    figure; hold on
    plot(NODE_TYPE(NODE_TYPE(:,1)==BALLAST,2),NODE_TYPE(NODE_TYPE(:,1)==BALLAST,3),'ob');
    plot(NODE_TYPE(NODE_TYPE(:,1)==SOIL,2),NODE_TYPE(NODE_TYPE(:,1)==SOIL,3),'om');
    plot(NODE_TYPE(NODE_TYPE(:,1)==FIXED,2),NODE_TYPE(NODE_TYPE(:,1)==FIXED,3),'.k');
    plot(NODE_TYPE(NODE_TYPE(:,1)==SLEEPER,2),NODE_TYPE(NODE_TYPE(:,1)==SLEEPER,3),'or');
    plot(NODE_TYPE(NODE_TYPE(:,1)==RAIL,2),NODE_TYPE(NODE_TYPE(:,1)==RAIL,3),'ok');
    for iCon = 1: 1: size(ad2Connection,1)
        Nodes = ad2Connection(iCon,2:3);
        if Nodes(1) == -1, Nodes(1) = [];
        elseif Nodes(2) == -1, Nodes(2) = []; end
        if length(Nodes) == 1
            COLOR = 'vk';
            plot(NODE_TYPE(Nodes,2),NODE_TYPE(Nodes,3),COLOR);
        elseif ad2Connection(iCon,1) == SPRING
            COLOR = '.-g';
%             if any(NODE_TYPE(Nodes)==SLEEPER) || any(NODE_TYPE(Nodes)==RAIL)
                plot(NODE_TYPE(Nodes,2),NODE_TYPE(Nodes,3),COLOR);
%             end
        elseif ad2Connection(iCon,1) == BEAM
            COLOR = '.-r';
            plot(NODE_TYPE(Nodes,2),NODE_TYPE(Nodes,3),COLOR);
        end
    end
end
%% Assemble stuctural elements
nNode = size(NODE_TYPE,1);
nElem = size(ad2Connection,1);
nDof = 2*nNode;
ad2M = zeros(16*nElem,3); NZT_M = 0;
ad2K = zeros(16*nElem,3); NZT_K = 0;
ad2C = zeros(16*nElem,3); NZT_C = 0;
% Assemble lumped masses
for iNode = 1:1:nNode
    if NODE_TYPE(iNode,1) == RAIL, continue; end
    if NODE_TYPE(iNode,1) == SLEEPER
        ad2M(NZT_M+(1:2),:) = [NODE_TYPE(iNode,4) 2*iNode-1 2*iNode-1;
                               NODE_TYPE(iNode,5) 2*iNode-0 2*iNode-0];
        NZT_M = NZT_M + 2;
    end
    if NODE_TYPE(iNode,1) == BALLAST || NODE_TYPE(iNode,1) == SOIL
        ad2M(NZT_M+(1:2),:) = [NODE_TYPE(iNode,4) 2*iNode-1 2*iNode-1;
                               NODE_TYPE(iNode,4) 2*iNode-0 2*iNode-0];
        NZT_M = NZT_M + 2;
    end
end
% Assemble elastic elements - springs and beams
for iElem = 1: 1: nElem
    kNode1 = ad2Connection(iElem,2);
    kNode2 = ad2Connection(iElem,3);
    ROW = [2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2];
    COL = [2*kNode1-1; 2*kNode1-1; 2*kNode1-1; 2*kNode1-1 ;
           2*kNode1  ; 2*kNode1  ; 2*kNode1  ; 2*kNode1   ;
           2*kNode2-1; 2*kNode2-1; 2*kNode2-1; 2*kNode2-1 ;
           2*kNode2  ; 2*kNode2  ; 2*kNode2  ; 2*kNode2  ];
    if ad2Connection(iElem,1) == BEAM
        EI = ad2Connection(iElem,4);
        CEI = ad2Connection(iElem,5);
        m = real(ad2Connection(iElem,6));
        l = NODE_TYPE(kNode2,2)-NODE_TYPE(kNode1,2);
        l2 = l*l;
        l3 = l2*l;
        
        Kelem = EI*[ 12/l3  6/l2  -12/l3  6/l2 ;
                      6/l2   4/l   -6/l2   2/l ;
                    -12/l3 -6/l2   12/l3 -6/l2 ;
                      6/l2   2/l   -6/l2   4/l ];
                  
        Celem =CEI*[ 12/l3  6/l2  -12/l3  6/l2 ;
                      6/l2   4/l   -6/l2   2/l ;
                    -12/l3 -6/l2   12/l3 -6/l2 ;
                      6/l2   2/l   -6/l2   4/l ];
                  
        Melem = m*l/420*[  156  22*l    54 -13*l ;
                          22*l  4*l2  13*l -3*l2 ;
                            54  13*l   156 -22*l ;
                         -13*l -3*l2 -22*l  4*l2];
        ad2K(NZT_K+(1:16),:) = [Kelem(:) ROW COL]; NZT_K = NZT_K + 16;
        ad2C(NZT_C+(1:16),:) = [Celem(:) ROW COL]; NZT_C = NZT_C + 16;
        ad2M(NZT_M+(1:16),:) = [Melem(:) ROW COL]; NZT_M = NZT_M + 16;
    elseif ad2Connection(iElem,1) == SPRING
        K = ad2Connection(iElem,8);
        C = ad2Connection(iElem,9);
        dofs = ad2Connection(iElem,4:7);
        if kNode1 == -1
            dofs = dofs(3:4);
            ROW = [2*kNode2-1; 2*kNode2;
                   2*kNode2-1; 2*kNode2];
            COL = [2*kNode2-1; 2*kNode2-1;
                   2*kNode2  ; 2*kNode2 ];
        elseif kNode2 == -1
            dofs = dofs(1:2);
            ROW = [2*kNode1-1; 2*kNode1;
                   2*kNode1-1; 2*kNode1];
            COL = [2*kNode1-1; 2*kNode1-1;
                   2*kNode1  ; 2*kNode1 ];
        end
        NZT = length(dofs)^2;
        Kelem = K*(dofs'*dofs);
        Celem = C*(dofs'*dofs);
        ad2K(NZT_K+(1:NZT),:) = [Kelem(:) ROW COL]; NZT_K = NZT_K + NZT;
        ad2C(NZT_C+(1:NZT),:) = [Celem(:) ROW COL]; NZT_C = NZT_C + NZT;
    end
end
ad2K = ad2K(1:NZT_K,:); ad2K = sparse(ad2K(:,2),ad2K(:,3),ad2K(:,1),nDof,nDof);
ad2C = ad2C(1:NZT_C,:); ad2C = sparse(ad2C(:,2),ad2C(:,3),ad2C(:,1),nDof,nDof);
ad2M = ad2M(1:NZT_M,:); ad2M = sparse(ad2M(:,2),ad2M(:,3),ad2M(:,1),nDof,nDof);
% Free dofs - remove last row of ballast nodes if bottom of lattice is
% fixed
ai1FreeDof = 1:1:nDof;
ai1FixedDof = sort([FIXED_BALLAST_NODES*2-1;
                    FIXED_BALLAST_NODES*2-0],'ascend');
if Nlat(end,1) ~= 0
    ai1FixedNode = find(((NODE_TYPE(:,1)==BALLAST)+(NODE_TYPE(:,1)==SOIL)).*(NODE_TYPE(:,3)==min(NODE_TYPE(:,3))));
    ai1FixedDof = [ai1FixedDof; ai1FixedNode*2-1; ai1FixedNode*2];
    ai1FixedDof = unique(sort(ai1FixedDof(:),'ascend'));
end
ai1FreeDof(ai1FixedDof) = [];
if PLOT
    ai1FixedNode = ai1FixedDof(2:2:end)/2;
    plot(NODE_TYPE(ai1FixedNode,2),NODE_TYPE(ai1FixedNode,3),'*m');
end
%% Newmark look
gamma = 0.5;
beta  = 0.25;
a0 = 1/(beta*dt^2);
a1 = gamma/(beta*dt);
a2 = 1/(beta*dt);
a3 = 1/(2*beta)-1;
a4 = gamma/beta-1;
a5 = dt*(gamma/(2*beta)-1);

% Time sample
kT = 0;
T = -Tmax:dt:((N+M)*D*sum(MidSections(:,1))/V+Tmax);
nOut = max(1,round(dtOut/dt));
Tout = T(1:nOut:end);
nT = length(T);
up = []; vp = [];
fprintf(1, '\n');
fprintf(1, 'Newmark loop on time steps (%d time intervals)\n', nT);
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;
xMin = min(NODE_TYPE(:,2));
xMax = max(NODE_TYPE(:,2));
kNodeRailLeft = find((NODE_TYPE(:,2)==xMin).*(NODE_TYPE(:,1)==RAIL));
kNodeRailRight = find((NODE_TYPE(:,2)==xMax).*(NODE_TYPE(:,1)==RAIL));
for t = T(1):dt:T(end)
    if mod(kT+1,round(nT/50))==0
        star = star + 1;
        if star <= 50 && STARS(star) == 0
            fprintf(1, '*');
            STARS(star) = 1;
        end
    end
    %% Calculate system matrices
    if kT == 0
        ad2Keq = ad2K + a1*ad2C + a0*ad2M;
        % Add Stiffness matrix at boundaries...
        % At left side - use transmitting boundaries of right
        dofs_left = 1:1:size(Fright,1);
        F0right = Fright(:,:,1);
        K0right = inv(F0right);
        ad2Keq(dofs_left,dofs_left) = ad2Keq(dofs_left,dofs_left) + K0right;
        % At right side - use transmitting boundaries of left
        dofs_right = (size(ad2Keq,1)-size(Fleft,1)+1):1:size(ad2Keq,1);
        if ~isempty(find(ai1FixedDof==size(ad2Keq,1))) || ...
                            ~isempty(find(ai1FixedDof==size(ad2Keq,1)-1))
            dofs_right = dofs_right-2;
        end
        F0left = Fleft(:,:,1);
        K0left = inv(F0left);
        ad2Keq(dofs_right,dofs_right) = ad2Keq(dofs_right,dofs_right) + K0left;
        if PLOT
            NodeLeft = dofs_left(2:2:end)/2;
            NodeRight = dofs_right(2:2:end)/2;
            plot(NODE_TYPE(NodeLeft,2),NODE_TYPE(NodeLeft,3),'*c');
            plot(NODE_TYPE(NodeRight,2),NODE_TYPE(NodeRight,3),'*c');
        end

        % Factoeize Keq
        ad2L = chol(ad2Keq(ai1FreeDof,ai1FreeDof), 'lower');
        ad2U = ad2L';
        % Interaction forces
        FintL = zeros(size(F0right,1),length(T));
        FintR = zeros(size(F0left ,1),length(T));
        % Previous interation valuess
        up = zeros(nDof,1);
        vp = zeros(nDof,1);
        ap = zeros(nDof,1);
        kTout = 0;
        ad2Displ = zeros(nDof,length(Tout));
        ad2Vel = zeros(nDof,length(Tout));
        ad2Accel = zeros(nDof,length(Tout));
        
        % Reshape Fright and Fleft to help in the convolutions
        Fright = reshape(Fright(:,:,2:end),size(Fright,1),size(Fright,2)*(size(Fright,3)-1));
        Fleft =  reshape(Fleft(:,:,2:end) ,size(Fleft,1) ,size(Fleft,2 )*(size(Fleft,3) -1));
    end
    kT = kT + 1;
    %% Calculate external force
    Fext = zeros(nDof,1);
    x = V*t;
    if x >= xMin && x <= xMax
        kNode1 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)<=x)); kNode1 = kNode1(end);
        kNode2 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)>x)); kNode2 = kNode2(1);
        x1 = NODE_TYPE(kNode1,2); x2 = NODE_TYPE(kNode2,2);
        l = x2-x1;
        Fact = (x-x1)/l;
        N1 = 1 -            3*Fact^2 + 2*Fact^3;
        N2 =     l*Fact - 2*l*Fact^2 + l*Fact^3;
        N3 =                3*Fact^2 - 2*Fact^3;
        N4 =               -l*Fact^2 + l*Fact^3;
        if kNode1 == kNodeRailLeft
            N1 = 0;
            N2 = 0;
            N3 = 0;
            N4 = 0;
        end
        if kNode2 == kNodeRailRight
            N1 = 0;
            N2 = 0;
            N3 = 0;
            N4 = 0;
        end
        Fext(2*kNode1-1) = N1;
        Fext(2*kNode1) = N2;
        Fext(2*kNode2-1) = N3;
        Fext(2*kNode2) = N4;
    end
    %% Add terms due to previous state
    Fext = Fext + ad2M*(a0*up+a2*vp+a3*ap) + ad2C*(a1*up+a4*vp+a5*ap);
    %% Calculate incident displacements on each side
    % Left side
    if t >= TincL(1) && t < TincL(end)
        kTimeIndex = find(TincL<=t); kTimeIndex = kTimeIndex(end);
        Fact2 = (t-TincL(kTimeIndex))/(TincL(kTimeIndex+1)-TincL(kTimeIndex));
        Fact1 = 1-Fact2;
        UincL = Uleft(1:length(dofs_left),kTimeIndex)*Fact1 + ...
                        Uleft(1:length(dofs_left),kTimeIndex+1)*Fact2;
    else
        UincL = zeros(length(dofs_left),1);
    end
    % Right side
    if t >= TincR(1) && t < TincR(end)
        kTimeIndex = find(TincR<=t); kTimeIndex = kTimeIndex(end);
        Fact2 = (t-TincR(kTimeIndex))/(TincR(kTimeIndex+1)-TincR(kTimeIndex));
        Fact1 = 1-Fact2;
        UincR = Uright(1:length(dofs_right),kTimeIndex)*Fact1 + ...
                        Uright(1:length(dofs_right),kTimeIndex+1)*Fact2;
    else
        UincR = zeros(length(dofs_right),1);
    end
    %% Calculate force history term on each side
    jTend = min(kT-1,length(Tdm)-1);
    
    FintLeft  = reshape(FintL(:,kT-(1:1:jTend)),size(FintL,1)*jTend,1);
    FintLeft  = [FintLeft;  zeros(size(Fright,2)-length(FintLeft),1)];
    UincL = UincL - Fright*FintLeft;
    
    FintRight = reshape(FintR(:,kT-(1:1:jTend)),size(FintR,1)*jTend,1);
    FintRight = [FintRight; zeros(size(Fleft,2)-length(FintRight),1)];
    UincR = UincR - Fleft*FintRight;
%     for jT = 1:1:min(kT-1,length(Tdm)-1)
%         UincL = UincL - Fright(:,:,jT) * FintL(:,kT-jT);
%         UincR = UincR - Fleft(:,:,jT)  * FintR(:,kT-jT);
%     end
    FhistL = K0right * UincL;
    FhistR = K0left  * UincR;
    Fext(dofs_left)  = Fext(dofs_left)  + FhistL;
    Fext(dofs_right) = Fext(dofs_right) + FhistR;
    %% Calculate new dispalcements, velocities and accelerations
    uc = zeros(nDof,1);
    un = ad2U\(ad2L\Fext(ai1FreeDof));
    uc(ai1FreeDof) = un;
    vc = a1*(uc-up) - a4*vp - a5*ap;
    ac = a0*(uc-up) - a2*vp - a3*ap;
    %% Calculate interaction forces and store them
    FintL(:,kT) = FhistL - K0right*uc(dofs_left);
    FintR(:,kT) = FhistR - K0left *uc(dofs_right);
    
    %% store old values
    up = uc;
    vp = vc;
    ap = ac;
    if mod(kT-1,nOut) == 0
        kTout = kTout + 1;
        ad2Displ(:,kTout) = uc;
        ad2Vel(:,kTout) = vc;
        ad2Accel(:,kTout) = ac;
    end
end
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');
T = Tout;
%% Store outputs:
% Time
OUTPUT.T = T;
% Mesh
OUTPUT.SLEEPER = SLEEPER;
OUTPUT.RAIL = RAIL;
OUTPUT.BALLAST = BALLAST;
OUTPUT.SOIL = SOIL;
OUTPUT.FIXED = FIXED;
OUTPUT.SLAB = SLAB;
OUTPUT.NODE_TYPE = NODE_TYPE;
OUTPUT.ad2Connection = ad2Connection;
OUTPUT.BEAM = BEAM;
OUTPUT.SPRING= SPRING;
% Displacements, velocities and accelerations
OUTPUT.ad2Displ = ad2Displ;
OUTPUT.ad2Vel = ad2Vel;
OUTPUT.ad2Accel = ad2Accel;
% Interaction forces
OUTPUT.FintL = FintL;
OUTPUT.FintR = FintR;
OUTPUT.dofs_left = dofs_left;
OUTPUT.dofs_right = dofs_right;

end