function OUTPUT = IncidentDisplacement_TD(Inputs,dt,Tmax,V,nCell,ADJUST)
if nargin < 6, ADJUST = 0; end

%% Track properties
Track = Inputs.Track;
DampEI = Track.DampRail;
EI = Track.EI; EI = EI*(1+1i*DampEI);
m = Track.m;
a = Track.a;
% Sleepers
Width = Track.Width;
Spacing = Track.Spacing;
Ms = Track.Ms;
Js = Track.Js;
% Rail pads
DampRP = Track.DampRP;
Kv = Track.Kv; Kv = Kv*(1+1i*DampRP);
Kt = Track.Kt; Kt = Kt*(1+1i*DampRP);
% Under sleeper pads
DampUSP = Track.DampUSP;
Ku = Track.Ku; Ku = Ku*(1+1i*DampUSP);
Kuh = Track.Kuh; Kuh = Kuh*(1+1i*DampUSP);

%% Ballast properties
Ballast = Inputs.Ballast;
Dampb = Ballast.Damp;
D = Ballast.D;
Hb = Ballast.H;
Rhob = Ballast.Rho;
Knb = Ballast.Kn*(1+1i*Dampb);
Ksb = Ballast.Ks*(1+1i*Dampb);
Kdb = Ballast.Kd*(1+1i*Dampb);
Mb  = Rhob*D^2*a;
Nb = round(Hb/D);
%% Foundation properties - ballast and soil - Left side
Soil = Inputs.Soil;
DampLeft = Soil.Left.Damp;
HLeft = Soil.Left.H;
RhoLeft = Soil.Left.Rho;
CsLeft = Soil.Left.Cs;
CpLeft = Soil.Left.Cp;
KvBotLeft = Soil.Left.KvBot;
KhBotLeft = Soil.Left.KhBot;
CvBotLeft = Soil.Left.CvBot;
ChBotLeft = Soil.Left.ChBot;
% Generate left lattice vectors
nLayerLeft = length(HLeft);
NLeft = zeros(nLayerLeft,1);
MLeft = zeros(nLayerLeft,1);
KnLeft = zeros(nLayerLeft,1);
KsLeft = zeros(nLayerLeft,1);
KdLeft = zeros(nLayerLeft,1);
for iLayer = 1: 1: nLayerLeft
    Cs = CsLeft(iLayer);
    Cp = CpLeft(iLayer);
    Rho = RhoLeft(iLayer);
    G = Cs^2*Rho;
    Lambda = Cp^2*Rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    KnLeft(iLayer) = B(1);
    KsLeft(iLayer) = B(2);
    KdLeft(iLayer) = (KnLeft(iLayer) - KsLeft(iLayer)) / 2;
    MLeft(iLayer) = Rho*D^2;
    NLeft(iLayer) = round(HLeft(iLayer)/D);
end
KnLeft = KnLeft.*(1+1i*DampLeft);
KsLeft = KsLeft.*(1+1i*DampLeft);
KdLeft = KdLeft.*(1+1i*DampLeft);
% Add ballast layer
KnLeft = [Knb;KnLeft];
KsLeft = [Ksb;KsLeft];
KdLeft = [Kdb;KdLeft];
MLeft = [Mb;MLeft];
NLeft = [Nb;NLeft];
% Add bottom layer
if ~isempty(KvBotLeft)
    KnLeft = [KnLeft;(KvBotLeft+1i*CvBotLeft)*D];
    KsLeft = [KsLeft;(KhBotLeft+1i*ChBotLeft)*D];
    KdLeft = [KdLeft;0];
    MLeft = [MLeft;0];
    NLeft = [NLeft;0];
%     KnLeft = [KnLeft;(KvBotLeft+1i*CvBotLeft)*D];
%     KsLeft = [KsLeft;(KhBotLeft+1i*ChBotLeft)*D];
%     KdLeft = [KdLeft;((KvBotLeft+1i*CvBotLeft)-(KhBotLeft+1i*ChBotLeft))*D/2];
%     MLeft = [MLeft;MLeft(end)];
%     NLeft = [NLeft;1];
end

%% Foundation properties - ballast and soil - Right side
DampRight = Soil.Right.Damp;
HRight = Soil.Right.H;
RhoRight = Soil.Right.Rho;
CsRight = Soil.Right.Cs;
CpRight = Soil.Right.Cp;
KvBotRight = Soil.Right.KvBot;
KhBotRight = Soil.Right.KhBot;
CvBotRight = Soil.Right.CvBot;
ChBotRight = Soil.Right.ChBot;
% Generate left lattice vectors
nLayerRight = length(HRight);
NRight = zeros(nLayerRight,1);
MRight = zeros(nLayerRight,1);
KnRight = zeros(nLayerRight,1);
KsRight = zeros(nLayerRight,1);
KdRight = zeros(nLayerRight,1);
for iLayer = 1: 1: nLayerRight
    Cs = CsRight(iLayer);
    Cp = CpRight(iLayer);
    Rho = RhoRight(iLayer);
    G = Cs^2*Rho;
    Lambda = Cp^2*Rho - 2*G;
    Mat = [1/2 -3/2; 1/2 1/2];
    B = Mat^-1*[Lambda;G];
    KnRight(iLayer) = B(1);
    KsRight(iLayer) = B(2);
    KdRight(iLayer) = (KnRight(iLayer) - KsRight(iLayer)) / 2;
    MRight(iLayer) = Rho*D^2;
    NRight(iLayer) = round(HRight(iLayer)/D);
end
KnRight = KnRight.*(1+1i*DampRight);
KsRight = KsRight.*(1+1i*DampRight);
KdRight = KdRight.*(1+1i*DampRight);
% Add ballast layer
KnRight = [Knb;KnRight];
KsRight = [Ksb;KsRight];
KdRight = [Kdb;KdRight];
MRight = [Mb;MRight];
NRight = [Nb;NRight];
% Add bottom layer
if ~isempty(KvBotRight)
    KnRight = [KnRight;(KvBotRight+1i*CvBotRight)*D];
    KsRight = [KsRight;(KhBotRight+1i*ChBotRight)*D];
    KdRight = [KdRight;0];
    MRight = [MRight;0];
    NRight = [NRight;0];
%     KnRight = [KnRight;(KvBotRight+1i*CvBotRight)*D];
%     KsRight = [KsRight;(KhBotRight+1i*ChBotRight)*D];
%     KdRight = [KdRight;((KvBotRight+1i*CvBotRight)-(KhBotRight+1i*ChBotRight))*D/2];
%     MRight = [MRight;MRight(end)];
%     NRight = [NRight;1];
end

%% Check compatibility of foundations
if any(NLeft ~= NRight) || length(HLeft) ~= length(HRight)
    fprintf(1, '\n');
    fprintf(1, 'ERROR: left and right foundations must have the same layering depths.\n');
end

Knlat = KnLeft;
Kslat = KsLeft;
Kdlat = KdLeft;
Mlat = MLeft;
Nlat = NLeft;
if any(Knlat~=KnRight) || any(Kslat~=KsRight) || any(Kdlat~=KdRight) || any(Mlat~=MRight)
    Knlat = [Knlat KnRight];
    Kslat = [Kslat KsRight];
    Kdlat = [Kdlat KdRight];
    Mlat  = [Mlat  MRight ];
end

%% Geometrcal approximations
N = round(Width/D)+1;
M = round(Spacing/D); M = M - N;
Ku  = Ku/N;
Kuh = Kuh/N;
%% Inputs
if Kuh == inf, Kuh = 100*Ku; end

PLOT = 0;
OUTPUT = [];
if size(Mlat,2) == 1
    Mlat = [Mlat Mlat];
    Knlat = [Knlat Knlat];
    Kslat = [Kslat Kslat];
    Nlat = [Nlat Nlat];
end
%% Assemble stiffness matrices - left
nNodePerColLat = sum(Nlat(:,1)) + 1;
nNodePerCell = nNodePerColLat*(N+M) + 2*(N+M);
nNodeTotal = nNodePerCell * nCell * size(Mlat,2);
NODE_TYPE = zeros(nNodeTotal,5); % [TYPE X Y Mass J]
SIDE = zeros(nNodeTotal,1);
RAIL = 1; SLEEPER = 2; BALLAST = 3;
BLOCK_SIZE = 100000;
ad2Connection = zeros(BLOCK_SIZE, 9); % [RAIL Node1 Node2 EI CEI m] or [SPRING Node1 Node2 Dofs1 Dofs2 K C]
kCon = 0;
BEAM = 1;
SPRING = 2;
kCell = 0;
nTotalCell = 2*nCell;
fprintf(1, '\n');
fprintf(1, 'Assembling element matrices for %d cells...\n', nTotalCell);
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;
for iSide = 1:1:size(Mlat,2)
    mlat =  Mlat(:,iSide);
    knlat = Knlat(:,iSide);
    kslat = Kslat(:,iSide);
    kdlat = (knlat-kslat)/2;
    for jCell = 1: 1: nCell
        if mod(kCell+1,round(nTotalCell/50))==0
            star = star + 1;
            if star <= 50 && STARS(star) == 0
                fprintf(1, '*');
                STARS(star) = 1;
            end
        end
        kNode_RR = kCell*nNodePerCell + 1;
        % Column by column assemble the elements
        SurfaceNodes = zeros(1,N+M);
        for iN = 0: 1: N+M-2
            if kCon + 25*sum(Nlat(:,iSide)) > size(ad2Connection,1)
                ad2Connection = [ad2Connection; zeros(BLOCK_SIZE,size(ad2Connection,2))];
            end
            %% Define rail nodes, ballast nodes and sleeper node
            kNode_RS = [];
            kNode_RL = kNode_RR;
            kNode_RM = kNode_RL + nNodePerColLat + 1;
            kNode_RR = kNode_RM + 1;
            kNode_BL = kNode_RL + 1;
            kNode_BR = kNode_RR + 1;
            if mod(N,2)==1 && iN == (N-1)/2
                kNode_RS = kNode_RL;
                kNode_S = kNode_RL + 1;
                kNode_RM = kNode_RM + 1;
                kNode_RR = kNode_RR + 1;
                kNode_BL = kNode_BL + 1;
                kNode_BR = kNode_BR + 1;
            elseif mod(N,2)==1 && iN == (N-1)/2-1
                kNode_BR = kNode_BR + 1;
            elseif mod(N,2)==0 && iN == N/2-1
                kNode_RS = kNode_RM;
                kNode_S = kNode_RM + 1;
                kNode_RR = kNode_RR + 1;
                kNode_BR = kNode_BR + 1;
            end
            if iN == 0
                kNode_RLS = kNode_RL;
                kNode_BLS = kNode_BL;
            end
            SurfaceNodes(iN+1) = kNode_BL;
            if iN == N+M-2, SurfaceNodes(iN+2) = kNode_BR; end
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],1) = RAIL;
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],2) = (kCell*(N+M)+iN-1+(1:0.5:2))*D;
            NODE_TYPE([kNode_RL kNode_RM kNode_RR],3) = N*D;
            NODE_TYPE([kNode_BL:(kNode_RM-1) kNode_BR:(kNode_BR+nNodePerColLat-1)],1) = BALLAST;
            NODE_TYPE(kNode_BL:(kNode_RM-1),2) = (kCell*(N+M)+iN)*D;
            NODE_TYPE(kNode_BL:(kNode_RM-1),3) = -(0:1:(nNodePerColLat-1))*D;
            NODE_TYPE(kNode_BR:(kNode_BR+nNodePerColLat-1),2) = (kCell*(N+M)+iN+1)*D;
            NODE_TYPE(kNode_BR:(kNode_BR+nNodePerColLat-1),3) = -(0:1:(nNodePerColLat-1))*D;
            SIDE([kNode_RL kNode_RM kNode_RR]) = iSide;
            SIDE([kNode_BL:(kNode_RM-1) kNode_BR:(kNode_BR+nNodePerColLat-1)]) = iSide;
            if ~isempty(kNode_RS)
                NODE_TYPE(kNode_S,1) = SLEEPER;
                NODE_TYPE(kNode_S,2) = NODE_TYPE(kNode_RS,2);
                NODE_TYPE(kNode_S,3) = N*D/2;
                NODE_TYPE(kNode_S,4) = Ms;
                NODE_TYPE(kNode_S,5) = Js;
                SIDE(kNode_S) = iSide;
            end

            %% Assemble rails
            kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RL kNode_RM real(EI) imag(EI) m];
            kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RM kNode_RR real(EI) imag(EI) m];

            %% Assemble rail pads
            if ~isempty(kNode_RS)
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_RS kNode_S [1 0 -1  0] real(Kv) imag(Kv)];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_RS kNode_S [0 1  0 -1] real(Kt) imag(Kt)];
                if iSide == 2 && jCell == 1
                    kNodeCentral = kNode_RS;
                end
            end

            %% Assemble ballast springs
            kDepth = 0;
            for iLayer = 1: 1: size(Nlat(:,iSide))
            for iDiv = 1: 1: Nlat(iLayer,iSide)
                % Add mass to the nodes
                NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],4) = ...
                    NODE_TYPE([kNode_BL+kDepth+(0:1) kNode_BR+kDepth+(0:1)],4) + 0.25*mlat(iLayer);
                if iN == 0
                    NODE_TYPE(kNode_BL+kDepth+(0:1),4) = ...
                        NODE_TYPE(kNode_BL+kDepth+(0:1),4) + 0.25*mlat(iLayer);
                elseif iN == N+M-2
                    NODE_TYPE(kNode_BR+kDepth+(0:1),4) = ...
                        NODE_TYPE(kNode_BR+kDepth+(0:1),4) + 0.25*mlat(iLayer);
                end
                % Add springs Left
                if iN == 0
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [0 1 0 -1] real(knlat(iLayer)) imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [1 0 -1 0] real(kslat(iLayer)) imag(kslat(iLayer))];
                else
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [0 1 0 -1] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+(0:1) [1 0 -1 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                end
                % Add springs right
                if iN == N+M-2
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [0 1 0 -1] real(knlat(iLayer)) imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [1 0 -1 0] real(kslat(iLayer)) imag(kslat(iLayer))];
                else
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [0 1 0 -1] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth+(0:1) [1 0 -1 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                end
                % Add top and bottom springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth [1 0 -1 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth [0 1 0 -1] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth+1 [1 0 -1 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth+1 [0 1 0 -1] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                % Add diagonal springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth kNode_BR+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(kdlat(iLayer)) imag(kdlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth+1 kNode_BR+kDepth sqrt(2)/2*[-1 -1 1 1] real(kdlat(iLayer)) imag(kdlat(iLayer))];
                kDepth = kDepth + 1;
            end
            if Nlat(iLayer,iSide) == 0
                % Add top and bottom springs
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                if iN == 0
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BL+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                elseif iN == N+M-2
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [0 1 0 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
                    kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR+kDepth -1 [1 0 0 0] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
                end
            end
            end
        end
        % Add springs between sleeper and ballast
        SurfaceNodes = SurfaceNodes(1:N);
        for iNode = 1: 1: N
            kNode = SurfaceNodes(iNode);
            % Vertical connection
            dist = (-(N-1)/2+iNode-1)*D;
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode kNode_S [0 -1 1 dist] real(Ku) imag(Ku)];
            % Horizontal connection
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode -1 [1 0 0 0] real(Kuh) imag(Kuh)];
        end
        kCell = kCell + 1;
        if jCell == 1
            % save this
            kNode_BR_Prev = kNode_BR;
            kNode_RR_Prev = kNode_RR;
            continue;
        end
        % Assemble connection with Previous cell
        kCon = kCon + 1; ad2Connection(kCon,1:6) = [BEAM kNode_RR_Prev kNode_RLS real(EI) imag(EI) m];
        kDepth = 0;
        for iLayer = 1: 1: size(Nlat,iSide)
        for iDiv = 1: 1: Nlat(iLayer,iSide)
            % Add top and bottom springs
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth [1 0 -1 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth [0 1 0 -1] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth+1 [1 0 -1 0] 0.5*real(knlat(iLayer)) 0.5*imag(knlat(iLayer))];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth+1 [0 1 0 -1] 0.5*real(kslat(iLayer)) 0.5*imag(kslat(iLayer))];
            % Add diagonal springs
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth kNode_BLS+kDepth+1 sqrt(2)/2*[-1 1 1 -1] real(kdlat(iLayer)) imag(kdlat(iLayer))];
            kCon = kCon + 1; ad2Connection(kCon,1:9) = [SPRING kNode_BR_Prev+kDepth+1 kNode_BLS+kDepth sqrt(2)/2*[-1 -1 1 1] real(kdlat(iLayer)) imag(kdlat(iLayer))];
            kDepth = kDepth + 1;
        end
        end
        kNode_BR_Prev = kNode_BR;
        kNode_RR_Prev = kNode_RR;

    end
end
%% Split the rail connections which are longer than D/2 into two,
%  for coherence purposes
nCon = kCon;
for iCon = 1: 1: nCon
    if ad2Connection(iCon,1) ~= BEAM, continue; end
    Node1 = ad2Connection(iCon,2);
    Node2 = ad2Connection(iCon,3);
    dX = NODE_TYPE(Node2,2) - NODE_TYPE(Node1,2);
    if abs(dX) < D/2*(1+1e-5), continue; end
    NODE_TYPE = [NODE_TYPE(1:Node2-1,:); (NODE_TYPE(Node2,:)+NODE_TYPE(Node1,:))/2; NODE_TYPE(Node2:end,:)];
    SIDE = [SIDE(1:Node2-1,:); SIDE(Node2,:); SIDE(Node2:end,:)];
    Node3 = Node2;
    ad2Connection(ad2Connection(:,2)>=Node3,2) = ad2Connection(ad2Connection(:,2)>=Node3,2) + 1;
    ad2Connection(ad2Connection(:,3)>=Node3,3) = ad2Connection(ad2Connection(:,3)>=Node3,3) + 1;
    if kNodeCentral >= Node3, kNodeCentral = kNodeCentral + 1; end
    kCon = kCon + 1;  ad2Connection(kCon,1:6) = [BEAM Node3 Node3+1 real(EI) imag(EI) m];
    ad2Connection(iCon,3) = Node3;
end
%% Shift the x coordinate so that first cell of right side is 0 at the
%  sleeper position
Xcenter = NODE_TYPE(kNodeCentral,2);
NODE_TYPE(:,2) = NODE_TYPE(:,2) - Xcenter;
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');
%% Make sure that there is one node at each loaded position (moving load)
if ADJUST
    fprintf(1, '\n');
    fprintf(1, 'Adjunting rail discretization to time steps...\n');
    fprintf(1, '[0%%                                            100%%]\n');
    fprintf(1, '[');
    STARS = zeros(50,1); star = 0;
    RAIL_NODES = find(NODE_TYPE(:,1) == RAIL);
    X_Rail = NODE_TYPE(RAIL_NODES,2);
    xMin = min(X_Rail);
    xMax = max(X_Rail);
    T = -Tmax:dt:Tmax;
    X = V*T;
    for iX = 1: 1: length(X)
        if mod(iX+1,round(length(X)/50))==0
            star = star + 1;
            if star <= 50 && STARS(star) == 0
                fprintf(1, '*');
                STARS(star) = 1;
            end
        end
        x = X(iX);
        if x<= xMin || x >= xMax, continue; end
        % Add node
        kNodeLeft = find((NODE_TYPE(:,1)==RAIL).*(NODE_TYPE(:,2)<=x));
        kNodeLeft = kNodeLeft(end);
        if abs(x-NODE_TYPE(kNodeLeft,2)) < V*dt/1000, continue; end
        kNodeRight = find((NODE_TYPE(:,1)==RAIL).*(NODE_TYPE(:,2)>x));
        kNodeRight = kNodeRight(1);
        if abs(x-NODE_TYPE(kNodeRight,2)) < V*dt/1000, continue; end

        % Find the element with these nodes
        jCon = find((ad2Connection(:,1)==BEAM).*...
                    (ad2Connection(:,2)==kNodeLeft).*...
                    (ad2Connection(:,3)==kNodeRight));
        if isempty(jCon), continue; end

        % Add node and dhift all nodes by greater than NodeRight by 1
        NODE_TYPE = [NODE_TYPE(1:kNodeRight-1,:);
                     RAIL x NODE_TYPE(kNodeLeft,3:end);
                     NODE_TYPE(kNodeRight:end,:)];
             SIDE = [SIDE(1:kNodeRight-1,:); SIDE(kNodeRight,:); SIDE(kNodeRight:end,:)];
        ad2Connection(ad2Connection(:,2)>=kNodeRight,2) = ad2Connection(ad2Connection(:,2)>=kNodeRight,2) + 1;
        ad2Connection(ad2Connection(:,3)>=kNodeRight,3) = ad2Connection(ad2Connection(:,3)>=kNodeRight,3) + 1;

        % Add new connection
        ad2Connection(jCon,3) = kNodeRight;
        kCon = kCon + 1;  ad2Connection(kCon,1:6) = [BEAM kNodeRight kNodeRight+1 real(EI) imag(EI) m];
    end
    for iStar = star+1:1:50
        fprintf(1, '*');
    end
    fprintf(1, ']\n');
end
ad2Connection = ad2Connection(1:kCon,:);

%% Plot modeled section
if PLOT
    figure; hold on
    plot(NODE_TYPE(NODE_TYPE(:,1)==BALLAST,2),NODE_TYPE(NODE_TYPE(:,1)==BALLAST,3),'ob');
    plot(NODE_TYPE(NODE_TYPE(:,1)==SLEEPER,2),NODE_TYPE(NODE_TYPE(:,1)==SLEEPER,3),'or');
    plot(NODE_TYPE(NODE_TYPE(:,1)==RAIL,2),NODE_TYPE(NODE_TYPE(:,1)==RAIL,3),'ok');
    for iCon = 1: 1: size(ad2Connection,1)
        Nodes = ad2Connection(iCon,2:3);
        if Nodes(1) == -1, Nodes(1) = [];
        elseif Nodes(2) == -1, Nodes(2) = []; end
        if length(Nodes) == 1
            COLOR = 'vk';
        elseif ad2Connection(iCon,1) == SPRING
            COLOR = '.-g';
        elseif ad2Connection(iCon,1) == BEAM
            COLOR = '.-r';
        end
        plot(NODE_TYPE(Nodes,2),NODE_TYPE(Nodes,3),COLOR);
    end
    axis equal
end
%% Assemble stuctural elements
nNode = size(NODE_TYPE,1);
nElem = size(ad2Connection,1);
nDof = 2*nNode;
ad2M = zeros(16*nElem,3); NZT_M = 0;
ad2K = zeros(16*nElem,3); NZT_K = 0;
ad2C = zeros(16*nElem,3); NZT_C = 0;
% Assemble lumped masses
for iNode = 1:1:nNode
    if NODE_TYPE(iNode,1) == RAIL, continue; end
    if NODE_TYPE(iNode,1) == SLEEPER
        ad2M(NZT_M+(1:2),:) = [NODE_TYPE(iNode,4) 2*iNode-1 2*iNode-1;
                               NODE_TYPE(iNode,5) 2*iNode-0 2*iNode-0];
        NZT_M = NZT_M + 2;
    end
    if NODE_TYPE(iNode,1) == BALLAST
        ad2M(NZT_M+(1:2),:) = [NODE_TYPE(iNode,4) 2*iNode-1 2*iNode-1;
                               NODE_TYPE(iNode,4) 2*iNode-0 2*iNode-0];
        NZT_M = NZT_M + 2;
    end
end
% Assemble elastic elements - springs and beams
for iElem = 1: 1: nElem
    kNode1 = ad2Connection(iElem,2);
    kNode2 = ad2Connection(iElem,3);
    ROW = [2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2;
           2*kNode1-1; 2*kNode1; 2*kNode2-1; 2*kNode2];
    COL = [2*kNode1-1; 2*kNode1-1; 2*kNode1-1; 2*kNode1-1 ;
           2*kNode1  ; 2*kNode1  ; 2*kNode1  ; 2*kNode1   ;
           2*kNode2-1; 2*kNode2-1; 2*kNode2-1; 2*kNode2-1 ;
           2*kNode2  ; 2*kNode2  ; 2*kNode2  ; 2*kNode2  ];
    if ad2Connection(iElem,1) == BEAM
        EI = ad2Connection(iElem,4);
        CEI = ad2Connection(iElem,5);
        m = real(ad2Connection(iElem,6));
        l = NODE_TYPE(kNode2,2)-NODE_TYPE(kNode1,2);
        l2 = l*l;
        l3 = l2*l;
        
        Kelem = EI*[ 12/l3  6/l2  -12/l3  6/l2 ;
                      6/l2   4/l   -6/l2   2/l ;
                    -12/l3 -6/l2   12/l3 -6/l2 ;
                      6/l2   2/l   -6/l2   4/l ];
                  
        Celem = CEI*[ 12/l3  6/l2  -12/l3  6/l2 ;
                       6/l2   4/l   -6/l2   2/l ;
                     -12/l3 -6/l2   12/l3 -6/l2 ;
                       6/l2   2/l   -6/l2   4/l ];
                  
        Melem = m*l/420*[  156  22*l    54 -13*l ;
                          22*l  4*l2  13*l -3*l2 ;
                            54  13*l   156 -22*l ;
                         -13*l -3*l2 -22*l  4*l2];
        ad2K(NZT_K+(1:16),:) = [Kelem(:) ROW COL]; NZT_K = NZT_K + 16;
        ad2C(NZT_C+(1:16),:) = [Celem(:) ROW COL]; NZT_C = NZT_C + 16;
        ad2M(NZT_M+(1:16),:) = [Melem(:) ROW COL]; NZT_M = NZT_M + 16;
    elseif ad2Connection(iElem,1) == SPRING
        K = ad2Connection(iElem,8);
        C = ad2Connection(iElem,9);
        dofs = ad2Connection(iElem,4:7);
        if kNode1 == -1
            dofs = dofs(3:4);
            ROW = [2*kNode2-1; 2*kNode2;
                   2*kNode2-1; 2*kNode2];
            COL = [2*kNode2-1; 2*kNode2-1;
                   2*kNode2  ; 2*kNode2 ];
        elseif kNode2 == -1
            dofs = dofs(1:2);
            ROW = [2*kNode1-1; 2*kNode1;
                   2*kNode1-1; 2*kNode1];
            COL = [2*kNode1-1; 2*kNode1-1;
                   2*kNode1  ; 2*kNode1 ];
        end
        NZT = length(dofs)^2;
        Kelem = K*(dofs'*dofs);
        Celem = C*(dofs'*dofs);
        ad2K(NZT_K+(1:NZT),:) = [Kelem(:) ROW COL]; NZT_K = NZT_K + NZT;
        ad2C(NZT_C+(1:NZT),:) = [Celem(:) ROW COL]; NZT_C = NZT_C + NZT;
    end
end
ad2K = ad2K(1:NZT_K,:); ad2K = sparse(ad2K(:,2),ad2K(:,3),ad2K(:,1),nDof,nDof);
ad2C = ad2C(1:NZT_C,:); ad2C = sparse(ad2C(:,2),ad2C(:,3),ad2C(:,1),nDof,nDof);
ad2M = ad2M(1:NZT_M,:); ad2M = sparse(ad2M(:,2),ad2M(:,3),ad2M(:,1),nDof,nDof);
% Free dofs - remove last row of ballast nodes if bottom of lattice is
% fixed
ai1FreeDof = 1:1:nDof;
ai1FixedDof = [];
if Nlat(end,1) ~= 0
    ai1FixedNode = find((NODE_TYPE(:,1)==BALLAST).*(NODE_TYPE(:,3)==min(NODE_TYPE(:,3))));
    ai1FixedDof = [ai1FixedNode*2-1; ai1FixedNode*2];
    ai1FixedDof = sort(ai1FixedDof(:),'ascend');
end
% Left boundary
Nodes_Left = find(NODE_TYPE(:,2)==min(NODE_TYPE(:,2)));
ai1FixedDof = [ai1FixedDof; 2*Nodes_Left(:)-1; 2*Nodes_Left(:)];
% Right boundary
Nodes_Right = find(NODE_TYPE(:,2)==max(NODE_TYPE(:,2)));
ai1FixedDof = [ai1FixedDof; 2*Nodes_Right(:)-1; 2*Nodes_Right(:)];
ai1FixedDof = unique(ai1FixedDof);
ai1FixedDof = sort(ai1FixedDof(:),'ascend');
if PLOT
    ai1FixedNode = ai1FixedDof(2:2:end)/2;
    plot(NODE_TYPE(ai1FixedNode,2),NODE_TYPE(ai1FixedNode,3),'*m');
end
ai1FreeDof(ai1FixedDof) = [];

%% Newmark look
gamma = 0.5;
beta  = 0.25;
a0 = 1/(beta*dt^2);
a1 = gamma/(beta*dt);
a2 = 1/(beta*dt);
a3 = 1/(2*beta)-1;
a4 = gamma/beta-1;
a5 = dt*(gamma/(2*beta)-1);

% Time sample
kT = 0;
T = -Tmax:dt:Tmax;
% Ensure that there is a time interval when the load crosses the left edge
Tl = T + max(NODE_TYPE(SIDE==1,2))/V;
% Ensure that there is a time interval when the load crosses the right edge
Tr = T + min(NODE_TYPE(SIDE==2,2))/V;
nT = length(T);
fprintf(1, '\n');
fprintf(1, 'Newmark loop on time steps (%d time intervals)\n', nT);
fprintf(1, '[0%%                                            100%%]\n');
fprintf(1, '[');
STARS = zeros(50,1); star = 0;

% Outputs
NodesLeft  = find(SIDE == 1);
OutputLeft = find(NODE_TYPE(NodesLeft,2) == max(NODE_TYPE(NodesLeft,2)));
OutputLeft = NodesLeft(OutputLeft);
DofsLeft = sort([2*OutputLeft(:)-1; 2*OutputLeft(:)],'ascend');

NodesRight  = find(SIDE == 2);
OutputRight = find(NODE_TYPE(NodesRight,2) == min(NODE_TYPE(NodesRight,2)));
OutputRight = NodesRight(OutputRight);
DofsRight = sort([2*OutputRight(:)-1; 2*OutputRight(:)],'ascend');
if PLOT
    ai1FixedNode = ai1FixedDof(2:2:end)/2;
    plot(NODE_TYPE(OutputLeft,2),NODE_TYPE(OutputLeft,3),'vm');
    plot(NODE_TYPE(OutputRight,2),NODE_TYPE(OutputRight,3),'vm');
end
ULeft  = zeros(length(DofsLeft),length(T));
URight = zeros(length(DofsRight),length(T));
IR_L = zeros(length(DofsLeft),length(DofsLeft),length(T));
IR_R = zeros(length(DofsRight),length(DofsRight),length(T));

for t = T(1):dt:T(end)
    if mod(kT+1,round(nT/50))==0
        star = star + 1;
        if star <= 50 && STARS(star) == 0
            fprintf(1, '*');
            STARS(star) = 1;
        end
    end
    %% Calculate system matrices
    if kT == 0
        ad2Keq = ad2K + a1*ad2C + a0*ad2M;

        % Factoeize Keq
        ad2L = chol(ad2Keq(ai1FreeDof,ai1FreeDof), 'lower');
        ad2U = ad2L';
        % Previous interation valuess
        up = zeros(nDof,2+length(DofsLeft)+length(DofsRight));
        vp = zeros(nDof,2+length(DofsLeft)+length(DofsRight));
        ap = zeros(nDof,2+length(DofsLeft)+length(DofsRight));
    end
    kT = kT + 1;
    %% Calculate external force
    Fext = zeros(nDof,2+length(DofsLeft)+length(DofsRight));
    x = V*T(kT);
    xl = V*Tl(kT);
    if xl >= min(NODE_TYPE(:,2)) && xl < max(NODE_TYPE(:,2))
        kNode1 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)<=xl)); kNode1 = kNode1(end);
        kNode2 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)>xl)); kNode2 = kNode2(1);
        x1 = NODE_TYPE(kNode1,2); x2 = NODE_TYPE(kNode2,2);
        l = x2-x1;
        Fact = (xl-x1)/l;
        N1 = 1 -            3*Fact^2 + 2*Fact^3;
        N2 =     l*Fact - 2*l*Fact^2 + l*Fact^3;
        N3 =                3*Fact^2 - 2*Fact^3;
        N4 =               -l*Fact^2 + l*Fact^3;
        Fext(2*kNode1-1,1) = N1;
        Fext(2*kNode1,1) = N2;
        Fext(2*kNode2-1,1) = N3;
        Fext(2*kNode2,1) = N4;
    end
    xr = V*Tr(kT);
    if xr >= min(NODE_TYPE(:,2)) && xr < max(NODE_TYPE(:,2))
        kNode1 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)<=xr)); kNode1 = kNode1(end);
        kNode2 = find((NODE_TYPE(:,1)==RAIL) .* (NODE_TYPE(:,2)>xr)); kNode2 = kNode2(1);
        x1 = NODE_TYPE(kNode1,2); x2 = NODE_TYPE(kNode2,2);
        l = x2-x1;
        Fact = (xr-x1)/l;
        N1 = 1 -            3*Fact^2 + 2*Fact^3;
        N2 =     l*Fact - 2*l*Fact^2 + l*Fact^3;
        N3 =                3*Fact^2 - 2*Fact^3;
        N4 =               -l*Fact^2 + l*Fact^3;
        Fext(2*kNode1-1,2) = N1;
        Fext(2*kNode1,2) = N2;
        Fext(2*kNode2-1,2) = N3;
        Fext(2*kNode2,2) = N4;
    end
    if abs(t) < dt/1000
        for iDof = 1: 1: length(DofsLeft)
            Fext(DofsLeft(iDof),2+iDof) = 1;
        end
        for iDof = 1: 1: length(DofsRight)
            Fext(DofsRight(iDof),2+length(DofsLeft)+iDof) = 1;
        end
    end
    %% Add terms due to previous state
    Fext = Fext + ad2M*(a0*up+a2*vp+a3*ap) + ad2C*(a1*up+a4*vp+a5*ap);
    %% Calculate new dispalcements, velocities and accelerations
    uc = zeros(nDof,2+length(DofsLeft)+length(DofsRight));
    un = ad2U\(ad2L\Fext(ai1FreeDof,:));
    uc(ai1FreeDof,:) = un;
    vc = a1*(uc-up) - a4*vp - a5*ap;
    ac = a0*(uc-up) - a2*vp - a3*ap;
    
    %% store old values
    up = uc;
    vp = vc;
    ap = ac;
    ULeft(:,kT) = uc(DofsLeft,1);
    URight(:,kT) = uc(DofsRight,2);
    IR_L(:,:,kT) = uc(DofsLeft,2+(1:length(DofsLeft)));
    IR_R(:,:,kT) = uc(DofsRight,2+length(DofsLeft)+(1:length(DofsRight)));
end
for iStar = star+1:1:50
    fprintf(1, '*');
end
fprintf(1, ']\n');
%% Remove fixed dofs from IR_L and IR_R
if Nlat(end,1) ~= 0
    IR_L = IR_L(1:end-2,1:end-2,:);
end
if Nlat(end,2) ~= 0
    IR_R = IR_R(1:end-2,1:end-2,:);
end
OUTPUT.ULeft = ULeft;
OUTPUT.URight = URight;
OUTPUT.IR_L = IR_L;
OUTPUT.IR_R = IR_R;
OUTPUT.T = T;
OUTPUT.Tl = Tl;
OUTPUT.Tr = Tr;
end
