% MATLAB script for the following paper:
% Berge S. H., De Winter, J. C. F., Dodou, D., Pooyan Afghari, A., Papadimitriou, E., Reddy, N., Dong, Y., Raju, N., & Farah, H. (2024). Understanding cyclists’ perception of driverless vehicles through eye-tracking and interviews. Transportation Research Part F. https://doi.org/10.1016/j.trf.2024.11.015

clear all;close all;clc %#ok<CLALL>

Labels={'person' % 1
    'bicycle' % 2
    'car' % 3
    'motorbike' % 4
    'aeroplane' % 5
    'bus' % 6
    'train' % 7
    'truck' % 8
    'boat' % 9
    'traffic light' % 10
    'fire hydrant'
    'stop sign' % 12
    'parking meter'
    'bench' % 14
    'bird'
    'cat'
    'dog'
    'horse'
    'sheep'
    'cow'
    'elephant'
    'bear'
    'zebra'
    'giraffe'
    'backpack'
    'umbrella'
    'handbag'
    'tie'
    'suitcase'
    'frisbee'
    'skis'
    'snowboard'
    'sports ball'
    'kite'
    'baseball bat'
    'baseball glove'
    'skateboard'
    'surfboard'
    'tennis racket'
    'bottle'
    'wine glass'
    'cup'
    'fork'
    'knife'
    'spoon'
    'bowl'
    'banana'
    'apple'
    'sandwich'
    'orange'
    'broccoli'
    'carrot'
    'hot dog'
    'pizza'
    'donut'
    'cake'
    'chair'
    'sofa'
    'pottedplant'
    'bed'
    'diningtable'
    'toilet'
    'tvmonitor'
    'laptop'
    'mouse'
    'remote'
    'keyboard'
    'cell phone'
    'microwave'
    'oven'
    'toaster'
    'sink'
    'refrigerator'
    'book'
    'clock'
    'vase'
    'scissors'
    'teddy bear'
    'hair drier'
    'toothbrush'};
%%
AgeGender=[   25     1 % 1 = female, 2 = male
    31     2
    23     1
    38     2
    24     2
    39     2
    NaN   NaN
    24     1
    24     1
    57     1
    24     2
    24     2
    31     1
    35     2
    23     2
    23     2
    51     2
    32     2
    32     2
    26     2
    36     2
    NaN   NaN
    38     2
    29     1
    26     1
    46     2
    25     2
    NaN   NaN
    28     1
    27     2
    38     1
    NaN   NaN
    29     2
    22     1
    57     2
    25     1
    26     1
    23     2
    28     2
    26     1
    67     2];
[landmarks,landmarkstext] = xlsread('Landmarks.xlsx'); %#ok<XLSRD>

LettersShown=[landmarkstext(6:8:end,end-1) landmarkstext(7:8:end,end-1) landmarkstext(8:8:end,end-1) landmarkstext(9:8:end,end-1)];
DriverPresence=[landmarkstext(2:8:end,end-7) landmarkstext(3:8:end,end-7) landmarkstext(4:8:end,end-7) landmarkstext(5:8:end,end-7) landmarkstext(6:8:end,end-7) landmarkstext(7:8:end,end-7) landmarkstext(8:8:end,end-7) landmarkstext(9:8:end,end-7)];
LettersShownCoded=strcmp(LettersShown,'HN')+2*strcmp(LettersShown,'NH')+3*strcmp(LettersShown,'OD');
DriverPresenceCoded=strcmp(DriverPresence,'No')+2*strcmp(DriverPresence,'Yes');
DriverPresenceCoded(DriverPresenceCoded == 0)=NaN;
LettersShownCodedCounts=[sum(LettersShownCoded' == 1)' sum(LettersShownCoded' == 2)' sum(LettersShownCoded' == 3)'];
disp('Letter counts of the four trials of Session 2 (row 1: HN, row 2: NH, row 3: OD)')
disp(histc(LettersShownCoded,1:3)) %#ok<HISTC>
%% Determine participant, session, and trial number for each landmark, based on the Excel file
Landmarks_participant_number=NaN(296,1);
Landmarks_trial_number=NaN(296,1);
Landmarks_session_number=NaN(296,1);
for i=2:length(landmarkstext(:,1)) % loop over rows; extract participant number from P1, P2, etc in first column
    landmarks_participant_number = cell2mat(landmarkstext(i,1));
    landmarks_trial_number = cell2mat(landmarkstext(i,2));
    Landmarks_participant_number(i-1) = str2double(landmarks_participant_number(2:end));
    Landmarks_trial_number(i-1) = str2double(landmarks_trial_number(7:end));
    Landmarks_session_number(i-1) = ceil((Landmarks_trial_number(i-1))/4);
end
%% Extract video stats from all videos and save into 'video_stats.mat', for each interaction separately
extract_video_stats = 0;
if extract_video_stats == 1
    landmarkstext(1,:)=[];

    for landmark_nr=1:size(landmarks,1) % loop over all landmarks
        Trial = landmarkstext{landmark_nr,2};
        Trial = str2num(Trial(6:end)); %#ok<ST2NM> % trial (1 to 8)
        Session = 1+floor( (Trial-1)/4); % Calculate session number (1 or 2) from trial number (1 to 8)
        videoname=[landmarkstext{landmark_nr,1} ' Session ' num2str(Session) ' Project.mp4'];

        if ismember(landmark_nr,1:8)
            videoname='P1 Session 1 and 2 Project.mp4'; % for participant 1, session 1 and 2 were combined into 1 video file
        end
        clear BB SS LL

        if exist(['..\Extracted data\Videos from project files\' videoname]) && ~isnan(landmarks(landmark_nr,6))
            vidObj=VideoReader(['..\Extracted data\Videos from project files\' videoname]);
            video_stats(landmark_nr,:)=[vidObj.NumFrames vidObj.FrameRate vidObj.CurrentTime vidObj.Width vidObj.Height vidObj.Duration]; %#ok<SAGROW>
        end
    end
    save('video_stats.mat','video_stats')
else
    load video_stats.mat
end
%% Apply YOLO to video frames and save as .mat file 'yoloresults_', for each interaction separately
execute_yolo=0; % if set to 1, then use YOLO4 on video frames of all videos
if execute_yolo == 1
    addpath('src')
    modelName = 'YOLOv4-coco';
    model = helper.downloadPretrainedYOLOv4(modelName);
    net = model.net;

    % Get classnames of COCO dataset
    classNames = helper.getCOCOClassNames;
    % Get anchors used in training of the pretrained model
    anchors = helper.getAnchors(modelName);

    % Detect objects in test image
    executionEnvironment = 'auto';
    %% Perform YOLO on video frames
    [landmarks,landmarkstext] = xlsread('Landmarks.xlsx'); %#ok<XLSRD>
    landmarkstext(1,:)=[];

    for landmark_nr=121 %[67:69 102:106 211:213] % loop over all landmarks
        Trial = landmarkstext{landmark_nr,2};
        Trial = str2num(Trial(6:end)); %#ok<ST2NM> % trial (1 to 8)
        Session = 1+floor( (Trial-1)/4); % Calculate session number (1 or 2) from trial number (1 to 8)
        videoname=[landmarkstext{landmark_nr,1} ' Session ' num2str(Session) ' Project.mp4'];

        if ismember(landmark_nr,1:8)
            videoname='P1 Session 1 and 2 Project.mp4'; % for participant 1, session 1 and 2 were combined into 1 video file
        end
        BB=cell(1, 13755);SS=BB;LL=BB;
        if exist(['..\Extracted data\Videos from project files\' videoname]) && ~isnan(landmarks(landmark_nr,6))
            vidObj=VideoReader(['..\Extracted data\Videos from project files\' videoname]);
            disp([vidObj.NumFrames vidObj.FrameRate vidObj.Width vidObj.Height vidObj.Duration]);
            try
                close(vr) % close video in case it still happens to be open
            catch error
            end
            vehicle_out_of_sight=landmarks(landmark_nr,6);  % first frame where the vehicle was out of sight
            for i=vehicle_out_of_sight-15*25:vehicle_out_of_sight+25 % loop over interval of frames
                disp([landmark_nr i])
                Fr = read(vidObj, i); % video frame

                [bboxes, scores, labels] = detectYOLOv4(net, Fr, anchors, classNames, executionEnvironment);
                annotations = string(labels);
                BB{i}=bboxes;
                SS{i}=scores;
                LL{i}=labels;
            end
            save(['yoloresults_' landmarkstext{landmark_nr,1} '_Session_' num2str(Session) '_Round_' num2str(Trial)],'BB','SS','LL','landmark_nr')
        end
    end
end % end of 'execute YOLO'
%% reformat the YOLO results , store in 'CarBox' and 'PedestrianBox' matrices
filenames=dir('yoloresults\yoloresults*');
[CarBox,PedestrianBox]=deal(NaN(length(filenames),59000,4));
yolofilestats=NaN(length(filenames),3); % matrix with participant number, session number, and trial number for the Yolo file names

for file_nr=1:length(filenames) % loop over files
    disp(['Reformat the YOLO results and store in CarBox and Pedestrian box matrices, file number: ' num2str(file_nr) ' of ' num2str(length(filenames))])
    filename=filenames(file_nr).name;
    load(['yoloresults/' filename])

    % extract participant number, session number, and trial number from
    % file name, and store in 'yolofilestats' matrix
    dash_indices=strfind(filename,'_');
    dot_indices=strfind(filename,'.');
    yolofilestats(file_nr,1)=str2double(filename(dash_indices(1)+2:dash_indices(2)-1)); % participant number
    yolofilestats(file_nr,2)=str2double(filename(dash_indices(3)+1:dash_indices(4)-1)); % session number
    yolofilestats(file_nr,3)=str2double(filename(dash_indices(5)+1:dot_indices-1)); % trial number

    clear labels bboxes scores labelsn
    Labelsn=NaN(length(BB),15);
    Bboxes=NaN(length(BB),15,7);

    for j=1:length(BB) % loop over all video frames

        labels=LL{j};
        bboxes=BB{j}; % (bounding boxes [x,y,w,h], classification scores, and labels)
        scores=SS{j};

        labelsn=NaN(size(scores));
        for l=1:length(labels)
            labelsn(l)=find(strcmp(Labels,char(labels(l))));
        end

        if ~isempty(bboxes)
            bboxes(:,5)=bboxes(:,1)+bboxes(:,3); % right x
            bboxes(:,6)=bboxes(:,2)+bboxes(:,4); % bottom y
            bboxes(:,7)=(bboxes(:,1)+bboxes(:,5))/2; % middle x
        end
        Labelsn(j,1:length(labelsn)) = labelsn;
        Bboxes(j,1:size(bboxes,1),1:size(bboxes,2)) = bboxes;
        IDcar=find(ismember(Labelsn(j,:),3)  ,1,'first'); % indices of car as first object
        if ~isempty(IDcar)
            CarBox(file_nr,j,1)=Bboxes(j,IDcar,1); % x
            CarBox(file_nr,j,2)=Bboxes(j,IDcar,2); % y
            CarBox(file_nr,j,3)=Bboxes(j,IDcar,5)-Bboxes(j,IDcar,1); % width
            CarBox(file_nr,j,4)=Bboxes(j,IDcar,6)-Bboxes(j,IDcar,2); % height
        end

        IDcar=find(ismember(Labelsn(j,:),1)  ,1,'first'); % indices of person as first object
        if ~isempty(IDcar)
            PedestrianBox(file_nr,j,1)=Bboxes(j,IDcar,1); % x
            PedestrianBox(file_nr,j,2)=Bboxes(j,IDcar,2); % y
            PedestrianBox(file_nr,j,3)=Bboxes(j,IDcar,5)-Bboxes(j,IDcar,1); % width
            PedestrianBox(file_nr,j,4)=Bboxes(j,IDcar,6)-Bboxes(j,IDcar,2); % height
        end
    end
end
%% Create a test/demo figure that shows bounding box height in blue and pedestrian box height in green
time=(1:size(CarBox,2))/24.9548317545243;
participant_number=35;
session_number=2;
trial=7;
yolofile_nr = find(yolofilestats(:,1) == participant_number & yolofilestats(:,2) == session_number & yolofilestats(:,3) == trial); % find yolo file number that corresponds to participant number and trial number

close all
figure;
tiledlayout(2,1);
nexttile
plot(time,CarBox(yolofile_nr,:,4),'b','Linewidth',2);hold on % plot bounding box height
plot(time,PedestrianBox(yolofile_nr,:,4),'g','Linewidth',2);hold on % plot bounding box height
grid on;ax=gca; ax.LineWidth=1.5;
ylabel('Bounding box height (pixels)')
title('Participant 35, Session 2, Trial 3 of 4')
nexttile
plot(time,CarBox(yolofile_nr,:,4),'b','Linewidth',2);hold on % plot bounding box height
plot(time,PedestrianBox(yolofile_nr,:,4),'g','Linewidth',2);hold on % plot bounding box height
grid on;ax=gca; ax.LineWidth=1.5;
ylabel('Bounding box height (pixels)')
title('Participant 35, Session 2, Trial 3 of 4')
fig=gcf;set(findall(fig,'-property','FontName'),'FontName','Arial','Fontsize',24)
set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
%% Read eye-tracking data, connect to video frame number, and store the gaze point coordinates (x and y) into 'Coordinates_store'
% save as .mat file Tobii 3 - Eye ...

FrameRate=24.9548317545243;

read_eye=0;

if read_eye == 1
    for participant_number = 1:41
        for session_number = 1:2

            filename=['Tobii 3 - Participant ' num2str(participant_number) ' Session ' num2str(session_number) '.mat'];
            if participant_number == 1
                filename=['Tobii 3 - Participant ' num2str(participant_number) ' Session 1and2.mat'];
            end
            if exist(filename) %#ok<*EXIST>

                load(filename)
                x = x(x(:,53) == 1,:); % Only select data points for which a video recording is available (as marked by the 53rd variable, making 'fullstream.mp4' or 'scenevideo.mp4')

                Time=x(:,1);
                Time=(Time-Time(1))/10^6; % create time vector that matches the video time
                eyetracker = find(x(:,52) == 2 & Time<=max(Time)-1); % indices of eye tracker data, and discard last 1 second
                x = x(eyetracker,:); % retain only the eye-tracker data
                Gaze_and_pd = x(:,[16 17 33 34]);

                Gaze_and_pd(Gaze_and_pd == 0)=NaN; % added; ensure that all zeros are also set to NaN

                Gaze_and_pd(:,3) = mean(Gaze_and_pd(:,3:4),2,'omitnan'); % compute mean pupil diameter of the two eyes
                Gaze_and_pd(:,4) = []; % remove rightmost column since it will not be used anymore

                Time = Time(eyetracker,:); % retain only the time vector for which eye-tracker data is available

                currentime=median(video_stats(Landmarks_participant_number == participant_number & Landmarks_session_number == session_number,3),'omitnan');
                FrameNumber=floor( (Time-currentime-0.025)*FrameRate); % compute video frame number that corresponds to gaze data

                FrameNumber(FrameNumber<1)=1; % make sure that the frame number cannot drop below 0

                Coordinates_store = NaN(max(FrameNumber),6); % pre-allocate a matrix of NaNs, equal length as the number of video frames. Four columns, for x & y filtered, and x & y unfiltered

                % Calculate 'miss_start' and 'miss_end'; they define the gaps in the data, for example due to blinks
                miss=isnan(Gaze_and_pd(:,1)); % missing eye tracking data
                miss_start=1+find(diff(miss)>0); % start indices of intervals with NaNs
                miss_end=NaN(size(miss_start));
                for i=1:length(miss_start) % loop over all start indices of gaps
                    try
                        miss_end(i)=miss_start(i)+find(Gaze_and_pd(miss_start(i):end,1)>0,1,'first')-2; % calculate end indices of intervals with NaNs (only search for Gaze X, since the missing data are the same for Gaze Y and Pupil diameter)
                    catch error
                    end
                end

                Gaze_and_pdC=Gaze_and_pd; % copy the gaze x and y data and pupil diameter data

                Gaze_and_pdCF = movmedian(Gaze_and_pdC,30,'omitnan'); % median filter to cancel out noise  (30 corresponds to 0.3 s)

                Gaze_and_pdCF(:,1) = fixgaps(Gaze_and_pdCF(:,1)); % linearly interpolate gaze x gaps. Call the function 'fixgaps'
                Gaze_and_pdCF(:,2) = fixgaps(Gaze_and_pdCF(:,2)); % linearly interpolate gaze y gaps. Call the function 'fixgaps'
                Gaze_and_pdCF(:,3) = fixgaps(Gaze_and_pdCF(:,3)); % linearly interpolate pupil diameter gaps. Call the function 'fixgaps'

                for frame_nr=1:max(FrameNumber) % Loop over all frames
                    disp(['Participant number:  ' num2str(participant_number) ', session number ' num2str(session_number) ', frame number: ' num2str(frame_nr)]) % output participant number, session number, and frame number
                    %                    eye_frame1=find(FrameNumber == frame_nr); % eye data indices of frame 'frame_number'
                    [a,b]=findInSorted(FrameNumber,frame_nr);
                    eye_frame1=a:b;
                    % filtered and interpolated data
                    if ~isempty(eye_frame1) && ~isnan(nanmean(Gaze_and_pdCF(eye_frame1,1)))  %#ok<NANMEAN> % if there is data for this frame f
                        GazeFrame=nanmean(Gaze_and_pdCF(eye_frame1,:),1); %#ok<NANMEAN> % gaze x, gaze y, and pupil diameter, corresponding to frame f
                        Coordinates_store(frame_nr,[1 2 5])=GazeFrame(:,1:3);
                    end

                    % raw data (not interpolated & not filtered)
                    if ~isempty(eye_frame1) && ~isnan(nanmean(Gaze_and_pd(eye_frame1,1)))  %#ok<NANMEAN> % if there is data for this frame f
                        GazeFrame=nanmean(Gaze_and_pd(eye_frame1,:),1); %#ok<NANMEAN> % gaze x, gaze y, and pupil diameter, corresponding to frame f
                        Coordinates_store(frame_nr,[3 4 6])=GazeFrame(:,1:3);
                    end

                end % end of loop over frames

                save(['Tobii 3 - Eye - Participant ' num2str(participant_number) ' Session ' num2str(session_number) '.mat'],'Coordinates_store')
            end % end if 'if exist filename'
        end % end of session number loop
    end % end of participant number loop
end
%% Compute angular distance to car, height of bounding box, and store bounding box data for each frame of each trial
[angular_distance_to_car, angular_distance_to_pedestrian, height_bounding_box, height_bounding_box_pedestrian] = deal(NaN(41,8,401));

f = 912.8; % f is the distance to the virtual screen in pixels (VD) for the Tobii 3
[StoreTrialInfo,StoreTrialInfo_pedestrian] = deal(NaN(41,8,401,6));
for participant_number=1:41
    for session_number=1:2
        filename=['Tobii 3/Tobii 3 - Eye - Participant ' num2str(participant_number) ' Session ' num2str(session_number) '.mat']; % load gaze-coordinate points
        if exist(filename)
            load(filename)
        end

        if session_number == 1
            Trials=1:4;
        elseif session_number == 2
            Trials=5:8;
        end
        for trial=Trials

            landmark_nr = find(Landmarks_participant_number == participant_number & Landmarks_trial_number == trial); % find landmark number that corresponds to participant number and trial number
            yolofile_nr = find(yolofilestats(:,1) == participant_number & yolofilestats(:,2) == session_number & yolofilestats(:,3) == trial); % find yolo file number that corresponds to participant number and trial number
            vehicle_out_of_sight=landmarks(landmark_nr,6);  % first frame number where the vehicle was out of sight

            counter=0;
            disp(['Computing angular distance gaze to car and pedestrian - Participant number:  ' num2str(participant_number) ', session number ' num2str(session_number) ', trial number: ' num2str(trial)])

            for frame_no=vehicle_out_of_sight-15*25:vehicle_out_of_sight+25 % loop over interval of frames (starting 15 seconds before, ending 1 second after; 25 fps)
                if ~isnan(frame_no)
                    counter=counter+1;

                    Gaze=Coordinates_store(frame_no,1:2);
                    determine_angular_distance; % determines angular_distance_to_car_min, CX, CY, and height_bb, for Car
                    angular_distance_to_car(participant_number,trial,counter) = angular_distance_to_car_min;
                    height_bounding_box(participant_number,trial,counter) = height_bb;
                    StoreTrialInfo(participant_number,trial,counter,:) = [squeeze(CarBox(yolofile_nr,frame_no,:))' Coordinates_store(frame_no,1:2)];

                    determine_angular_distance_pedestrian; % determines angular_distance_to_pedestrian_min, CX, CY, and height_bb, for Pedestrian (experimenter holding the sign for the letter-reading task)
                    angular_distance_to_pedestrian(participant_number,trial,counter) = angular_distance_to_pedestrian_min;
                    height_bounding_box_pedestrian(participant_number,trial,counter) = height_bb;
                    StoreTrialInfo_pedestrian(participant_number,trial,counter,:) = [squeeze(CarBox(yolofile_nr,frame_no,:))' Coordinates_store(frame_no,1:2)];

                end % end of ~isnan condition
            end % end of frame loop
        end % end of trial loop

    end % end of session loop
end % end of participant loop

angular_distance_to_car=real(angular_distance_to_car);
%% Figure 6. Percentage of trials in which participants looked at the vehicle, defined for each video frame across a 10-s window preceding the encounter

% Note that
% Participant 7 does not exist (or is labelled Participant 41)
% Participant 22 does not exist
% Participant 28 does not exist
% Participant 32 does not exist
% There is no data for Participant 25, Session 2
% There is no data for Participant 29, Session 2
% There is no eye-tracking data for Participant 38, Session 2
% Eye-tracking data for Participant 27, Session 1, Trial 1 not used because the car did not move (experimenter error)

v1=squeeze(angular_distance_to_car<4 & angular_distance_to_car>=0); % looking at car
v2=squeeze(angular_distance_to_car>=0); % looking angle available
v1=double(v1);
v2=double(v2);
v1([7 22 28 32],:,:)=NaN;
v2([7 22 28 32],:,:)=NaN;
v1([25 29 38],5:8,:)=NaN;
v2([25 29 38],5:8,:)=NaN;
v1(27,1,:)=NaN;
v2(27,1,:)=NaN;
pc=100.*v1./v2; % percentage of participants looking at car
pc_session1=squeeze(mean(pc(:,1:4,:),2,'omitnan'));
pc_session2=squeeze(mean(pc(:,5:8,:),2,'omitnan'));
v1_session1=squeeze(mean(v1(:,1:4,:),2,'omitnan'));
v1_session2=squeeze(mean(v1(:,5:8,:),2,'omitnan'));
v2_session1=100*squeeze(mean(v2(:,1:4,:),2,'omitnan'));
v2_session2=100*squeeze(mean(v2(:,5:8,:),2,'omitnan'));

v1_pedestrian=squeeze(angular_distance_to_pedestrian<4 & angular_distance_to_pedestrian>=0); % looking at pedestrian
v2_pedestrian=squeeze(angular_distance_to_pedestrian>=0); % looking angle available
v1_pedestrian=double(v1_pedestrian);
v2_pedestrian=double(v2_pedestrian);
v1_pedestrian([7 22 28 32],:,:)=NaN;
v2_pedestrian([7 22 28 32],:,:)=NaN;
v1_pedestrian([25 29 38],5:8,:)=NaN;
v2_pedestrian([25 29 38],5:8,:)=NaN;
v1_pedestrian(27,1,:)=NaN;
v2_pedestrian(27,1,:)=NaN;
pc_pedestrian=100.*v1_pedestrian./v2_pedestrian; % percentage of participants looking at pedestrian
pc_session1_pedestrian=squeeze(mean(pc_pedestrian(:,1:4,:),2,'omitnan'));
pc_session2_pedestrian=squeeze(mean(pc_pedestrian(:,5:8,:),2,'omitnan'));
v1_session1_pedestrian=squeeze(mean(v1_pedestrian(:,1:4,:),2,'omitnan'));
v1_session2_pedestrian=squeeze(mean(v1_pedestrian(:,5:8,:),2,'omitnan'));
v2_session1_pedestrian=100*squeeze(mean(v2_pedestrian(:,1:4,:),2,'omitnan'));
v2_session2_pedestrian=100*squeeze(mean(v2_pedestrian(:,5:8,:),2,'omitnan'));

time=(-15*25:25)/25; % from -15 s to 1 s
figure;
plot(time,mean(pc_session1,'omitnan'),'k','Linewidth',3);
hold on
grid on;ax=gca; ax.LineWidth=1.5;
plot(time,mean(pc_session2,'omitnan'),'g','Linewidth',3)
xlabel('Time relative to passing the vehicle (s)')
ylabel('Percentage of trials in which participants looked at vehicle')
legend('Session 1 (no instruction)','Session 2 (instructed to detect driver presence/absence)','location','northwest')
set(gca,'xlim',[-10 0])
h=findobj('FontName','Helvetica'); set(h,'FontSize',24,'Fontname','Arial')
set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
%% Statistical tests - Vehicle
looking_at_vehicle = sign(sum(v1(:,:,126:376),3,'omitnan')); % from - 10 s to  0 %
looking_at_vehicle([7 22 28 32],:,:)=NaN;
looking_at_vehicle([25 29 38],5:8,:)=NaN;
looking_at_vehicle(27,1,:)=NaN;

lav_s1 = 100*mean(looking_at_vehicle(:,1:4),2,'omitnan'); % looking at vehicle in Session 1 (mean of 4 trials)
lav_s2 = 100*mean(looking_at_vehicle(:,5:8),2,'omitnan'); % looking at vehicle in Session 2 (mean of 4 trials)
disp(' ')
disp('Looked at vehicle Session 1 vs. Session 2 (binary variable, no or yes), p-value based on Wilcoxon signed-rank test')
p=signrank(lav_s1,lav_s2);
disp(p)
looking_at_vehicle_percentage = mean(pc(:,:,126:376),3,'omitnan');
looking_at_vehicle_percentage([7 22 28 32],:,:)=NaN;
looking_at_vehicle_percentage([25 29 38],5:8,:)=NaN;
looking_at_vehicle_percentage(27,1,:)=NaN;
lavp_s1 = mean(looking_at_vehicle_percentage(:,1:4),2,'omitnan'); % looking at vehicle in Session 1 (mean of 4 trials)
lavp_s2 = mean(looking_at_vehicle_percentage(:,5:8),2,'omitnan'); % looking at vehicle in Session 2 (mean of 4 trials)
disp('Looked at vehicle Session 1 vs. Session 2 (continuous variable, % of time), results of paired-samples t-test')
ttest_output(lavp_s1,lavp_s2,1);
disp(' ')
%% Figure 5. Angular distance between gaze point and vehicle for a single cyclist-AV encounter.
time=(-15*25:25)/25;
figure
plot(time,squeeze(angular_distance_to_car(35,3,:)),'k','Linewidth',3)
hold on
plot([-10 0],[4 4],'r:','Linewidth',3)
hold on
grid on;ax=gca; ax.LineWidth=1.5;
xlabel('Time relative to passing the vehicle (s)')
ylabel(['Angular distance (', char(176), ')'])
set(gca,'xlim',[-10 0])
h=findobj('FontName','Helvetica'); set(h,'FontSize',24,'Fontname','Arial')
set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
%% Figure B2. Proportion of trials in which the YOLO algorithm was able to detect the vehicle, as a function of elapsed time in the trial.
figure
plot(time,mean(v2_session1,'omitnan'),'k','Linewidth',3)
hold on
grid on;ax=gca; ax.LineWidth=1.5;
plot(time,mean(v2_session2,'omitnan'),'g','Linewidth',3)
xlabel('Time relative to passing the vehicle (s)')
ylabel('Percentage of trials in which YOLO identified vehicle')
legend('Session 1 (no instruction)','Session 2 (instructed to detect driver presence/absence)','location','southwest')
set(gca,'xlim',[-10 0])
h=findobj('FontName','Helvetica'); set(h,'FontSize',24,'Fontname','Arial')
set(gca,'LooseInset', [0.01 0.01 0.01 0.01])

figure
plot(time,mean(v2_session1_pedestrian,'omitnan'),'k','Linewidth',3)
hold on
grid on;ax=gca; ax.LineWidth=1.5;
plot(time,mean(v2_session2_pedestrian,'omitnan'),'g','Linewidth',3)
xlabel('Time relative to passing the pedestrian (s)')
ylabel('Percentage of trials in which YOLO identified pedestrian')
legend('Session 1 (no instruction)','Session 2 (instructed to detect driver presence/absence)','location','southwest')
set(gca,'xlim',[-10 0])
h=findobj('FontName','Helvetica'); set(h,'FontSize',24,'Fontname','Arial')
set(gca,'LooseInset', [0.01 0.01 0.01 0.01])
%% Create Screenshot example figures - vehicle
demo_video=0;
clear trial
if demo_video == 1
    yolofile_nr = find(yolofilestats(:,1) == 35 & yolofilestats(:,2) == 1 & yolofilestats(:,3) == 2); % find yolo file number that corresponds to participant number and trial number
% 
    %    load('Tobii 3 - Participant 35 Session 1.mat') %
    load(['Tobii 3 - Eye - Participant ' num2str(35) ' Session ' num2str(1) '.mat'])

    vidObj=VideoReader('P35 Session 1.mp4'); % 16830 frames
    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
    vr = VideoWriter('Test_video.mp4', 'Mpeg-4'); %#ok<*TNMLP>
    vr.Quality = 100; % store the video at maximal quality
    vr.FrameRate = 25; % and use a frame rate of 25 Hz
    open(vr);

    %    for frame_no=[4528:4703 7343:7518 10198:10373 13538:13713]
    for frame_no=7343:7518
        disp(frame_no)
        Fr = read(vidObj, frame_no); % video frame

        Gaze=Coordinates_store(frame_no,1:2);
        determine_angular_distance; % determines angular_distance_to_car_min, CX, and CY
        Fr = insertShape(Fr,'filledcircle',[Gaze 20],'color','red','linewidth',5,'opacity',1); % insert a circular shape for the gaze point
        if ~isnan(CX)
            Fr = insertShape(Fr,'filledcircle',[CX CY 20*ones(size(Gaze,1),1)],'color','blue','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            if angular_distance_to_car_min>=4 % if not looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_car_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','yellow','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            elseif angular_distance_to_car_min<4 % if looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_car_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom','BoxColor','green');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','green','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            end
        end
        if ~isnan(squeeze(CarBox(yolofile_nr,frame_no,1))')
            rm = insertObjectAnnotation(Fr, 'rectangle', squeeze(CarBox(yolofile_nr,frame_no,1:4))','Car','fontsize',35,'linewidth',3,'color','blue','textcolor','white'); % x y w h
        end
        if frame_no/50 == round(frame_no/50)
            imwrite(rm,['screenshot_frameno_' num2str(frame_no) '.png'])
        end
        writeVideo(vr,rm);

        %   figure
        %    imshow(rm)
    end

    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
end
%% Create Screenshot example figures - pedestrian
demo_video=0;
clear trial
if demo_video == 1
    yolofile_nr = find(yolofilestats(:,1) == 35 & yolofilestats(:,2) == 2 & yolofilestats(:,3) == 7); % find yolo file number that corresponds to participant number and trial number

    %    load('Tobii 3 - Participant 35 Session 1.mat') %
    load(['Tobii 3 - Eye - Participant ' num2str(35) ' Session ' num2str(2) '.mat'])

    vidObj=VideoReader('P35 Session 2.mp4'); % 16830 frames
    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
    vr = VideoWriter('Test_video.mp4', 'Mpeg-4'); %#ok<*TNMLP>
    vr.Quality = 100; % store the video at maximal quality
    vr.FrameRate = 25; % and use a frame rate of 25 Hz
    open(vr);

    %    for frame_no=[4528:4703 7343:7518 10198:10373 13538:13713]
    for frame_no=6900:7200
        disp(frame_no)
        Fr = read(vidObj, frame_no); % video frame

        Gaze=Coordinates_store(frame_no,1:2);

        % car
        determine_angular_distance; % determines angular_distance_to_car_min, CX, and CY
        Fr = insertShape(Fr,'filledcircle',[Gaze 20],'color','red','linewidth',5,'opacity',1); % insert a circular shape for the gaze point
        if ~isnan(CX)
            Fr = insertShape(Fr,'filledcircle',[CX CY 20*ones(size(Gaze,1),1)],'color','blue','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            if angular_distance_to_car_min>=4 % if not looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_car_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','yellow','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            elseif angular_distance_to_car_min<4 % if looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_car_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom','BoxColor','green');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','green','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            end
        end
        if ~isnan(squeeze(CarBox(yolofile_nr,frame_no,1))')
            rm = insertObjectAnnotation(Fr, 'rectangle', squeeze(CarBox(yolofile_nr,frame_no,1:4))','Car','fontsize',35,'linewidth',3,'color','blue','textcolor','white'); % x y w h
        end
        try
            Fr = rm;
        catch erorr
            Fr=Fr; %#ok<ASGSL> % take the original frame without additional markers
        end
        % pedestrian
        determine_angular_distance_pedestrian; % determines angular_distance_to_car_min, CX, and CY
        Fr = insertShape(Fr,'filledcircle',[Gaze 20],'color','red','linewidth',5,'opacity',1); % insert a circular shape for the gaze point
        if ~isnan(CX)
            Fr = insertShape(Fr,'filledcircle',[CX CY 20*ones(size(Gaze,1),1)],'color','blue','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            if angular_distance_to_pedestrian_min>=4 % if not looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_pedestrian_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','yellow','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            elseif angular_distance_to_pedestrian_min<4 % if looking at the car
                Fr = insertText(Fr,[ (CX+Gaze(1))/2 (CY+Gaze(2))/2], [sprintf('%0.0f',angular_distance_to_pedestrian_min) ' deg'],'fontsize',30,'anchorpoint','centerbottom','BoxColor','green');
                Fr = insertShape(Fr,'line',[CX CY Gaze],'color','green','linewidth',5,'opacity',1); % insert a small cyan circular shape for the identified dot
            end
        end
        if ~isnan(squeeze(PedestrianBox(yolofile_nr,frame_no,1))')
            rm = insertObjectAnnotation(Fr, 'rectangle', squeeze(PedestrianBox(yolofile_nr,frame_no,1:4))','Person','fontsize',35,'linewidth',3,'color','blue','textcolor','white'); % x y w h
        end
        if frame_no/25 == round(frame_no/25)
            imwrite(rm,['screenshot_pedestrian_frameno_' num2str(frame_no) '.png'])
        end
        writeVideo(vr,rm);

    end

    try
        close(vr) % close video in case it still happens to be open
    catch error
    end
end
%% Create figure of speeds
% 1. Walkway 1 (1) - Manhole 1 (2) (0 - 13.7 m)
% 2. Manhole 1 (2) - Start Freezone (3) (13.7 - 47.47 m)
% 3. Start FreeZone (3) - Manhole 2 (5) (47.47 - 63.61 m) This is where the letter task is
% 4. Manhole 2 (5) - End Freezone (7) (63.61 - 91.72 m) This is where the AV is
% 5. End FreeZone - Manhole 3 (91.72 - 113.71 m)
% 6. Manhole 3 - Walkway 2 (113.71 - 150.35 m)
% 7. Walkway 2 - Walkway 3 (150.35 - 170.57 m)
% 8. Walkway 3 - Manhole 4 (170.57 - 182.16 m)

distances=[13.7	33.77	16.14	28.11	21.99	36.64	20.22	11.59];
frame_numbers=landmarks(1:296,1:11);

disp('Mean time difference between letters and AV, in seconds')
disp(mean(frame_numbers(:,6)-frame_numbers(:,4),'omitnan')/24.9548317545243)

disp('Mean time difference between start freezone and AV, in seconds')
disp(mean(frame_numbers(:,6)-frame_numbers(:,3),'omitnan')/24.9548317545243)

v1=frame_numbers(:,4)-frame_numbers(:,3); % start freezone to letters
v2=frame_numbers(:,5)-frame_numbers(:,3); % start freezone to manhole 2
cumdistance_of_letters = distances(3)*v1./v2+sum(distances(1:2));

v1=frame_numbers(:,6)-frame_numbers(:,5); % manhole 2 to AV
v2=frame_numbers(:,7)-frame_numbers(:,5); % manhole 2 to end of freezone
cumdistance_of_AV = distances(4)*v1./v2+sum(distances(1:3));

disp('Mean distance between letters and AV (m)')
disp(mean(cumdistance_of_AV-cumdistance_of_letters,'omitnan'))

disp('Mean distance between start of freezone and AV (m)')
disp(mean(cumdistance_of_AV-sum(distances(1:2)),'omitnan'))

frame_numbers_diff=[frame_numbers(:,2)-frame_numbers(:,1) frame_numbers(:,3)-frame_numbers(:,2) frame_numbers(:,5)-frame_numbers(:,3) frame_numbers(:,7)-frame_numbers(:,5) frame_numbers(:,8)-frame_numbers(:,7) frame_numbers(:,9)-frame_numbers(:,8) frame_numbers(:,10)-frame_numbers(:,9) frame_numbers(:,11)-frame_numbers(:,10)];
frame_numbers_diff(frame_numbers_diff>10000)=NaN; % remove one outlier because the camera stopped
elapsed_times=frame_numbers_diff/24.9548317545243; % 296 x 8 matrix of elapsed_times
speeds=3.6*repmat(distances,size(elapsed_times,1),1)./elapsed_times; % 296 x 8 matrix of speeds
speeds(:,end+1)=3.6*sum(distances)./sum(elapsed_times,2); % add a 9th column containing the overall mean mean over the 8 segments
distancescum=cumsum(distances);

% "manhole 2, car, freezone"
Speed=NaN(41,8,9); % 41 participants x 8 trials x 8 speed measurements
Laptime=NaN(41,6); % 41 participants x 6 lap-times (trial 1-2, trial 2-3, trial 3-4, trial 5-6, trial 6-7, trial 7-8
for participant_number=1:41 % loop over participants
    for session_number=1:2 % loop over sessions
        if session_number == 1
            Trials = 1:4;
        elseif session_number == 2
            Trials = 5:8;
        end
        for trial = Trials % loop over trials
            landmark_nr = find(Landmarks_participant_number == participant_number & Landmarks_trial_number == trial,3); % find landmark number that corresponds to participant number and trial number
            if ~isempty(landmark_nr)
                Speed(participant_number,trial,:) = speeds(landmark_nr,:);
            end
        end
        laptime = diff(landmarks(Landmarks_participant_number == participant_number,6))/24.9548317545243;
        if ~isempty(laptime)
            Laptime(participant_number,1:6)=laptime([1:3 5:7]);
        end
    end
end
%%
maxspeed = max(Speed,[],3,'omitnan');
maxspeed([7 22 28 32],:,:)=NaN;
maxspeed([25 29 38],5:8,:)=NaN; % Session 2 missing or 3 participants
maxspeed(27,1,:)=NaN;
maxspeed_s1 = mean(maxspeed(:,1:4),2,'omitnan'); % looking at car in Session 1 (mean of 4 trials)
maxspeed_s2 = mean(maxspeed(:,5:8),2,'omitnan'); % looking at car in Session 2 (mean of 4 trials)
disp('Maximum speed of Session 1 vs. Session 2 (km/h)')
ttest_output(maxspeed_s1,maxspeed_s2,1);

disp(['The average time to complete a lap was ' sprintf('%0.1f',mean(mean(Laptime,2,'omitnan'),'omitnan')) ' s'])
disp(['The corresponding standard deviation accross participants was ' sprintf('%0.1f',std(mean(Laptime,2,'omitnan'),'omitnan')) ' s'])
%% Figure 9 left
opengl('save','software') %#ok<OPGLO>

disp('Correlation coefficient and p-value pertaining to Figure 9 left (Maximum speed)')
[c,p]=corr(maxspeed_s1,maxspeed_s2,'rows','pairwise');
disp([c p])
figure;hold on;grid on;box on;ax=gca; ax.LineWidth=1.5;
subplot(1,2,1)
plot([0 100],[0 100],'k:','Linewidth',3);hold on
for i=1:41 % loop over 36 participants
    if AgeGender(i,2) == 1 % female
        scatter1=scatter(maxspeed_s1(i),maxspeed_s2(i),700,'markerfacecolor','m','markeredgecolor','k'); % plot marker
    elseif AgeGender(i,2) == 2 % male
        scatter1=scatter(maxspeed_s1(i),maxspeed_s2(i),700,'markerfacecolor','b','markeredgecolor','k'); % plot marker
    end
    hold on;
    scatter1.MarkerFaceAlpha = .4;
end
grid on;ax=gca; ax.LineWidth=1.5;
set(gcf,'innerposition',[1 49 1000 1000])
set(gca,'pos',[0.10 0.12 0.865 0.865],'xlim',[11 27], 'ylim',[11 27],'xtick',0:3:100,'ytick',0:3:100)
xlabel('Maximum speed - Session 1 (km/h)')
ylabel('Maximum speed - Session 2 (km/h)')
h=findobj('FontName','Helvetica');
set(h,'FontSize',28,'Fontname','Arial')
%% Figure 9 right
minspeed = min(Speed,[],3,'omitnan');
minspeed = minspeed./maxspeed; % relative minimum speed
minspeed([7 22 28 32],:,:)=NaN;
minspeed([25 29 38],5:8,:)=NaN;
minspeed(27,1,:)=NaN;
minspeed_s1 = mean(minspeed(:,1:4),2,'omitnan'); % looking at car in Session 1 (mean of 4 trials)
minspeed_s2 = mean(minspeed(:,5:8),2,'omitnan'); % looking at car in Session 2 (mean of 4 trials)
disp(newline)
disp('Relative minimum speed')
ttest_output(minspeed_s1,minspeed_s2,3);

disp('Relative minimum speed of the four trials of Session 1')
disp(round(mean(minspeed(:,1:4),'omitnan'),3))
disp('Relative minimum speed of the four trials of Session 2')
disp(round(mean(minspeed(:,5:8),'omitnan'),3))

disp('Correlation coefficient and p-value pertaining to Figure 9 right (Relative minimum speed)')
[c,p]=corr(minspeed_s1,minspeed_s2,'rows','pairwise');
disp([c p])
figure;hold on;grid on;box on;ax=gca; ax.LineWidth=1.5;
subplot(1,2,1)
plot([0 100],[0 100],'k:','Linewidth',3);hold on
for i=1:41 % loop over 36 participants
    if AgeGender(i,2) == 1 % female
        scatter1=scatter(minspeed_s1(i),minspeed_s2(i),700,'markerfacecolor','m','markeredgecolor','k'); % plot marker
    elseif AgeGender(i,2) == 2 % male
        scatter1=scatter(minspeed_s1(i),minspeed_s2(i),700,'markerfacecolor','b','markeredgecolor','k'); % plot marker
    end
    hold on;
    scatter1.MarkerFaceAlpha = .4;
end
grid on;ax=gca; ax.LineWidth=1.5;
set(gcf,'innerposition',[1 49 1000 1000])
set(gca,'pos',[0.13 0.11 ...
    0.865 0.865],'xlim',[0.69 0.95], 'ylim',[0.69 0.95],'xtick',0:0.05:100,'ytick',0:.05:100)
xlabel('Relative minimum speed - Session 1')
ylabel('Relative minimum speed - Session 2')
h=findobj('FontName','Helvetica');
set(h,'FontSize',28,'Fontname','Arial')
%% Estimate distances corresponding to bounding box heights
BBHeightLandMark=NaN(41,8,5);
clear T
for i=1:41 % loop over participants
    for s=1:8 % loop over sessions
        landmark_nr = find(Landmarks_participant_number == i & Landmarks_trial_number == s); % find landmark number that corresponds to participant number and trial number
        walkway1=landmarks(landmark_nr,6)-landmarks(landmark_nr,1);
        manhole1=landmarks(landmark_nr,6)-landmarks(landmark_nr,2);
        start_freezone=landmarks(landmark_nr,6)-landmarks(landmark_nr,3); % number of frames before passing vehicle
        letters=landmarks(landmark_nr,6)-landmarks(landmark_nr,4);
        manhole2=landmarks(landmark_nr,6)-landmarks(landmark_nr,5);

        if size(StoreTrialInfo,3)-walkway1-25>0
            BBHeightLandMark(i,s,1)=StoreTrialInfo(i,s,end-25-walkway1,4); % bounding box height for walkway 1
        end
        if size(StoreTrialInfo,3)-manhole1-25>0
            BBHeightLandMark(i,s,2)=StoreTrialInfo(i,s,end-25-manhole1,4); % bounding box height for manhole 1 - 87.93 m to car
        end
        if size(StoreTrialInfo,3)-start_freezone-25>0
            BBHeightLandMark(i,s,3)=StoreTrialInfo(i,s,end-25-start_freezone,4); % bounding box height for start of freezone - 54.70 m to car
        end
        if size(StoreTrialInfo,3)-letters-25>0
            BBHeightLandMark(i,s,4)=StoreTrialInfo(i,s,end-25-letters,4); % bounding box height for letters - 28.30 m to car
        end
        if size(StoreTrialInfo,3)-manhole2-25>0
            BBHeightLandMark(i,s,5)=StoreTrialInfo(i,s,end-25-manhole2,4); % bounding box height for manhole2 - 31.30 m to car
        end
    end
end

d=[17.6896202450707 27.4774401964446 50.4534496127851 300  % mean bounding box height for manhole 1, start of freezone, manhole 2
    88.1             54.47            31.59            5]'; % corresponding distances to the AV
%figure;plot(d(:,1),d(:,2),'k-o')
%% Figure 7, Figure 8.
% Heatmaps of gaze distribution relative to the bounding box surrounding the vehicle, for different distances to the vehicle, for Session 1 (Figure 7) and Session 2 (Figure 8)
heightprctiles=[9999 60:-5:0];

N=NaN(2,length(heightprctiles)-1);
looking_at_vehicle_percentage_distance=NaN(41,2,length(heightprctiles)-1);
participant_number = permute(repmat((1:41)', 1, 8, 401), [1, 2, 3]);
for session = 1:2 % loop over the two sessions
    if session == 1
        % x, y, w, h, gazex, gazey
        StoreTrialInfoReshape=reshape(StoreTrialInfo(:,1:4,:,:),[41*4*401,6]);
        angular_distance_to_carReshape = reshape(angular_distance_to_car(:,1:4,:),[41*4*401,1]);
        participant_numberReshape = reshape(participant_number(:,1:4,:),[41*4*401,1]);
    else
        StoreTrialInfoReshape=reshape(StoreTrialInfo(:,5:8,:,:),[41*4*401,6]);
        angular_distance_to_carReshape = reshape(angular_distance_to_car(:,5:8,:),[41*4*401,1]);
        participant_numberReshape = reshape(participant_number(:,5:8,:),[41*4*401,1]);
    end
    figure
    nnan = ~isnan(sum(StoreTrialInfoReshape,2));
    StoreTrialInfoReshape=StoreTrialInfoReshape(nnan,:); % only retain rows which do not contain NaNs
    angular_distance_to_carReshape=angular_distance_to_carReshape(nnan,:); % only retain rows which do not contain NaNs
    participant_numberReshape=participant_numberReshape(nnan,:); % only retain rows which do not contain NaNs

    DistanceToVehicle=1550./StoreTrialInfoReshape(:,4); % convert to height
    tiledlayout(4,3,'tilespacing','tight','padding','tight');
    for i=2:length(heightprctiles)-1

        idx = find(DistanceToVehicle<heightprctiles(i) & DistanceToVehicle > heightprctiles(i+1)); % find indices that correspond to Car box height
        whratio= StoreTrialInfoReshape(idx,3)./StoreTrialInfoReshape(idx,4); % weight height ratio of the bounding boxes
        idx(whratio > 1.9 | whratio < 1 ) = []; % remove anomalous bounding boxes

        CX=StoreTrialInfoReshape(idx,1)+StoreTrialInfoReshape(idx,3)*0.5; % center of car x
        CY=StoreTrialInfoReshape(idx,2)+StoreTrialInfoReshape(idx,4)*0.5; % center of car y
        GDX=StoreTrialInfoReshape(idx,5)-CX; % gaze point deviation from bounding box center in x direction
        GDY=StoreTrialInfoReshape(idx,6)-CY; % gaze point deviation from bounding box center in y direction
        N(session,i)=length(GDX);
        for p=1:41
            idxr = idx(participant_numberReshape(idx)==p); % indices of participant p;
            looking_at_vehicle_percentage_distance(p,session,i) = mean(angular_distance_to_carReshape(idxr)<4);
        end
        nexttile
        if ismember(i,[4 7 10 13])
            colorbaradded=1;
        else
            colorbaradded=0;
        end

        createHeatmaps(GDX,GDY,colorbaradded)
        mw=mean(StoreTrialInfoReshape(idx,3)); % mean width in pixels
        mh=mean(StoreTrialInfoReshape(idx,4)); % mean height in pixels
        mhm=mean(DistanceToVehicle(idx)); % mean height in meters
        hold on

        plot3([-mw/2 mw/2],[mh/2 mh/2],[1000,1000],'k:','Linewidth',2);hold on      % lower line of the box
        plot3([-mw/2 mw/2],[-mh/2 -mh/2],[1000,1000],'k:','Linewidth',2); hold on % upper line of the box
        plot3([mw/2 mw/2],[-mh/2 mh/2],[1000,1000],'k:','Linewidth',2);hold on    % right line of the box
        plot3([-mw/2 -mw/2],[-mh/2 mh/2],[1000,1000],'k:','Linewidth',2)          % left line of the box
        text(610,-275,2000, ['Session: ' num2str(session)],'color','w','fontsize',16)
        text(610,-225,2000, ['Distance: ' sprintf('%0.0f',heightprctiles(i)) '–' sprintf('%0.0f',heightprctiles(i+1)) ' m'],'color','w','fontsize',16)
        text(610,-175,2000, ['Gaze data: ' sprintf('%0.1f',N(session,i)/25) ' s'],'color','w','fontsize',16)
    end
    set(gcf, 'Color', 'w');
    set(gca, 'Color', 'k');
    set(gcf, 'InvertHardcopy', 'off');
end
disp(newline)
disp('Percentage looking at vehicle 50-45 m')
ttest_output(squeeze(100*looking_at_vehicle_percentage_distance(:,1,4)),100*squeeze(looking_at_vehicle_percentage_distance(:,2,4)),1);
disp('Percentage looking at vehicle 10-5 m')
ttest_output(squeeze(100*looking_at_vehicle_percentage_distance(:,1,12)),100*squeeze(looking_at_vehicle_percentage_distance(:,2,12)),1);