import numpy as np
import matplotlib.pyplot as plt
import os
import matplotlib as mpl

mpl.rcParams.update({
    "text.usetex": False,                 # disable external LaTeX
    "font.family": "serif",
    "font.serif": ["Times New Roman"],    # use TNR for text
    "font.size": 24,
    "mathtext.fontset": "stix",           # STIX = Times-like math
    "mathtext.rm":    "Times New Roman",  # roman math = TNR
    "mathtext.it":    "Times New Roman:italic"  # italic math = TNR italic
})


filename = 'X120_Y120_dx2_dy2_Xc-30_Yc-30_f150mm_DMD100Hz_output_2D_array.npy'
data = np.load(filename)

import numpy as np
import matplotlib.pyplot as plt

img = data[1,:,:]
hgt, wdt = img.shape

pixel_size = 0.363  # m per pixel
x = np.arange(wdt) * pixel_size
y = np.arange(hgt) * pixel_size
X, Y = np.meshgrid(x, y)

deg = 5
# exponents (i,j) with i+j <= deg
exponents = [(i, j)
             for i in range(deg+1)
             for j in range(deg+1-i)]

# each column is (X**i * Y**j).ravel()
A = np.vstack([ (X**i * Y**j).ravel()
                for i, j in exponents ]).T

Z = img.ravel()

# coeffs shape = (num_basis,)
coeffs, *_ = np.linalg.lstsq(A, Z, rcond=None)

# fitted surface
Z_fit = (A @ coeffs).reshape(hgt, wdt)

# residual
residual = img - Z_fit

plt.figure(figsize=(5,5))
plt.imshow(residual, cmap='gray', extent=[0, img.shape[1]*0.363, 0, img.shape[0]*0.363])
plt.xlabel(r'$\mathit{x}$, m')
plt.ylabel(r'$\mathit{x}$, m')
plt.colorbar()
plt.savefig('1um_PSL.svg')
plt.show()


# Calculate background signal
plt.figure()
plt.imshow(residual[:20, :])
plt.show()

background = residual[:20, :]

background = background - np.mean(background)
background.max() - background.min()

import os
import numpy as np
import matplotlib.pyplot as plt
from data_analysis import create2D
from scipy.ndimage import label, center_of_mass
from matplotlib.patches import Circle

def circular_max_min(npy_array, x, y, r):
    """
    Computes max - min for slices [0,:,:] and [1,:,:] in a circular region
    centered at (x, y) with radius r.
    """
    h, w = npy_array.shape[1], npy_array.shape[2]
    Y, X = np.ogrid[:h, :w]

    # Create circular mask
    mask = (X - x)**2 + (Y - y)**2 <= r**2

    results = {}
    for idx in [0, 1]:
        slice_data = npy_array[idx, :, :]
        region = slice_data[mask]
        if region.size == 0:
            results[f'SD_{idx}'] = None
        else:
            results[f'SD_{idx}'] = np.max(region) - np.min(region)

    return results

def analyze_contamination_map(npy_array, minThreshold=1.0, maxThreshold=1.0, max_cluster_size=30, radius=5, xmin=0, xmax=-1, ymin=0, ymax=-1):
    contamination_map = npy_array[2, :, :]  # Slice used for detection

    # Intensity thresholding
    threshold = np.mean(contamination_map) + minThreshold * np.std(contamination_map)
    binary_mask = (contamination_map > threshold) & (contamination_map < (maxThreshold * np.max(contamination_map)))
    binary_mask[:, :xmin] = 0
    binary_mask[:, xmax:] = 0
    binary_mask[:ymin, :] = 0
    binary_mask[ymax:, :] = 0

    # Label connected regions
    labeled_array, num_features = label(binary_mask)

    results = []
    for i in range(1, num_features + 1):
        region_mask = labeled_array == i
        region_size = np.sum(region_mask)

        if region_size > max_cluster_size:
            continue

        region_intensity = contamination_map[region_mask]
        peak_intensity = np.max(region_intensity)
        y, x = center_of_mass(region_mask)
        results.append({
            'x': x,
            'y': y,
            'peak_intensity': peak_intensity,
            'size': region_size,
            'distance': np.sqrt(x**2 + y**2),
            'peak-peak_0': circular_max_min(npy_array, x, y, radius)['SD_0'],
            'peak-peak_1': circular_max_min(npy_array, x, y, radius)['SD_1']
        })

    # Plot all three slices with circles
    slice_titles = ['Slice 0', 'Slice 1', 'Contamination Map (Slice 2)']
    fig, axes = plt.subplots(1, 3, figsize=(18, 6))

    for i in range(3):
        ax = axes[i]
        ax.imshow(npy_array[i, :, :], cmap='gray')
        ax.set_title(slice_titles[i])
        ax.set_aspect('equal')
        ax.set_xlim(0, npy_array.shape[2])
        ax.set_ylim(npy_array.shape[1], 0)

        for r in results:
            circle = Circle((r['x'], r['y']), radius=radius, fill=False, edgecolor='cyan', linewidth=1)
            ax.add_patch(circle)
            if i == 2:  # Label only on main detection slice
                ax.text(r['x'] + 1, r['y'], f"{r['peak_intensity']:.1f}", color='white', fontsize=6)

    plt.suptitle(f"Detected Particles: {len(results)} (max size: {max_cluster_size})")
    plt.tight_layout()
    plt.show()

    # Print results
    for i, r in enumerate(results, 1):
        print(f"Particle {i}: x={r['x']:.2f}, y={r['y']:.2f}, peak_intensity={r['peak_intensity']:.2f}, "
              f"size={r['size']}, distance={r['distance']:.2f}, peak-peak_0={r['peak-peak_0']:0.3f}, peak-peak_1={r['peak-peak_1']:0.3f}")

    return results

particles = analyze_contamination_map(data - Z_fit, 1.9, 1., 25, radius=5, xmin=5, xmax=50, ymin=15, ymax=40)

