import numpy as np
import matplotlib.pyplot as plt
import os

# load
folder = r'\data\'
xc = 40
yc = 0
corr = np.load(os.path.join(folder, f'xc{xc}_yc{yc}_corrected_cam.npy'))
ref  = np.load(os.path.join(folder, f'xc{xc}_yc{yc}_reference_cam.npy'))

# parameters
half = 10
norm = np.max(corr)

# corrected
iy0, ix0 = np.unravel_index(np.argmax(corr), corr.shape)
ix0 += 1
y0min, y0max = max(0, iy0-half), min(corr.shape[0], iy0+half)
x0min, x0max = max(0, ix0-half), min(corr.shape[1], ix0+half)
corr = corr / norm
yslice_corr = corr[iy0, x0min:x0max]      # X profile (horizontal)
xslice_corr = corr[y0min:y0max, ix0]      # Y profile (vertical)
patch_corr = corr[y0min:y0max, x0min:x0max]

# reference
iy1, ix1 = np.unravel_index(np.argmax(ref), ref.shape)
y1min, y1max = max(0, iy1-half), min(ref.shape[0], iy1+half)
x1min, x1max = max(0, ix1-half), min(ref.shape[1], ix1+half)
ref = ref / norm
yslice_ref = ref[iy1, x1min:x1max]        # X profile (horizontal)
xslice_ref = ref[y1min:y1max, ix1]        # Y profile (vertical)
patch_ref = ref[y1min:y1max, x1min:x1max]

pixel_size = 3.5  # m per pixel
N = xslice_corr.shape[0]
distance = np.arange(N) * pixel_size

# stack into shape (N, 5): distance, xslice_corr, xslice_ref, yslice_corr, yslice_ref
data = np.vstack([
    distance,
    xslice_corr,
    xslice_ref,
    yslice_corr,
    yslice_ref
]).T

np.savetxt(
    'focus_spot_slice_after_correction.csv',
    data,
    delimiter=',',
    header='distance_um,xslice_corr,xslice_ref,yslice_corr,yslice_ref',
    comments=''
)

# plot patches with crosshairs
fig = plt.figure(figsize=(4,4))
ax_ref  = plt.subplot(1,2,1)
ax_corr = plt.subplot(1,2,2)
linewidth = 3
ax_ref.imshow(patch_ref, cmap='gray')
ax_ref.axvline(ix1 - x1min, color='red',  linestyle='--', linewidth=linewidth)  # x-line (vertical) dotted
ax_ref.axhline(iy1 - y1min, color='blue', linestyle='--', linewidth=linewidth)  # y-line (horizontal) dotted
ax_ref.axis('off'); ax_ref.set_title('original')

ax_corr.imshow(patch_corr, cmap='gray')
ax_corr.axvline(ix0 - x0min, color='red',  linestyle='-', linewidth=linewidth)  # x-line solid
ax_corr.axhline(iy0 - y0min, color='blue', linestyle='-', linewidth=linewidth)  # y-line solid
ax_corr.axis('off'); ax_corr.set_title('corrected')

fig.savefig('focus_spots.svg', format='svg', transparent=True, bbox_inches='tight', pad_inches=0)

# X profile (horizontal through peak) -> red
plt.figure()
plt.plot(yslice_corr, 'r-', label='corr X')
plt.plot(yslice_ref,  'r:', label='ref X')
plt.legend(); plt.title('X profile'); plt.xlabel('pixels'); plt.ylabel('norm. intensity')

# Y profile (vertical through peak) -> blue
plt.figure()
plt.plot(xslice_corr, 'b-', label='corr Y')
plt.plot(xslice_ref,  'b:', label='ref Y')
plt.legend(); plt.title('Y profile'); plt.xlabel('pixels'); plt.ylabel('norm. intensity')

plt.show()
