import numpy as np
import matplotlib.pyplot as plt
import os

def fill_missing_by_nn(a, missing_is_zero=False, max_iters=200):
    """
    Fill missing values (NaNs by default) by averaging nearest valid neighbors.
    Works iteratively so corner blocks get filled from the inside out.
    """
    a = np.array(a, dtype=float).copy()
    if missing_is_zero:
        a[a == 0] = np.nan

    nan_mask = np.isnan(a)
    if not np.any(nan_mask):
        return a  # nothing to do

    for _ in range(max_iters):
        prev_mask = nan_mask.copy()
        sums   = np.zeros_like(a, dtype=float)
        counts = np.zeros_like(a, dtype=float)

        # 8-neighborhood
        for dy, dx in [(-1,0),(1,0),(0,-1),(0,1),(-1,-1),(-1,1),(1,-1),(1,1)]:
            s = np.roll(a, shift=(dy, dx), axis=(0, 1))
            # prevent wrap-around contributions
            if dy == -1: s[-1, :] = np.nan
            if dy ==  1: s[ 0, :] = np.nan
            if dx == -1: s[:, -1] = np.nan
            if dx ==  1: s[:,  0] = np.nan

            valid = ~np.isnan(s)
            sums[valid]   += s[valid]
            counts[valid] += 1

        fill_here = nan_mask & (counts > 0)
        a[fill_here] = sums[fill_here] / counts[fill_here]
        nan_mask = np.isnan(a)

        # stop if no progress
        if np.array_equal(nan_mask, prev_mask) or not np.any(nan_mask):
            break

    return a

xc_list = np.arange(-200, 201, 25)
yc_list = np.arange(-100, 101, 25)

folder = r'data'

modes = [(2, 0), (2, 2), (2, -2), (3, 1), (3, -1), (3, 3), (3, -3), (4, 0), (4, 2), (4, -2), (5, 1), (5, -1), (5, 3), (5, -3)]
zernike_coeffs_after_opt = np.zeros((len(yc_list), len(xc_list), len(modes)))

for i, yc in enumerate(yc_list):
    for j, xc in enumerate(xc_list):
        filename = f"xc{xc}_yc{yc}_final_coeffs.npy"
        try:      
            data = np.load(os.path.join(folder, filename), allow_pickle=True).item()
            for k, (n, m) in enumerate(modes):
                zernike_coeffs_after_opt[i,j,k] = data[(n, m)]
        except:
            zernike_coeffs_after_opt[i,j,:] = np.nan
            
zernike_coeffs_after_opt[3,-3,:] = np.nan
zernike_coeffs_after_opt[-3,0,:] = np.nan
zernike_coeffs_after_opt = fill_missing_by_nn(zernike_coeffs_after_opt)

plt.figure()
plt.imshow(zernike_coeffs_after_opt[:,:,0])
plt.xticks([])
plt.yticks([])
plt.colorbar()
plt.savefig('Figure_12a.svg')
plt.show()

plt.figure()
plt.imshow(zernike_coeffs_after_opt[:,:,1])
plt.xticks([])
plt.yticks([])
plt.colorbar()
plt.savefig('Figure_12b.svg')
plt.show()

plt.figure()
plt.imshow(zernike_coeffs_after_opt[:,:,2])
plt.xticks([])
plt.yticks([])
plt.colorbar()
plt.savefig('Figure_12c.svg')
plt.show()

