function [cone_sum, dTh, dPh] = sph2surf(Image, method, Nsecs, Apsp, Nrads, varargin)

%%

sz = size(Image);

if numel(varargin)>0
    scaling = varargin{1};
    Image = ellip2sph(Image, 0, 'SameMinor', sz/2, scaling);
end

%%
all_theta = acos(1 - 2*(1:Nsecs)./Nsecs);   % Compute angles for each Fibonacci number
all_phi = rem(pi * (1 + sqrt(5)) * (1:Nsecs), 2*pi);

dTh = diff(all_theta);
dTh = [dTh(end) dTh];
dPh = diff(all_phi);
dPh = [dPh(1) dPh];
dPh(dPh<0) = dPh(dPh<0)+2*pi;

switch method
    case 'cones'
        %% Method used in publication

        Rmax =  floor(sz/2)+0.5*(1-rem(sz,2));

        Asec = Apsp/Nsecs;                      % Surface area per section
        theta_cap = acos(1-Asec/(2*pi));
        
        [xcoords, ycoords, zcoords] = ndgrid(-Rmax(1):Rmax(1), -Rmax(2):Rmax(2), -Rmax(3):Rmax(3));
        
        pol_coords = car2sph([xcoords(:), ycoords(:), zcoords(:)]); % [rr, theta, phi]
        R_coords = pol_coords(:,1);
        theta_coords = pol_coords(:,2);
        phi_coords = pol_coords(:,3);
        
        idx_cone = find(all([theta_coords<theta_cap, R_coords<=min(Rmax)],2));
        cart_cone = [xcoords(idx_cone), ycoords(idx_cone), zcoords(idx_cone)];
        cone_sum = zeros(Nsecs, 3);

        for aa = 1:Nsecs

            rot_coords = rot3d(cart_cone, 1, [all_theta(aa), 0, all_phi(aa)]);
            rot_coords(sqrt(sum(rot_coords.^2,2))>min(Rmax),:) = [];

            rot_coords = rot_coords+Rmax+1;

            floorCoords = min(max(floor(rot_coords),1), sz);
            ceilCoords = min(max(ceil(rot_coords),1), sz);
            modCoords = mod(rot_coords,1);
            remmodCoords = 1-modCoords;

            intCoords = [sub2ind(size(Image),floorCoords(:,1), floorCoords(:,2), floorCoords(:,3)),...
                sub2ind(size(Image),floorCoords(:,1), floorCoords(:,2), ceilCoords(:,3)),...
                sub2ind(size(Image),floorCoords(:,1), ceilCoords(:,2), floorCoords(:,3)),...
                sub2ind(size(Image),floorCoords(:,1), ceilCoords(:,2), ceilCoords(:,3)),...
                sub2ind(size(Image),ceilCoords(:,1), floorCoords(:,2), floorCoords(:,3)),...
                sub2ind(size(Image),ceilCoords(:,1), floorCoords(:,2), ceilCoords(:,3)),...
                sub2ind(size(Image),ceilCoords(:,1), ceilCoords(:,2), floorCoords(:,3)),...
                sub2ind(size(Image),ceilCoords(:,1), ceilCoords(:,2), ceilCoords(:,3))];

            CoordVals = [remmodCoords(:,1).* remmodCoords(:,2).* remmodCoords(:,3),...
                remmodCoords(:,1).* remmodCoords(:,2).* modCoords(:,3),...
                remmodCoords(:,1).* modCoords(:,2).* remmodCoords(:,3),...
                remmodCoords(:,1).* modCoords(:,2).* modCoords(:,3),...
                modCoords(:,1).* remmodCoords(:,2).* remmodCoords(:,3),...
                modCoords(:,1).* remmodCoords(:,2).* modCoords(:,3),...
                modCoords(:,1).* modCoords(:,2).* remmodCoords(:,3),...
                modCoords(:,1).* modCoords(:,2).* modCoords(:,3)];


            newvals = sum(Image(intCoords).*CoordVals,2);

            rot_coords = round(rot_coords);
            rot_coords(any([rot_coords<1, rot_coords>sz],2),:) = [];

            cone_sum(aa,:) = [sum(newvals), all_theta(aa), all_phi(aa)];

        end

    case 'no_overlap'
        %% Alternative method

        Rmax = min((sz-1)/2);
        rads = Rmax/Nrads*(1:Nrads);

        [gridTheta, ~] = meshgrid(all_theta, rads);          % Spherical coordinates grid
        [gridPhi, gridR] = meshgrid(all_phi, rads);

        gridcar = sph2car([gridR(:), gridTheta(:), gridPhi(:)]);    % Spherical coordinates grid parametrised in cartesian coordinates
        gridcar = gridcar + floor(sz/2);
        gridIm = get_subpixel(Image, [gridcar(:,1), gridcar(:,2), gridcar(:,3)], 'cubic');  % Image resampled in spherical coordinates grid

        gridImRes = reshape(gridIm, Nrads, Nsecs);
        gridImSum = squeeze(sum(gridImRes.*(rads.^2)' ,1));      % Sum over radii and correct for different #pixels per radius

        cone_sum(:,1) = gridImSum;
        cone_sum(:,2) = all_theta;
        cone_sum(:,3) = all_phi;
end

%% Plotting

% max5 = max(Image,[],'all');
% max3 = -min(Image,[],'all');
% figure();
% subplot(1,2,1)
% isoplot(Image, max5*0.7,'red',0.5);
% hold on
% isoplot(-Image, max3*0.7,'cyan',0.5);
% view(-210,30)
%
% x = sin(cone_sum(:,2,1)) .* cos(cone_sum(:,3,1));
% y = sin(cone_sum(:,2,1)) .* sin(cone_sum(:,3,1));
% z = cos(cone_sum(:,2,1));
% k = convhull([x,y,z]);
%
% subplot(1,2,2)
% trisurf(k,x, y, z, cone_sum(:,1,1));
% % xlabel('X');
% % ylabel('Y');
% % zlabel('Z');
% axis equal
% shading interp; % Smooth shading
% colormap jet; % Choose a colormap (e.g., jet)
% view(-210,30)
% hold on

end