%{
Automated rotational alignment and axis estimation of ellipsoid objects
Requires DIPImage (https://diplib.org/), Parallel Computing Toolbox, Signal Processing Toolbox

Functions: 
            dipstart
            CentreIm
            AxesRatio
            estimatebg
            ellip2sph
            imrot
            PSP_Align

Desired folder structure: head folder (eg date_dimensions) > labeled position (eg 1000to3000) > colour (Orange or Red) > images


Copyright Enya Berrevoets, TU Delft, 2024
Licensed under the Apache License, Version 2.0
%}
%% Initialise
close all
clear

addpath(genpath(pwd))

[message] = PSP_CodeCheck();
if ~isempty(message)
    return
end

dipstart
%% User input

projDir = 'Demo';        % Directory with the raw images
locDir = 'Demo';      % Directory to store rotated images
   
headDir = 'Demo_data';        % Folder with desired image set

savedata = 1;             % Save the resulting data [1=yes, 0=no]
runcol = [1,1,1];           % Colour channels to rotate [Orange, Red, Sum]
sellocs = 1;              % Selected probe locations [depends on data set]       

Npar = 2;              % Number of core to use in parallelisation

Vthr = 0.5;             % Intensity threshold relative to image max for including pixel in volume estimation

pcaCol = 'Sum';        % Perform PCA on 'Orange'/'Red'/'Both'/'Sum'

%%
    
[locs, Nlocs, leg] = PSP_getLocs([projDir  '\' headDir]);

%% Check if cell array already exist to add to

loaddir = [locDir  '\' headDir '\PCA_' pcaCol];    

fileNames = {'Orange_cellvars', 'Red_cellvars', 'Sum_cellvars', 'Orange_cellints', ...
             'Red_cellints', 'cellRvol', 'cellOutbox', 'cellVol'};
cellNames = {'cellVarA', 'cellVarB', 'cellVarSum', 'cellIntA', ...
            'cellIntB', 'cellRvol', 'cellOutbox', 'cellVol'};

for i = 1:numel(fileNames)
    filePath = fullfile(loaddir, [fileNames{i} '.mat']);
    if exist(filePath, 'file')
        loadedData = load(filePath);
        eval([cellNames{i} ' = loadedData.' cellNames{i} ';']);
    end
end

%%
for ll = sellocs
    
    OrangeDir = dir([projDir  '\' headDir '\' locs(ll).name '\Orange']);
    RedDir = dir([projDir  '\' headDir '\' locs(ll).name '\Red']);
    
    ImNames = struct2table(OrangeDir);
    ImNames = ImNames.name(3:end);
    
    im1 = tiffreadVolume([projDir  '\' headDir '\' locs(ll).name '\Orange\' ImNames{1}]);
    sz = size(im1);
    
    Nims = size(ImNames,1);
    
    if runcol(1)
        StackA = zeros(sz(1), sz(2),sz(3),Nims);
        VarA = zeros(Nims,3);
        IntA = zeros(Nims,1);
    end

    if runcol(2)
        StackB = zeros(sz(1), sz(2),sz(3),Nims);
        VarB = zeros(Nims,3);
        IntB = zeros(Nims,1);
    end

    if runcol(3)
        StackSum = zeros(sz(1), sz(2),sz(3),Nims);
        VarSum = zeros(Nims,3);
    end
    
    Rvol = zeros(Nims,3);
    Vol = zeros(Nims,1);
    outbox = zeros(Nims,1);
%%

    if Npar==1
        parSegs = [1,Nims+1];
    else
        parSegs = [round(linspace(1,Nims+1,Npar+1))];
    end

    for ps = 1:Npar
        LS{ps} = parSegs(ps):(parSegs(ps+1)-1);
    end

    VarA_par = cell(Npar,1);
    VarB_par = cell(Npar,1);
    VarSum_par = cell(Npar,1);

    IntA_par = cell(Npar,1);
    IntB_par = cell(Npar,1);

    StackA_par = cell(Npar,1);
    StackB_par = cell(Npar,1);
    StackSum_par = cell(Npar,1);

    Rvol_par = cell(Npar,1);
    Vol_par = cell(Npar,1);
    outbox_par = cell(Npar,1);

    %%
    parfor ps = 1:Npar
        pp = 'C:\Program Files\DIPimage 2.9\common\dipimage';
        addpath(pp);
        dip_initialise;
        count = 0;
        if ps==1
        msg = fprintf('%.2f %% of position %d/%d', round((0/LS{ps}(end))*100, 2), ll, Nlocs);
        end
        for ss = LS{ps}
            count = count+1;
        
            if ps==1
                'h'
                fprintf(repmat('\b',1,msg+17))
                msg = fprintf('%.2f %% of position %d/%d', round((ss/LS{ps}(end))*100, 2), ll, Nlocs);
            end
            
            imA = double(tiffreadVolume([projDir  '\' headDir '\' locs(ll).name '\Orange\' ImNames{ss}]));
            if isfile([projDir  '\' headDir '\' locs(ll).name '\Red\' ImNames{ss}])
                imB = double(tiffreadVolume([projDir  '\' headDir '\' locs(ll).name '\Red\' ImNames{ss}]));
            else
                imB = double(tiffreadVolume([projDir  '\' headDir '\' locs(ll).name '\Orange\' ImNames{ss}]));
            end

            [bgA, imA_nobg] = estimatebg(imA);      % Estimate background from raw image
            [bgB, imB_nobg] = estimatebg(imB);

            Amax = mean(maxk(imA_nobg(:),round(prod(sz)/100)));
            Bmax = mean(maxk(imB_nobg(:),round(prod(sz)/100)));

            imSum = imA_nobg./Amax + imB_nobg./Bmax;

            [bgSum, imSum_nobg] = estimatebg(imSum);

            if contains(pcaCol, "Orange")
                im = imA_nobg; 
                bg = bgA;
            elseif contains(pcaCol, "Red")
                im = imB_nobg; 
                bg = bgB;
            elseif contains(pcaCol, "Sum")
                im = imSum; 
                bg = 0;
            end

            if nnz(~isnan(im)) == 0
                Vol_par{ps}(count) = NaN;
            else
                [~,im_nobg] = estimatebg(im, bg);
                Vol_par{ps}(count) = nnz(im_nobg>=max(im_nobg,[],'all')*Vthr);
            end

            
            %% Align particle with principal axes
        
            if contains(pcaCol, "Orange") || contains(pcaCol, "Both")           % Images to calculate the rotation from
    
                [rotmatA, comA, VarA_par{ps}(count,:)] = PSP_Align(imA, bgA);    % Obtain rotation matrix from A and align images A
            
                if contains(pcaCol, "Orange")
                    
                    rotmatB = rotmatA;
                    comB = comA;
    
                elseif contains(pcaCol, "Both")
     
                    [rotmatB, comB, VarB_par{ps}(count,:)] = PSP_Align(imB, bgB);   
                    
                end
        
            elseif contains(pcaCol, "Red")
        
                [rotmatB, comB, VarB_par{ps}(count,:)] = PSP_Align(imB, bgB);    % Obtain rotation matrix from B and align images B
                
                rotmatA = rotmatB;
                comA = comB;
        
            elseif contains(pcaCol, "Sum")

                [rotmatSum, comSum, VarSum_par{ps}(count,:)] = PSP_Align(imSum, bgSum);

                rotmatA = rotmatSum;
                rotmatB = rotmatSum;
                comA = comSum;
                comB = comSum;

            end

%%
            if runcol(1)
            imArot = imrot(imA, rotmatA, comA);
            
            imAsave = imArot;
            StackA_par{ps}(:,:,:,count) = imAsave;

            if any(isnan(imAsave),'all')
                VarA_par{ps}(count,:) = [NaN, NaN, NaN];
            else
                [~, ~, eigvalsA] = PSP_Align(imAsave, bgA);
                VarA_par{ps}(count,:) = sqrt(eigvalsA);
            end

            IntA_par{ps}(count) = sum(imA_nobg,'all');
            end

            if runcol(2)
                imBrot = imrot(imB, rotmatB, comB);
    
                imBsave = imBrot;
                StackB_par{ps}(:,:,:,count) = imBsave;
    
                if any(isnan(imBsave),'all')
                    VarB_par{ps}(count,[2,1,3]) = [NaN, NaN, NaN];
                else
                    [~, ~, eigvalsB] = PSP_Align(imBsave, bgB);
                    VarB_par{ps}(count,:) = sqrt(eigvalsB);
                end
    
                IntB_par{ps}(count) = sum(imB_nobg,'all');
            end

            if runcol(3)
                imSumrot = imrot(imSum, rotmatSum, comSum);
                
                imSumsave = imSumrot;
                StackSum_par{ps}(:,:,:,count) = imSumsave;
    
                if any(isnan(imSumsave),'all')
                    VarSum_par{ps}(count,:) = [NaN, NaN, NaN];
                else
                    [~, ~, eigvalsSum] = PSP_Align(imSumsave, bgSum);
                    VarSum_par{ps}(count,:) = sqrt(eigvalsSum);
                end

            end

            if contains(pcaCol, "Orange")
                im = imArot; 
                bg = bgA;
            elseif contains(pcaCol, "Red")
                im = imBrot; 
                bg = bgB;
            elseif contains(pcaCol, "Sum")
                im = imSumrot; 
                bg = 0;
            end

            if nnz(~isnan(im)) == 0
                Rvol_par{ps}(count,:) = [NaN, NaN, NaN];
                outbox_par{ps}(count) = NaN;
            else
                [~,im_nobg] = estimatebg(im, bg);
                [Rvol_par{ps}(count,:),dPSP, outbox_par{ps}(count)] = AxesRatio(im_nobg, 0.5, 0);
            end

        
        end
        if ps==1
        fprintf('\n');
        end

    end
%%
    for ps = 1:Npar

        if runcol(1)
        VarA(LS{ps},:) = VarA_par{ps};
        IntA(LS{ps},:) = IntA_par{ps};
        StackA(:,:,:,LS{ps}) = StackA_par{ps};
        end

        if runcol(2)
        VarB(LS{ps},:) = VarB_par{ps};
        IntB(LS{ps},:) = IntB_par{ps};
        StackB(:,:,:,LS{ps}) = StackB_par{ps};
        end

        if runcol(3)
        VarSum(LS{ps},:) = VarSum_par{ps};
        StackSum(:,:,:,LS{ps}) = StackSum_par{ps};
        end
        
        Rvol(LS{ps},:) = Rvol_par{ps};
        outbox(LS{ps}) = outbox_par{ps};
        Vol(LS{ps}) = Vol_par{ps};
    end

    cellVarA{ll} = VarA;
    cellIntA{ll} = IntA;

    if runcol(2)
    StackB = StackB;
    cellVarB{ll} = VarB;
    cellIntB{ll} = IntB;
    end

    if runcol(3)
    StackSum = StackSum;
    cellVarSum{ll} = VarSum;
    end

    cellRvol{ll} = Rvol;
    cellOutbox{ll} = outbox;
    cellVol{ll} = Vol;

    if savedata
        namedir = [projDir  '\' headDir '\' locs(ll).name ];
        
        if ~exist([namedir '\PCA_' pcaCol],'dir')      % Make folder to save reference segmentations in
            mkdir([namedir '\PCA_' pcaCol]);
        end
        
        save([namedir '\PCA_' pcaCol '\Orange_aligned','.mat',''], 'StackA','-v7');
        if runcol(2)
        save([namedir '\PCA_' pcaCol '\Red_aligned','.mat',''], 'StackB','-v7');
        end

        if runcol(3)
        save([namedir '\PCA_' pcaCol '\Sum_aligned','.mat',''], 'StackSum','-v7');
        end

    end

end
%%
if savedata
    if ~exist([locDir  '\' headDir '\PCA_' pcaCol],'dir')      % Make folder to save reference segmentations in
            mkdir([locDir  '\' headDir '\PCA_' pcaCol]);
    end

    if runcol(1)
        save([locDir  '\' headDir '\PCA_' pcaCol '\Orange_cellvars','.mat',''],'cellVarA','-v7');
        save([locDir  '\' headDir '\PCA_' pcaCol '\Orange_cellints','.mat',''],'cellIntA','-v7');
    end
    if runcol(2)
        save([locDir  '\' headDir '\PCA_' pcaCol '\Red_cellvars','.mat',''],'cellVarB','-v7');
        save([locDir  '\' headDir '\PCA_' pcaCol '\Red_cellints','.mat',''],'cellIntB','-v7');
    end
    if runcol(3)
        save([locDir  '\' headDir '\PCA_' pcaCol '\Sum_cellvars','.mat',''],'cellVarSum','-v7');
    end

    save([locDir  '\' headDir '\PCA_' pcaCol '\cellRvol','.mat',''],'cellRvol','-v7');
    save([locDir  '\' headDir '\PCA_' pcaCol '\cellOutbox','.mat',''],'cellOutbox','-v7');
    save([locDir  '\' headDir '\PCA_' pcaCol '\cellVol','.mat',''],'cellVol','-v7');

end