using UnityEngine;
using System.Collections;

using DTStacks.UnityComponents.Communication.MQTT;
using DTStacks.DataType.Generic.Custom;
using DTStacks.UnityComponents.Generic.Handler;


namespace DTStacks.UnityComponents.Generic.Subscriber.Custom
{
    /// <summary>
    /// Mqtt client used to subscribe to the topic where the joint pose data is published to. 
    /// Subscribes to the topic upon start of the application, receives the published data and forwards it 
    /// the individual joint handlers.
    /// </summary>
    public class Softgripper4JointSubscriber : DTS_MQTTSubscriber
    {
        //The joint state message. (Custom version of a standard ROS joint state message
        [Tooltip("The latest received Joint State message")]
        public Softgripper4Joint Msg;
        // The individual handlers for the joint states attached to each bone of the model
        [Tooltip("The joint state handlers for each individual joint.")]
        public SoftgripperJointHandler[] handlers;
        // Scaling factor to translate the received units (e.g. mm) to metric (unity default)
        [Tooltip("Factor  to translate the received data from the original unit to metric.")]
        public float scalingFactor;
        // The origin of the coordinate system of the received data
        [Tooltip("Origin of the coordinate system/reference frame.")]
        public Vector3 referencePoint;

        /// <summary>
        /// Feeds the latest received data to the joint state message. Essentially takes the JSON message and overwrites the current message.
        /// </summary>
        /// <param name="s">JSON-string containing the relevant information</param>
        public void FeedData(string s)
        {
            Msg.FeedDataFromJSON(s);
        }

        /// <summary>
        /// Entry point for starting the data pipeline. Overrides the current message and distributes the information to all handlers
        /// </summary>
        /// <param name="msg">JSON-string containing the relevant infromation</param>
        public override void ProcessMessage(string msg)
        {
            Debug.Log("new state received");
            FeedData(msg);
            
            for (int i = 0; i*6 < Msg.position.Length; i++)
            {               
              handlers[i].FeedData(GetPositionAndRotation(Msg.position, i));
            }
            
        }
        /// <summary>
        /// Retrieves the specific position and orientation vectors from the float[] according to the index of the handler
        /// </summary>
        /// <param name="data">float[] containing all position and orientation information</param>
        /// <param name="index">The index of the handler in question</param>
        /// <returns> Vector[2] containing the position (XYZ) and the (euler) rotation(XYZ) </returns>
        public Vector3[] GetPositionAndRotation(float[] data, int index)
        {

            Vector3[] vec = new Vector3[2];

            Vector3 pos = new Vector3(data[index * 6] * scalingFactor, data[1 + index * 6] * scalingFactor, data[2 + index * 6] * scalingFactor);
            if(index == 0)
            {
                //referencePoint = pos;
            }
            Vector3 rot = - new Vector3(data[3 + index * 6], data[4 + index * 6], data[5 + index * 6]);
            vec[0] = pos;
            vec[1] = rot;
            return vec;
        }


        // Use this for extras during initialization
        public override void ExtendedStart()
        {

        }

        // Extended Update function, will be executed after regular update
        public override void ExtendedUpdate()
        {

        }

    }
}
