"""
Using pennylane, this module contains all of the Hamiltonians whose expectaion
values are measured in this project.
"""
import pennylane as pl
import itertools

import numpy as np
from scipy.linalg import ishermitian


def U_1_gauge_H_z(
        n_qubits: int, 
        m: float,
        J: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    There are contributions from two different terms, easiest to split.
    However both are a sum over the same Pauli matrices, so same operators.

    The sum in the second term is reordered from the original paper to fit
    the pennylane conventions but is equivalent.

    TODO: check the final element is right in the sum!
    TODO: check the indexing from 0 doesn't mess with the mod(2)

    """
    term_one_coefficients = [(-1)**qubit * m/2 for qubit in range(1,n_qubits+1)]
    term_two_coefficients = [ (-J/2) * sum([l%2 for l in range(qubit, n_qubits)])
                                for qubit in range(1,n_qubits)]
    term_two_coefficients.append(0) # sum only goes to n_qubits-1
    operators = [pl.PauliZ(qubit) for qubit in range(n_qubits)]
    return pl.Hamiltonian(
                        term_one_coefficients+term_two_coefficients,
                        operators+operators)


def U_1_gauge_H_pm(
        n_qubits: int,
        w: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    In the Muschik 2016 paper this term is written in terms of creation and 
    annihilation operators, but pennylane did not like this. The combination
    was Hermitian but I think that pennylane did not register that it was, and
    thus did not allow it in the Hamiltonian. See the latex for derivation
    of this alternate form.
    """
    coefficients = [w/2 for _ in range(2*(n_qubits-1))]
    X_operators = [pl.PauliX(qubit) @ pl.PauliX(qubit+1)
                 for qubit in range(n_qubits-1)]
    Y_operators = [pl.PauliY(qubit) @ pl.PauliY(qubit+1)   \
                 for qubit in range(n_qubits-1)]
    return pl.Hamiltonian(coefficients, X_operators + Y_operators)


def U_1_gauge_H_zz(
        n_qubits: int,
        J: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    coefficients = [n_qubits - l \
                    for n in range(1,n_qubits-1) \
                    for l in range(n+1,n_qubits)]
    operators = [pl.PauliZ(n) @ pl.PauliZ(l) \
                 for n in range(n_qubits-2) \
                 for l in range(n+1, n_qubits-1)]
    return pl.Hamiltonian(coefficients, operators)




def U_1_gauge_Hamiltonian(
        n_qubits: int, 
        m: float,
        J: float,
        w: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    See Muschik et al. 2016 for the definition of this Hamiltonain.
    Will be split into the same three Hamiltonians as in the paper.
    H = H_zz + H_pm + H_z

    The constants correspond to:
        m: fermion mass
        J: electric field energy
        w: pair production rate

    No specific boundary conditions have been applied yet.

    All the sums in the paper are indexed from 1, but our qubit convention 
    starts at 0. Therefore, all prefactors are computed via indexing from 1
    but the qubits are labelled from 0.
    """
    Hamiltonian = U_1_gauge_H_z(n_qubits, m, J) +\
                  U_1_gauge_H_pm(n_qubits, w) +\
                  U_1_gauge_H_zz(n_qubits, J)
    return Hamiltonian



def Ising_Hamiltonian(
        n_qubits: int,
        J: float,
        mu: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    Magnetic field in the same axis as the spins i.e. classical Ising model.
    """
    local_coefs = [mu for _ in range(n_qubits)]
    local_observables = [pl.PauliZ(qubit) for qubit in range(n_qubits)]
    kinetic_coefs = [J for _ in range(n_qubits)]
    kinetic_observables = [pl.PauliZ(qubit)@pl.PauliZ((qubit+1)%n_qubits) \
                    for qubit in range(n_qubits)]
    Hamiltonian = pl.Hamiltonian(local_coefs+kinetic_coefs, 
                    local_observables+kinetic_observables)
    return Hamiltonian


def transverse_Ising_Hamiltonian(
        n_qubits: int,
        J: float,
        mu: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    Ising model with magentic field in the x axis with spins in the z. 
    This is an inherently quantum model as you cannot know the projection of the
    spin along x and z at the same time.
    """
    local_coefs = [mu for _ in range(n_qubits)]
    local_observables = [pl.PauliX(qubit) for qubit in range(n_qubits)]
    kinetic_coefs = [J for _ in range(n_qubits)]
    kinetic_observables = [pl.PauliZ(qubit)@pl.PauliZ((qubit+1)%n_qubits) \
                    for qubit in range(n_qubits)]
    Hamiltonian = pl.Hamiltonian(local_coefs+kinetic_coefs, 
                    local_observables+kinetic_observables)
    return Hamiltonian


def double_transverse_Ising_Hamiltonian(
        n_qubits: int,
        J: float,
        mu_x: float,
        mu_y: float) -> pl.ops.qubit.hamiltonian.Hamiltonian:
    """
    Ising model with magentic field in the x and y axes with spins in the z. 
    This is an inherently quantum model as you cannot know the projection of the
    spin along x, y and z at the same time.
    """
    local_coefs_x = [mu_x for _ in range(n_qubits)]
    local_observables_x = [pl.PauliX(qubit) for qubit in range(n_qubits)]
    local_coefs_y = [mu_y for _ in range(n_qubits)]
    local_observables_y = [pl.PauliY(qubit) for qubit in range(n_qubits)]
    kinetic_coefs = [J for _ in range(n_qubits)]
    kinetic_observables = [pl.PauliZ(qubit)@pl.PauliZ((qubit+1)%n_qubits) \
                    for qubit in range(n_qubits)]
    Hamiltonian = pl.Hamiltonian(local_coefs_x+local_coefs_y+kinetic_coefs, 
                    local_observables_x+local_observables_y+kinetic_observables)
    return Hamiltonian


if __name__ == "__main__":
    Hamiltonian = U_1_gauge_Hamiltonian(4, 1, 1, 1)
