#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""

This script simulates a quantum circuit with Clifford gates (matrices) and random rotations.
It calculates mutual information for various probabilities of measurement
using Rényi entropy.

Author: caganstu
"""

import numpy as np
import math
import random
from tqdm import tqdm
from qiskit.quantum_info import random_clifford, partial_trace, random_unitary, entropy
from qiskit import QuantumRegister, ClassicalRegister, QuantumCircuit, Aer, transpile
import json
import sys
from qiskit.circuit.library import RXGate, RYGate, RZGate


# Constants
N_time = 250  # Number of time steps
steps = 250  # Number of probability steps

# Qiskit backend and parameters
simulator = Aer.get_backend('statevector_simulator')
p = np.linspace(0, 0.6, steps)
theta = 0
N = 20  # Number of qubits

mutinfo_array = []  # Array to store mutual information results

# Loop over different measurement probabilities
for k in tqdm(range(len(p))):
    q = QuantumRegister(N,'q')
    r = ClassicalRegister(N, 'r')
    qc = QuantumCircuit(q, r)
    
    # Apply random Clifford gates and rotations for N_time steps
    for time_step in range(N_time):
        if (int(time_step) % 2) == 0: # even U-layer
            for i in range(int(N/2)): # laying down unitaries
                i = i*2
                U_C = random_clifford(2)
                qc.append(U_C,[i,i+1])
            for i in range(N): # Apply random rotations
                randrotation = random.sample([RXGate(theta), RYGate(theta), RZGate(theta)], 1)[0]
                qc.append(randrotation, [i])
        
        else: # odd U-layer
            for i in range(int(N/2)): # laying down unitaries
                if i <= int(N/2)-2:
                    i = i*2
                    U_C = random_clifford(2)
                    qc.append(U_C,[i+1,i+2])
                else: 
                    i = i*2
                    U_C = random_clifford(2)
                    qc.append(U_C,[0,i+1])
        
        for i in range(int(N)): # Measurements after U-layer
            prob = np.random.uniform()
            if prob <= p[k]:
                qc.measure(q[i], r[i])
    
    qc = transpile(qc, simulator)
    result = simulator.run(qc).result()
    psi = result.get_statevector(qc) 
    
    # Take proper partial traces and split the systems
    separation = int(N/2)
    subsyst_size = int(N/4)
    qubit_indices = list(range(N))  
    traced_out_indic_sys1 = list(range(subsyst_size))
    traced_out_indic_sys2 = list(range(separation, separation + subsyst_size))
    
    subsyst1 = [value for value in qubit_indices if value not in traced_out_indic_sys1]
    subsyst2 = [value for value in qubit_indices if value not in traced_out_indic_sys2]
    
    
    systemA = partial_trace(psi, subsyst1)
    systemB = partial_trace(psi, subsyst2)
    systemFull = partial_trace(psi, (traced_out_indic_sys1 + traced_out_indic_sys2))
    

    renyi_A = -math.log(systemA.purity(), 2)
    renyi_B = -math.log(systemB.purity(), 2)
    renyi_full = -math.log(systemFull.purity(),2)
    
    mutinfo = renyi_A + renyi_B - renyi_full
    mutinfo_array.append(mutinfo)
    
unique_identifier = sys.argv[1]
unique_identifier2 = sys.argv[2]
output_filename = "Mutinfo_Clifford+T_{}q_{}depth_{}steps_{}pi_run{}.{}.json".format(N, N_time, steps, theta/np.pi, unique_identifier, unique_identifier2)
with open(output_filename, 'w') as json_file:
    json.dump(mutinfo_array, json_file)

print(f"Data saved to {output_filename}")


