classdef LaboratoryConfig < Config
    % LABORATORYCONFIG  Configuration for a [Laboratory].

    properties
        % SEED  Seed for random number generator.
        %
        % A negative value indicates no fixed seed should be used.
        %
        % Two runs with the same seed give the same result. However, this does not
        % necessarily hold if the runs are with different MATLAB versions.
        seed (1, 1) {mustBeInteger} = -1;
        % CREATE_LOGGER  Creates a [Logger] for an [Experiment].
        create_logger (1, 1) = @(it) Logger();

        % PARALLEL  Whether to run experiments in parallel.
        parallel (1, 1) logical = true;
        % PARALLEL_MAX_WORKERS  The maximum number of parallel workers, or a negative
        % number to not impose a limit.
        parallel_max_workers (1, 1) {mustBeInteger} = -1;
        % REPEAT_COUNT  The number of times to repeat each [Experiment].
        repeat_count (1, 1) {mustBeInteger, mustBePositive} = 3;

        % EXP_CONFS  The [ExperimentConfig]s for the experiments to run.
        exp_confs (:, 1) cell;  % cell<ExperimentConfig>

        % PART_COUNT  Number of parts to split the laboratory into, useful if the
        % laboratory is too much work to finish in one go on a cluster.
        %
        % Parts are essentially just an implicit way of splitting up experiments.
        part_count (1, 1) {mustBeInteger, mustBePositive} = 1;
        % PART_IDX  The part to run in this laboratory, or 0 to run all experiments.
        %
        % If [part_idx] is nonzero, plots are disabled.
        part_idx (1, 1) {mustBeInteger, mustBeNonnegative} = 0;

        % CACHE_DIR  The directory to cache any results in.
        cache_dir (1, 1) {mustBeText} = "cache/";
        % SAVE_VERSION  The save file version identifier, to force a re-run despite the
        % presence of appropriate cache files.
        save_version (1, 1) {mustBeText} = "v9";
        % SAVE_ENABLED  `true` if and only if experiments should be saved after
        % completion.
        save_enabled (1, 1) logical = true;
        % LOAD_ENABLED  `true` if and only if completed experiments should be loaded
        % instead of re-running them.
        load_enabled (1, 1) logical = true;
        % LOAD_VALIDATE  `true` if and only if cached files should be checked for
        % integrity before loading.
        %
        % If a cached file fails the check, the cache file is treated as if it does not
        % exist.
        load_validate (1, 1) logical = true;
        % LOAD_EXP_BEHAVIOR  The way existing cached [Experiment]s should be loaded.
        %
        % * `"run"` always re-runs [Experiment]s.
        % * `"load"` loads a cached [Experiment] if it exists, and runs the [Experiment]
        %   otherwise.
        % * `"skip"` skips the [Experiment] if cached ouput exists, and runs the
        %   [Experiment] otherwise. This may result in missing [Experiment]s, so
        %   plotting is disabled.
        %
        % Must be one of: "run", "load", and "skip".
        load_exp_behavior (1, 1) {mustBeMember(load_exp_behavior, ["run", "load", "skip"])} = "load";
        % SAVE_LAB_TARGET  The file path relative to [cache_dir] for saving a completed
        % laboratory of experiments, expressed as a format string where (in order) `%s`
        % is the [save_version] and `%s` is the [Laboratory]'s [cache_id].
        save_lab_target (1, 1) {mustBeText, ...
                                mustContainSubstring(save_lab_target, "%s")} = "lab-%s-%s.mat";
        % SAVE_EXP_TARGET  The file path relative to [cache_dir] for saving experiments,
        % expressed as a format string where (in order) `%s` is the [save_version], `%s`
        % is the [Experiment]'s [cache_id], and `%d` identifies the repetition number of
        % the experiment.
        save_exp_target (1, 1) {mustBeText, ...
                                mustContainSubstring(save_exp_target, "%s"), ...
                                mustContainSubstring(save_exp_target, "%d")} = "exp-%s-%s-%d.mat";

        % PLOT_DIR  The directory to save plots in.
        plot_dir (1, 1) {mustBeText} = "figs/";
        % PLOT_CONFS  The [PlotConfig]s for the plots to make after completing all
        % [Experiment]s.
        plot_confs (:, 1) cell = {};  % cell<PlotConfig>
        % PLOT_SHOW  `true` if and only if obtained plots should be shown.
        plot_show (1, 1) logical = false;
        % PLOT_SAVE  `true` if and only if obtained plots should be saved to a file.
        %
        % Plots are saved in the [save_directory] folder.
        plot_save (1, 1) logical = true;
        % PLOT_SAVE_FORMATS  The file formats to save obtained plots as.
        plot_save_formats (:, 1) {mustBeText} = ["svg", "epsc", "png"];
    end


    methods
        function obj = LaboratoryConfig(args)
            arguments% (Input)
                args.?LaboratoryConfig;
            end
            % arguments (Output)
            %     obj (1, 1) LaboratoryConfig;
            % end

            obj = obj@Config(args);
        end


        function new_obj = set(obj, args)
            arguments% (Input)
                obj (1, 1) LaboratoryConfig;
                args.?LaboratoryConfig;
            end
            % arguments (Output)
            %     new_obj (1, 1) LaboratoryConfig;
            % end

            new_obj = set@Config(obj, args);
        end

        function h = cache_id(obj)
            arguments% (Input)
                obj (1, 1) LaboratoryConfig;
            end
            % arguments (Output)
            %     h (1, 1) {mustBeText};
            % end

            % Only add fields that affect experiment output.
            % Do not include fields that only affect output paths.
            % Do not include fields that only affect plot outputs.
            h = cache_id@Config(LaboratoryConfig(create_logger = 0, ...
                                                 seed = obj.seed, ...
                                                 repeat_count = obj.repeat_count, ...
                                                 exp_confs = obj.exp_confs));
        end
    end

    methods (Static)
        function partial_obj = partial(args)
            arguments% (Input)
                args.?LaboratoryConfig;
            end
            % arguments (Output)
            %     partial_obj (1, :) cell;
            % end

            partial_obj = partial@Config(args);
        end
    end
end
