#  Copyright 2022
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#  MA 02110-1301, USA.
#
#  Data and code based on
#  R.J.A.M. Stevens, L.A. Martínez Tossas, C. Meneveau,
#  Comparison of large eddy simulations using actuator disk or actuator line models with wind tunnel experiments,
#  Renewable Energy, 116 (A), 470-478 (2018).
#  https://doi.org/10.1016/j.renene.2017.08.072
#
#  Contact information
#  Richard Stevens (r.j.a.m.stevens@utwente.nl)
#  Luis 'Tony' Martinez Tossas (Luis.Martinez@nrel.gov)
#  Charles Meneveau (meneveau@jhu.edu) 

import matplotlib
import os
import matplotlib.pyplot as plt
import numpy as np
import h5py

# Line styles for ADM and ALM with and without structure
admls='r-'
admNTls='b--'
almls='g-'
almNTls='k--'

################################################################################
##                      The main function of this file                        ##
################################################################################
def main():
    # Create directories to save plots
    if not os.path.exists('./plots'):
        os.makedirs('./plots')

    # Velocity profiles
    multiplot(name='Figure6A.pdf', turbines=[1,2,3,4],diameters=[1,2,3,4] )
    multiplot(name='Figure6B.pdf', turbines=[5,6,7,10],diameters=[1,2,3,4] )

    # Turbulence intensity profiles
    multiplotup(name='Figure7A.pdf', turbines=[1,2,3,4],diameters=[1,2,3,4] )
    multiplotup(name='Figure7B.pdf', turbines=[5,6,7,10],diameters=[1,2,3,4] )

################################################################################
##                                Multiplot                                   ##
################################################################################
def multiplot(name='subplot.pdf', turbines=[1], diameters=[1, 2, 3, 5]):
    '''
    Plot multi-figure for different turbines
    '''
    turbines = turbines
    diameters = diameters
    name = name

    # Turbine Diameter
    D = 0.15
  
    # Size of the ticks
    tsize = 10

    # Font size for legend
    lsize = 16

    plt.clf()
    f, ax = plt.subplots(len(turbines), len(diameters), sharex='col',sharey='row')
    # Eliminate horizontal spacing
    f.subplots_adjust(hspace=0, wspace = 0)

    # Loop through every turbine
    for i, t in enumerate(turbines):
        for j, d in enumerate(diameters):

            # Figure properties
            fig = ax[i][j]

            fig.set_xticks([0.4, 0.6, 0.8, 1, ])
            fig.set_yticks([0.0, 1.0, 2.0])

            fig.set_xlim([0.3, 1.25])
            fig.set_ylim([0, 0.4 / D])
            fig.axhline(y=(0.125 - 0.075) / D)
            fig.axhline(y=(0.125 + 0.075) / D)

            fig.tick_params(axis='both', which='major', labelsize=tsize)
            fig.tick_params(axis='both', which='minor', labelsize=tsize)

            '''
            Plot experimental data
            '''
            xEXP, yEXP = np.loadtxt('./Data_Windfarmcase/EPFL_EXP_location_'+str((t-1) * 5 + 1 + j)+'.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yEXP, xEXP, 'o', label='Experiment', ms=4,markerfacecolor='none',color='grey',)

            '''
            Plot ALM data
            '''
            xALM, yALM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_' +str((t-1) * 5 + 1 + j)+'_ALM.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almls, lw=1, label='ALM')

            xALM, yALM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_' +str((t-1) * 5 + 1 + j)+'_ALM_str.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almNTls, lw=1, label='ALM+structure')

            '''
            Plot ADM data
            '''
            xADM, yADM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_'+str((t-1) * 5 + 1 + j)+'_ADM.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admls, lw=1, label='ADM')

            xADM, yADM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_'+str((t-1) * 5 + 1 + j)+'_ADM_str.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admNTls, lw=1, label='ADM+structure')

            # Write turbine number and diameter downstream
            # Location in fig
            xx1 = fig.get_xlim()[0] + (fig.get_xlim()[1] -fig.get_xlim()[0]) * 0.05
            yy1 = fig.get_ylim()[1] - (fig.get_ylim()[1] -fig.get_ylim()[0]) * 0.05

            fig.text(xx1, yy1, 'Row '+str(t)+'\n'+str(d) + r'D', fontsize=tsize,horizontalalignment='left', verticalalignment='top')

    f.text(.05, 0.5, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')
    f.text(.5, 0.0, r'$u/U_{H}$', fontsize=lsize,horizontalalignment='center', verticalalignment='center')

    # Legend
    lgd = plt.legend(loc='center right',bbox_to_anchor=(2.4, 2.),borderaxespad=0.1,ncol=1, fontsize=lsize-6)

    f.savefig('./plots/' + name, bbox_inches='tight',)
    plt.clf()

################################################################################
##                                Multiplot                                   ##
################################################################################
def multiplotup(name='subplot.pdf', turbines=[1], diameters=[1, 2, 3, 5]):

    turbines = turbines
    diameters = diameters
    name = name

    # Turbine Diameter
    D = 0.15
   
    # Size of the ticks
    tsize = 10

    # Font size for legend
    lsize = 16

    plt.clf()
    f, ax = plt.subplots(len(turbines), len(diameters), sharex='col',sharey='row')
    # Eliminate horizontal spacing
    f.subplots_adjust(hspace=0, wspace = 0)
    
    # Loop through every turbine
    for i, t in enumerate(turbines):
        for j, d in enumerate(diameters):

            # Figure properties
            fig = ax[i][j]

            fig.set_xticks([0.06, 0.12, 0.18, ])
            fig.set_yticks([0.0, 1.0, 2.0])

            fig.set_xlim([0.04, 0.22])
            fig.set_ylim([0, 0.4 / D])
            fig.axhline(y=(0.125 - 0.075) / D)
            fig.axhline(y=(0.125 + 0.075) / D)

            fig.tick_params(axis='both', which='major', labelsize=tsize)
            fig.tick_params(axis='both', which='minor', labelsize=tsize)

            '''
            Plot experimental data
            '''
            xEXP, yEXP = np.loadtxt('./Data_Windfarmcase/EPFL_EXP_location_'+str((t-1) * 5 + 1 + j)+'_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yEXP, xEXP, 'o', label='Experiment', ms=4,markerfacecolor='none',color='grey',)

            '''
            Plot ALM data
            '''
            xALM, yALM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_' +str((t-1) * 5 + 1 + j)+'_ALM_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almls, lw=1, label='ALM')

            xALM, yALM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_' +str((t-1) * 5 + 1 + j)+'_ALM_str_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almNTls, lw=1, label='ALM+structure')
            
            '''
            Plot ADM data
            '''
            xADM, yADM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_'+str((t-1) * 5 + 1 + j)+'_ADM_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admls, label='ADM', lw=1)

            xADM, yADM = np.loadtxt('./Data_Windfarmcase/SMM18RE_location_'+str((t-1) * 5 + 1 + j)+'_ADM_str_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admNTls, label='ADM+structure', lw=1)
            
            # Location in fig
            xx1 = fig.get_xlim()[1] - (fig.get_xlim()[1] -fig.get_xlim()[0]) * 0.05
            yy1 = fig.get_ylim()[1] - (fig.get_ylim()[1] -fig.get_ylim()[0]) * 0.05
            fig.text(xx1, yy1, 'Row '+str(t)+'\n'+str(d) + r'D', fontsize=tsize,horizontalalignment='right', verticalalignment='top')

    f.text(.05, 0.5, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')
    f.text(.5, 0.00, r'$u^\prime/U_{H}$', fontsize=lsize,horizontalalignment='center', verticalalignment='center')

    # Legend
    lgd = plt.legend(loc='center right',bbox_to_anchor=(2.4, 2.),borderaxespad=0.1,ncol=1, fontsize=lsize-6)
    
    f.savefig('./plots/' + name, bbox_inches='tight',)
    plt.clf()

##########################################################
# Run the main function
if __name__ == '__main__':
    main()
