#  Copyright 2022
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#  MA 02110-1301, USA.
#
#  Data and code based on
#  R.J.A.M. Stevens, L.A. Martínez Tossas, C. Meneveau,
#  Comparison of large eddy simulations using actuator disk or actuator line models with wind tunnel experiments,
#  Renewable Energy, 116 (A), 470-478 (2018).
#  https://doi.org/10.1016/j.renene.2017.08.072
#
#  Contact information
#  Richard Stevens (r.j.a.m.stevens@utwente.nl)
#  Luis 'Tony' Martinez Tossas (Luis.Martinez@nrel.gov)
#  Charles Meneveau (meneveau@jhu.edu) 


import matplotlib
import os
import matplotlib.pyplot as plt
import numpy as np
import h5py

# Line styles for ADM and ALM with and without structure
admls='r-'
admNTls='b--'
almls='g-'
almNTls='k--'

################################################################################
##                      The main function of this file                        ##
################################################################################
def main():
    # Create directories to save plots
    if not os.path.exists('./plots'):
        os.makedirs('./plots')

    # Note location 1 is the inflow profile measured 1D upstream. All other locations are downstream the turbine
    # Velocity profiles
    multiplot(name='figure4A.pdf', turbines=[1],diameters=[1,2,3,5,7,10,14,20] )

    # Turbulence intensity profiles
    multiplotup(name='figure4B.pdf', turbines=[1],diameters=[1,2,3,5,7,10,14,20] )

################################################################################
##                                Multiplot                                   ##
################################################################################
def multiplot(name='subplot.pdf', turbines=[1], diameters=[1,2,3,5,7,10,14,20]):
    '''
    Plot multi-figure for different turbines
    '''
    turbines = turbines
    diameters = diameters
    name = name

    # Turbine Diameter
    D = 0.15
  
    # Size of the ticks
    tsize = 10

    # Font size for legend
    lsize = 14

    plt.clf()
    f, ax = plt.subplots(2, 4, sharex='col',sharey='row')

    # Loop through every turbine
    for i, t in enumerate(turbines):
        for j, d in enumerate(diameters):

            # Figure properties
            if j<4:
                fig = ax[0][j]
                ax[0, j].set_aspect(0.8/1.75)
                if j>0:
                    ax[0, j].set_title(str(diameters[j]) + r'D', fontsize=tsize)
                if j==0:
                    ax[0, j].set_title('-' + str(diameters[j]) + r'D', fontsize=tsize)
            if j>3:
                fig = ax[1][j-4]
                ax[1, j-4].set_aspect(0.8/1.75)
                ax[1, j-4].set_title(str(diameters[j]) + r'D', fontsize=tsize)

            fig.set_xticks([0.4, 0.6, 0.8, 1.0, 1.2, ])
            fig.set_yticks([0.0, 0.5, 1.0, 1.5 ])
            plt.subplots_adjust(wspace=0.3, hspace=-0.35)


            fig.set_xlim([0.4, 1.2])
            fig.set_ylim([0, 1.75])
            fig.axhline(y=(0.125 - 0.075) / D, color='k',linestyle='dashed', lw=1)
            fig.axhline(y=(0.125 + 0.075) / D, color='k',linestyle='dashed', lw=1)
            fig.axhline(y=(0.125) / D, color='k',linestyle='dashed', lw=1)

            fig.tick_params(axis='both', which='major', labelsize=tsize)
            fig.tick_params(axis='both', which='minor', labelsize=tsize)

            '''
            Plot experimental data
            '''
            uh=2.2253
            yEXP, xEXP = np.loadtxt('./Data_turbinecase/EPFL_EXP_location_'+str(diameters[j])+'.txt',skiprows=0, usecols=(0, 1), unpack=True)
            fig.plot(yEXP/uh, xEXP, 'o', label='Experiment', ms=4,markerfacecolor='none',color='grey',)

            '''
            Plot ALM data
            '''
            xALM, yALM = np.loadtxt('./Data_turbinecase/SMM18RE_location_' +str(diameters[j])+'_ALM.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almls, lw=1, label='ALM')

            xALM, yALM = np.loadtxt('./Data_turbinecase/SMM18RE_location_' +str(diameters[j])+'_ALM_str.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almNTls, lw=1, label='ALM+structure')

            '''
            Plot ADM data
            '''
            xADM, yADM = np.loadtxt('./Data_turbinecase/SMM18RE_location_'+str(diameters[j])+'_ADM.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admls, lw=1, label='ADM')

            xADM, yADM = np.loadtxt('./Data_turbinecase/SMM18RE_location_'+str(diameters[j])+'_ADM_str.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admNTls, lw=1, label='ADM+structure')

    f.text(.05, 0.33, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')
    f.text(.05, 0.64, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')

    f.text(.13, 0.713, r'Top',fontsize=tsize-2,)
    f.text(.13, 0.648, r'Hub',fontsize=tsize-2,)
    f.text(.13, 0.588, r'Bottom',fontsize=tsize-2,)

    f.text(.25, 0.15, r'$u/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.45, 0.15, r'$u/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.65, 0.15, r'$u/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.85, 0.15, r'$u/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    # Legend
    lgd = plt.legend(loc='center right',bbox_to_anchor=(2.75, 1.),borderaxespad=0.1,ncol=1, fontsize=lsize-3)

    f.savefig('./plots/' + name, bbox_inches='tight',)
    plt.clf()

################################################################################
##                                Multiplot                                   ##
################################################################################
def multiplotup(name='subplot.pdf', turbines=[1], diameters=[1, 2, 3, 5]):

    turbines = turbines
    diameters = diameters
    name = name

    # Turbine Diameter
    D = 0.15
   
    # Size of the ticks
    tsize = 10

    # Font size for legend
    lsize = 14

    plt.clf()
    f, ax = plt.subplots(2, 4, sharex='col',sharey='row')
    
    # Loop through every turbine
    for i, t in enumerate(turbines):
        for j, d in enumerate(diameters):

            # Figure properties
            if j<4:
                fig = ax[0][j]
                ax[0, j].set_aspect(0.14/1.75)
                if j>0:
                    ax[0, j].set_title(str(diameters[j]) + r'D', fontsize=tsize)
                if j==0:
                    ax[0, j].set_title('-' + str(diameters[j]) + r'D', fontsize=tsize)                
            if j>3:
                fig = ax[1][j-4]
                ax[1, j-4].set_aspect(0.14/1.75)
                ax[1, j-4].set_title(str(diameters[j]) + r'D', fontsize=tsize)

            fig.set_xticks([0.05, 0.10, 0.15, ])
            fig.set_yticks([0.0, 0.5, 1.0, 1.5 ])
            plt.subplots_adjust(wspace=0.15, hspace=-0.35)


            fig.set_xlim([0.03, 0.17])
            fig.set_ylim([0, 1.75])
            fig.axhline(y=(0.125 - 0.075) / D, color='k',linestyle='dashed', lw=1)
            fig.axhline(y=(0.125 + 0.075) / D, color='k',linestyle='dashed', lw=1)
            fig.axhline(y=(0.125) / D, color='k',linestyle='dashed', lw=1)

            fig.tick_params(axis='both', which='major', labelsize=tsize)
            fig.tick_params(axis='both', which='minor', labelsize=tsize)

            '''
            Plot experimental data
            '''
            xEXP, yEXP = np.loadtxt('./Data_turbinecase/EPFL_EXP_location_'+str(diameters[j])+'_turb.txt',skiprows=0, usecols=(0, 1), unpack=True)
            fig.plot(xEXP, yEXP, 'o', label='Experiment', ms=4,markerfacecolor='none',color='grey',)
            
            '''
            Plot ALM data
            '''
            xALM, yALM = np.loadtxt('./Data_turbinecase/SMM18RE_location_' +str(diameters[j])+'_ALM_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almls, lw=1, label='ALM')

            xALM, yALM = np.loadtxt('./Data_turbinecase/SMM18RE_location_' +str(diameters[j])+'_ALM_str_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)                                                    
            fig.plot(yALM, xALM, almNTls, lw=1, label='ALM+structure')
            
            '''
            Plot ADM data
            '''
            xADM, yADM = np.loadtxt('./Data_turbinecase/SMM18RE_location_'+str(diameters[j])+'_ADM_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admls, label='ADM', lw=1)

            xADM, yADM = np.loadtxt('./Data_turbinecase/SMM18RE_location_'+str(diameters[j])+'_ADM_str_turb.txt',skiprows=1, usecols=(0, 1), unpack=True)
            fig.plot(yADM, xADM, admNTls, label='ADM+structure', lw=1)
            
    f.text(.05, 0.33, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')
    f.text(.05, 0.64, r'$z/D$', rotation='vertical', fontsize=lsize,horizontalalignment='center', verticalalignment='center')


    f.text(.23, 0.718, r'Top',fontsize=tsize-2,)
    f.text(.23, 0.648, r'Hub',fontsize=tsize-2,)
    f.text(.23, 0.583, r'Bottom',fontsize=tsize-2,)
    
    f.text(.25, 0.15, r'$\sigma/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.45, 0.15, r'$\sigma/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.65, 0.15, r'$\sigma/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')
    f.text(.85, 0.15, r'$\sigma/U_{H}$', fontsize=lsize,horizontalalignment='right', verticalalignment='center')

    # Legend
    lgd = plt.legend(loc='center right',bbox_to_anchor=(2.75, 1.),borderaxespad=0.1,ncol=1, fontsize=lsize-3)
    
    f.savefig('./plots/' + name, bbox_inches='tight',)
    plt.clf()

##########################################################
# Run the main function
if __name__ == '__main__':
    main()
