function [U,U_real_av] = finitedifference_carreau_dimless(dPdx,H,b,fig_on)

%clear all; close all;

% system parameters
% dPdx = -25000;
% H = 100e-6;
% b = 5e-6;

% % carreau fluid parameters for 0.05 wt% XG solution
% mu_0 = 0.089; % zero-shear viscosity [Pa s]
% mu_inf = 8.9e-4; % infinite-shear viscosity [Pa s]
% lambda = 5.0; % onset of zero-shear viscosity [s]
% n_power = 0.563; % power index
% a_carreau = 2; % for the Carreau model a = 2

% carreau fluid parameters for 0.1 wt% XG solution
mu_0 = 0.259; % zero-shear viscosity [Pa s]
mu_inf = 8.9e-4; % infinite-shear viscosity [Pa s]
lambda = 5.9; % onset of zero-shear viscosity [s]
n_power = 0.503; % power index
a_carreau = 2; % for the Carreau model a = 2

% % carreau fluid parameters for 1 wt% CMC solution
% mu_0 = 0.48; % zero-shear viscosity [Pa s]
% mu_inf = 8.9e-4; % infinite-shear viscosity [Pa s]
% lambda = 0.34; % onset of zero-shear viscosity [s]
% n_power = 0.57; % power index
% a_carreau = 2; % for the Carreau model a = 2

% calculation of dimensionless parameters
U_0 = -H^2*dPdx/mu_0;
MU = mu_inf/mu_0;
tau = lambda*U_0/H;
B = b/H;

n = 99; % number of intervals, i.e. no. of grid points is n+1; use a uneven
% number to avoid having a 'singularity' in the middle of the channel where
% dUdy = 0 (in case of no-slip)
deltax = 1/n; % step size in finite differencing

% initial guess of u by poiseuille equation using mu_0
Y_init = linspace(0,1,n+1);
%U_init = 4.*Y_init.*(1-Y_init);
U_init = (Y_init.*(1-Y_init) + B.*(1-Y_init)./(1+B))./(1/6 + B/2./(1+B));
U_init_av = -H^2*dPdx/12/mu_0 - b*H*dPdx/4/mu_0/(1+b/H);

% solve velocity profile numerically
options = optimset('MaxFunEvals',1e8,'MaxIter',1e5,'Display','off','TolFun',1e-12,'TolX',1e-12); %'iter'
[U,F] = fsolve(@carreau,U_init,options,B,MU,tau,n_power,a_carreau,n);

% compute dimensionalised solution
Y_real = Y_init .* H;
U_real = U .* U_0;

% compute shear rate and viscosity
dUdy = zeros(n+1,1);
for k= 2:n
    dUdy(k) = (U(k+1) - U(k-1)) /(2*deltax);
end
dUdy(1) = (-3*U(1) + 4*U(2) - U(3))/(2*deltax);
dUdy(n+1) = (3*U(n+1) - 4*U(n) + U(n-1))/(2*deltax);
vis = MU + (1-MU).*(1+(tau.*dUdy).^a_carreau).^((n_power-1)/a_carreau);
dUdy_real = U_0/H.*dUdy;
vis_real = vis.*mu_0;

% compute average velocity using Simpson's rule
U_real_av = simps(Y_real,U_real)/H;

% analytical velocity profile for zero slip for power-law fluid
% h = H/2;
% K = 1;
% y_anal = linspace(h,H,n/2+1);
% u_anal = ((-dPdx*h)/(K)).^(1/n_power) .* h ./ (1/n_power+1)...
%     .* (1 - ((y_anal-h)./h).^(1/n_power+1)); % only for 0 < y < h
% u_anal_rescaled = u_anal./max(u_anal).*max(U);

if fig_on == 1
    % Carreau viscosity
    nn = 0;
    nn = nn + 1;
    gamma = logspace(-2,3,51);
    mu_anal = mu_inf + (mu_0-mu_inf).*(1+(lambda.*gamma).^a_carreau).^((n_power-1)/a_carreau);
    figure(nn)
    loglog(gamma,mu_anal,'o-');
    ylabel('\mu [Pa s]')
    xlabel('du/dy [1/s]')
    axis tight
    title('Flow curve Carreau fluid')
    
    % initial guess
    nn = nn + 1;
    figure(nn)
    plot(Y_init,U_init,'o');
    ylabel('u* (initial guess)')
    xlabel('y*')
    axis tight
    title('Initial guess for velocity profile based on Poiseuille flow')
    
    % computed dimenionless solution
    nn = nn + 1;
    figure(nn)
    plot(Y_init,U,'o');
    ylabel('u*')
    xlabel('y*')
    axis tight
    title('Dimensionless velocity Carreau fluid flow')
    
    % shear rate
    nn = nn + 1;
    figure(nn)
    plot(Y_real,dUdy_real,'o')
    %ylim([1e-2 mu_0])
    ylabel('du/dy [1/s]')
    xlabel('y [m]')
    axis tight
    title('Shear rate Carreau fluid flow')
    
    % viscosity
    nn = nn + 1;
    figure(nn)
    plot(Y_real,log10(vis_real),'o')
    ylim([1e-2 mu_0])
    ylabel('log_{10}(\mu) [Pa s]')
    xlabel('y [m]')
    axis tight
    title('Viscosity Carreau fluid flow')
    
    % plot solutions
    nn = nn + 1;
    figure(nn)
    plot(Y_real,U_real,'o');
    ylabel('u [m/s]')
    xlabel('y [m]')
    axis tight
    title('Velocity Carreau fluid flow')
    
    % display messages
    disp(['The maximum velocity is ' num2str(max(U_real)) ' m/s']);
    disp(['The average velocity is ' num2str(U_real_av) ' m/s']);
    disp(['The maximum and minimum visocity are ' num2str(max(vis_real))...
        ' and ' num2str(min(vis_real)) ' Pa s']);
    disp(['The average velocity for Poiseuille flow is ' num2str(U_init_av) ' m/s']);
else
end

    function F = carreau(U,B,MU,tau,n_power,a_carreau,n)
    
    dudy = zeros(n+1,1);
    for k= 2:n
        dudy(k) = (U(k+1) - U(k-1)) /(2*deltax);
    end
    dudy(1) = (-3*U(1) + 4*U(2) - U(3))/(2*deltax);
    dudy(n+1) = (3*U(n+1) - 4*U(n) + U(n-1))/(2*deltax);
    viscosity = MU + (1-MU).*(1+(tau.*dudy).^a_carreau).^((n_power-1)/a_carreau);
    A = zeros(n+1,1);
    for k = 2:n
    A(k) = (viscosity(k+1).*dudy(k+1)- viscosity(k-1).*dudy(k-1))/(2*deltax);
    end
    F = 1+A; % A is a function of u
    F(1) = U(1)-B*(dudy(1)); % dudy is a function of u, and BC u(1) = b dudy(1)
    F(n+1) = U(n+1); % BC u(n+1) = 0
    end
end
