function finitedifference_carreau

clear all; close all;

% system parameters
H = 100e-6;
mu = 1e-3;
dPdx = 10000;
b = 5e-6;

% carreau fluid parameters
mu_0 = 2.05; % zero-shear viscosity [Pa s]
mu_inf = 8.9e-4; % infinite-shear viscosity [Pa s]
lambda = 21.2;  % onset of zero-shear viscosity [s]
n_power = 0.406; % power index
a_carreau = 2; % for the Carreau model a = 2

n = 25; % number of intervals, i.e. no. of grid points is n+1
deltax = H/n; % step size in finite differencing

% initial guess of u by poiseuille equation
y_init = linspace(0,H,n+1);
u_init = 1/2/mu*dPdx.*y_init.*(H-y_init)...
    +b*H*1/2/mu*dPdx/(1+b/H).*(1-y_init/H);

% plot initial solutions
% figure(1)
% plot(y_init,u_init);
% axis tight

% solve velocity profile numerically
options = optimset('MaxFunEvals',1e8,'MaxIter',1e5,'Display','iter');
[u,F] = fsolve(@carreau,u_init,options,dPdx,b,mu_0,mu_inf,lambda,n_power,a_carreau,n);

% analytical velocity profile for zero slip
h = H/2;
K = 1;
y_anal = linspace(h,H,n/2+1);
u_anal = ((dPdx*h)/(K)).^(1/n_power) .* h ./ (1/n_power+1)...
    .* (1 - ((y_anal-h)./h).^(1/n_power+1)); % only for 0 < y < h
u_anal_rescaled = u_anal./max(u_anal).*max(u);

% plot solutions
figure(2)
plot(y_init,u,'o',y_init,u_init,'s');
legend('u','u init')
axis tight

figure(3)
plot(y_init,u,'o');
legend('u')
axis tight

    function F = carreau(u,dPdx,b,mu_0,mu_inf,lambda,n_power,a_carreau,n)
    
    dudy = zeros(n+1,1);
    for k= 2:n
        dudy(k) = (u(k+1) - u(k-1)) /(2*deltax);
    end
    dudy(1) = (-3*u(1) + 4*u(2) - u(3))/(2*deltax);
    dudy(n+1) = (3*u(n+1) - 4*u(n) + u(n-1))/(2*deltax);
    %shear_rate = max(abs(dudy),1e-2); % the minimum value has to be decreased when dPdx goes to 0
    viscosity = mu_inf + (mu_0-mu_inf).*(1+(lambda.*dudy).^a_carreau).^((n_power-1)/a_carreau);
    A = zeros(n+1,1);
    for k = 2:n
    A(k) = (viscosity(k+1).*dudy(k+1)- viscosity(k-1).*dudy(k-1))/(2*deltax);
    end
    F = -dPdx-A; % A is a function of u % why needs dPdx to be negative for a positive flow direction?
    F(1) = u(1)-b*abs(dudy(1)); % dudy is a function of u, and BC u(1) = b dudy(1)
    F(n+1) = u(n+1); % BC u(n+1) = 0
    end

end
