# -*- coding: utf-8 -*-
# Created by mjribeiro at 9-10-2020

import os
from bluesky.tools.simtime import timed_function
import bluesky as bs
import numpy as np
import bluesky.tools.pathfinder as pf
import bluesky.tools.structurefinder as sf
from bluesky.tools import geo
from bluesky.tools.aero import nm, ft, kts
from datetime import datetime
import bluesky.tools.cityboundaries as cb


AC_TYPE = 'MAVIC'
TAS_MAX = 30.0  * kts # kts - 15.4m/s
TAS_CURVE = 10.0 * kts  # kts -6.2m/s
TURNRAD = 3 / nm  # 3 meter

LAYER_SEP = 5  #5 ft
ASAS_PZH = bs.settings.asas_pzh # ft
ALT_MIN = 65 # ft

#ALTS = [65, 105, 145, 185, 225, 265, 305, 345, 385, 425, 465]
ALTS = np.arange(ALT_MIN, ALT_MIN + (LAYER_SEP + ASAS_PZH)*17, LAYER_SEP + ASAS_PZH) # in ft
NUMBER_LAYERS = 6

TRAFFIC_LAYER_ID = 0
FAST_LAYER_ID = 1
SLOW_LAYER_ID = 2

STATE_DIM = 12

ALTS_all = np.arange(ALT_MIN, ALT_MIN + (LAYER_SEP + ASAS_PZH) * 17, LAYER_SEP + ASAS_PZH)
ALTS_all = ALTS_all * ft  # pass to meters
ACTION_TIME = 5


def init_plugin():
    # Addtional initilisation code
    global inflowcontrol
    inflowcontrol = inflowcontrol()

    # Configuration parameters
    config = {
        # The name of your plugin
        'plugin_name': 'INFLOWCONTROL',

        # The type of this plugin. For now, only simulation plugins are possible.
        'plugin_type': 'sim',

        # The update function is called after traffic is updated.
        'update': inflowcontrol.update,

        # The reset function
        'reset': inflowcontrol.reset
    }
    stackfunctions = {}

    # init_plugin() should always return these two dicts.
    return config, stackfunctions


class inflowcontrol():

    def __init__(self):
        super(inflowcontrol, self).__init__()

    @timed_function('INFLOWCONTROL', dt=1.0)
    def update(self, dt):
        if self.previous_read_points != self.getPointFileName():
            self.readDeliveryPoints()

        remove_ddpg_actions = []
        for it in range(len(self.DDPG_actions_ac)):
            ac_name = self.DDPG_actions_ac[it]
            ac_id = bs.traf.id2idx(ac_name)
            iactwp = bs.traf.ap.route[ac_id].iactwp
            #next_alt = bs.traf.ap.route[ac_id].wpalt[iactwp]
            if bs.sim.simt - self.DDPG_actions_time[it] >= ACTION_TIME or iactwp >= len(bs.traf.ap.route[ac_id].wpalt) -1: # next_alt != ALTS_all[int(self.DDPG_actions_alt[it])] or
                self.setDDPGreward(ac_id, self.DDPG_actions[it],  self.DDPG_actions_conf[it], self.DDPG_actions_los[it], iactwp)
                remove_ddpg_actions.append(it)

        for it in remove_ddpg_actions[::-1]:
            self.DDPG_actions.pop(it)
            self.DDPG_actions_ac.pop(it)
            self.DDPG_actions_alt.pop(it)
            self.DDPG_actions_conf.pop(it)
            self.DDPG_actions_los.pop(it)
            self.DDPG_actions_time.pop(it)

        if not cb.FINISHED_READING_DATA and len(cb.EDGES_PER_NODE) < 523:
            cb.definecity([37.7648, 37.738, -122.474, -122.51])
        else:
            self.generate_new_aircraft()

    def reset(self):
        self.previous_structure = None
        self.createdAircraft = np.array([])
        self.previous_read_points = None
        self.aircraftNames = np.array([])
        self.generation_times = np.array([])
        self.deliverypoints = dict()
        self.aircraftRoutes = dict()
        self.prevconfpairs = 0
        self.time_first_structure_set = None
        self.aicraft_starting_per_layer  = np.zeros(NUMBER_LAYERS)

        self.alts_list = []
        self.route_list = []
        self.vars_list = []
        self.ac_names_to_create = []
        self.DDPG_actions = []
        self.DDPG_actions_ac = []
        self.DDPG_actions_time = []
        self.DDPG_actions_alt = []
        self.DDPG_actions_conf = []
        self.DDPG_actions_los = []

    def generate_new_aircraft(self):
        last_created_it = 0

        # can we generate new aircraft?
        confpairs_new = len(bs.traf.cd.confpairs_all) - self.prevconfpairs
        self.prevconfpairs = len(bs.traf.cd.confpairs_all_cruising)

        heading_per_layer, active_layers = sf.getCurrentStructure()

        if heading_per_layer is None:
            # a structurbatcbatche hasnt been set yet
            return

        total_sum = 0
        for hdq_layer in heading_per_layer:
            if type(hdq_layer) in [int, np.int32, np.int64]:
                total_sum += hdq_layer
            else:
                total_sum += np.sum(hdq_layer)

        if total_sum == 0:
            # a structurbatcbatche hasnt been set yet
            return

        if self.time_first_structure_set is None:
            self.time_first_structure_set = bs.sim.simt

        if self.previous_structure is not None and not self.areStructureTheSame(self.previous_structure, heading_per_layer):
            #change the route of all existing aircraft, as flight levels have probably changed
            for aircraft_idx in range(0, bs.traf.ntraf):
                aicraftname = bs.traf.id[aircraft_idx]
                route_left = self.aircraftRoutes[aicraftname][bs.traf.ap.route[aircraft_idx].iactwp:]
                if len(route_left) > 1:
                    alts, route, vars, self.aicraft_starting_per_layer = \
                        pf.get_path(aicraftname, route_left, heading_per_layer, active_layers, self.aicraft_starting_per_layer)
                    self.aircraftRoutes[aicraftname] = route
                    bs.traf.ap.route[aircraft_idx].delrte()# clear old route
                    self.setAircraftRoute(aicraftname, alts, route, vars)
        self.previous_structure = heading_per_layer

        if len(self.createdAircraft) > 0 or len(self.ac_names_to_create) > 0:
            last_created_it = 0
            last_planned_it = 0

            if len( self.createdAircraft) > 0:
                aircraft_name = self.createdAircraft[-1]
                last_created_it = (np.where(self.aircraftNames == aircraft_name)[0]).item(0)

            if len(self.ac_names_to_create) > 0:
                aircraft_name2 = self.ac_names_to_create[-1]
                last_planned_it = (np.where(self.aircraftNames == aircraft_name2)[0]).item(0)

            last_created_it = max(last_created_it, last_planned_it)
            # go to the next
            last_created_it += 1

        while last_created_it < len(self.generation_times) and \
                self.generation_times[last_created_it] <= bs.sim.simt - self.time_first_structure_set:
            # create new Aircraft
            aicraftname = self.aircraftNames[last_created_it]
            alts, route, vars, self.aicraft_starting_per_layer = \
                pf.get_path(aicraftname, self.aircraftRoutes[aicraftname], heading_per_layer, active_layers, self.aicraft_starting_per_layer)

            self.createAircraft(aicraftname, alts, route, vars)
            self.setAircraftRoute(aicraftname, alts, route, vars)
            self.createdAircraft = np.append(self.createdAircraft, aicraftname)
            # self.alts_list.append(alts)
            # self.route_list.append(route)
            # self.vars_list.append(vars)
            # self.ac_names_to_create.append(aicraftname)
            #
            # self.askStartingLayer(alts, route)
            last_created_it += 1

    def getDistanceToAC(self, lat0, lon0,current_layer, intended_layer ):
        tlos_frontac = 3000
        tlos_backac = 3000

        ac_intruders = np.where(current_layer == intended_layer)[0]
        #print('ac_intruders', intended_layer)
        if len(ac_intruders) > 0:
            qdr, dist = geo.kwikqdrdist_matrix(bs.traf.lat[ac_intruders], bs.traf.lon[ac_intruders],
                                               np.ones(len(ac_intruders)) * lat0,
                                               np.ones(len(ac_intruders)) * lon0)
            dist *= nm
            if len(dist) > 1:
                qdr_idx = np.where(np.logical_or(qdr >= 340, qdr <= 20))[0]
                if len(qdr_idx) > 0 and np.any(dist[qdr_idx]) > 0:
                    indx = np.where(dist[qdr_idx] > 0)[0]
                    if len(indx) > 0:
                        dist_frontac = min(dist[qdr_idx][indx])
                        frontac = int(np.where(dist[qdr_idx] == dist_frontac)[0])
                        dvs = TAS_MAX - bs.traf.gs[frontac]
                        tlos_frontac = (dist_frontac - bs.traf.cd.hpz) / -dvs
                        if tlos_frontac < 0:  # it happened in the past
                            tlos_frontac = 3000
                qdr_idx = np.where(np.logical_and(qdr > 160, qdr < 190))[0]
                if len(qdr_idx) > 0 and np.any(dist[qdr_idx]) > 0:
                    dist_backac = min(dist[qdr_idx])
                    backac = int(np.argmin(dist[qdr_idx]))
                    dvs = TAS_MAX - bs.traf.gs[backac]
                    tlos_backac = (dist_backac - bs.traf.cd.hpz) / -dvs
                    if tlos_backac < 0: # it happened in the past
                        tlos_backac = 3000

        return min(tlos_frontac, 3000), min(tlos_backac, 3000), len(ac_intruders)


    def askStartingLayer(self, alts, route):
        current_layer = alts[0]
        layer_wpt = alts[1]
        next_wpt = 0
        while layer_wpt == current_layer and next_wpt < len(alts):
            layer_wpt = alts[next_wpt]
            next_wpt += 1

        current_layer_all = np.zeros(bs.traf.ntraf)

        for ac_id in range(bs.traf.ntraf):
            # get aircraft at each layer
            layer_now = min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[ac_id]))
            # if the ac is in the middle of two rows, we need to decide which one is correct based on vs direction
            if bs.traf.vs[ac_id] > 0 and bs.traf.alt[ac_id] > ALTS_all[layer_now]:
                layer_now += 1
            elif bs.traf.vs[ac_id] < 0 and bs.traf.alt[ac_id] < ALTS_all[layer_now]:
                layer_now -= 1

            layer_now = min(16, layer_now)
            current_layer_all[ac_id] = layer_now

        lat0, lon0 = cb.getLatLonnode(cb.GRAPH, route[0])
        tlos_frontac_traffic_layer, tlos_backac_traffic_layer, number_ac_trafficlayer = self.getDistanceToAC( lat0, lon0, current_layer_all, current_layer)
        tlos_frontac_fastlayer, tlos_backac_fastlayer,number_ac_fastlayer  = self.getDistanceToAC(lat0, lon0, current_layer_all, current_layer + 1)
        tlos_frontac_slowLayer, tlos_backac_slowlayer, number_ac_slowlayer = self.getDistanceToAC( lat0, lon0, current_layer_all, min(current_layer + 2, 16))

        #print('ASKING FOR ACTION', ac_name,bs.traf.alt[ac_id], self.current_layer[ac_id], '-->', self.wpt_layer[ac_id])
        layer_change = 1
        if self.wpt_layer[ac_id] < self.current_layer[ac_id]:
            layer_change = -1

        state = [TAS_MAX, TRAFFIC_LAYER_ID,
                 number_ac_trafficlayer, tlos_frontac_traffic_layer, tlos_backac_traffic_layer,
                 number_ac_fastlayer, tlos_frontac_fastlayer, tlos_backac_fastlayer,
                 number_ac_slowlayer, tlos_frontac_slowLayer, tlos_backac_slowlayer,
                 next_wpt, layer_change]  # dim = 10

        if bs.sim.isRLcreated():
            bs.sim.RLactionCreateAC(bs.stack.get_scenname(), state)
        else:
            bs.sim.RLcreateCreateAC(bs.stack.get_scenname(), state)

    def updateWaypoints(self, alt_dif, alts):
        current_alt_traffic = alts[0]
        alts[0] = min( alts[0] + alt_dif,16)
        it_wpt = 1
        while it_wpt < len(alts) and alts[it_wpt] == current_alt_traffic:
            alts[it_wpt] = min( alts[it_wpt] + alt_dif,16)
            it_wpt += 1

        return alts

    def setStartingLayer(self, action):
        self.DDPG_actions.append(action)
        alts = self.alts_list.pop(0)
        route = self.route_list.pop(0)
        vars = self.vars_list.pop(0)
        aicraftname = self.ac_names_to_create.pop(0)

        layer = np.argmax(action[STATE_DIM:])
        if layer == 1:
            alts = self.updateWaypoints(1, alts)
        elif layer == 2:
            alts = self.updateWaypoints(2 , alts)

        self.DDPG_actions_alt.append(alts[0])
        self.DDPG_actions_ac.append(aicraftname)
        self.DDPG_actions_time.append(bs.sim.simt)
        self.DDPG_actions_conf.append(len(bs.traf.cd.confpairs_all_cruising))
        self.DDPG_actions_los.append(len(bs.traf.cd.lospairs_all_cruising))

        self.aircraftRoutes[aicraftname] = route
        self.createAircraft(aicraftname, alts, route, vars)
        self.setAircraftRoute(aicraftname, alts, route, vars)
        self.createdAircraft = np.append(self.createdAircraft,aicraftname)

    def getTransitLayers(self, layer):
        if layer in [0, 3, 6, 9, 12, 15]:  # aircraft is in a traffic layer
            current_layer_id = 0

        elif layer in [1, 4, 7, 10, 13, 16]:  # aircraft is in a fast layer
            current_layer_id = 1

        elif layer in [-1, 2, 5, 8, 11, 14]:  # aircraft is in a slow layer
            current_layer_id = 2

        return current_layer_id

    def getLoss(self, rewards, los_begin_action, layer_now):
        new_los = bs.traf.cd.lospairs_all_cruising[int(los_begin_action):]
        lospairs_unique = {frozenset(pair) for pair in new_los}
        new_los_pairs = [tuple(x) for x in lospairs_unique]
        for pair in new_los_pairs:
            idx0 = bs.traf.id2idx(pair[0])
            idx1 = bs.traf.id2idx(pair[1])
            if min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[idx0])) == layer_now or \
                    min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[idx1])) == layer_now:
                rewards -= 10
        return rewards

    def getNearMisses(self, rewards, conf_begin_action, layer_now):
        new_conf = bs.traf.cd.confpairs_all_cruising[int(conf_begin_action):]
        confpairs_unique = {frozenset(pair) for pair in new_conf}
        new_conf_pairs = [tuple(x) for x in confpairs_unique]
        for pair in new_conf_pairs:
            idx0 = bs.traf.id2idx(pair[0])
            idx1 = bs.traf.id2idx(pair[1])
            if min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[idx0])) == layer_now or \
                    min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[idx1])) == layer_now:
                rewards -= 1
        return rewards

    def setDDPGreward(self,ac_id, action, conf_begin_action,los_begin_action, iactwp):
        current_layer = min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[ac_id]))
        current_layer_id = self.getTransitLayers(current_layer)

        current_layer_all = np.zeros(bs.traf.ntraf)
        for ac_id in range(bs.traf.ntraf):
            # get aircraft at each layer
            layer_now = min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.alt[ac_id]))
            # if the ac is in the middle of two rows, we need to decide which one is correct based on vs direction
            if bs.traf.vs[ac_id] > 0 and bs.traf.alt[ac_id] > ALTS_all[layer_now]:
                layer_now += 1
            elif bs.traf.vs[ac_id] < 0 and bs.traf.alt[ac_id] < ALTS_all[layer_now]:
                layer_now -= 1

            layer_now = min(16, layer_now)
            current_layer_all[ac_id] = layer_now

        if current_layer_id == TRAFFIC_LAYER_ID:
            tlos_frontac_traffic_layer,  tlos_back_trafficlayer, number_ac_trafficlayer = self.getDistanceToAC(bs.traf.lat[ac_id],bs.traf.lon[ac_id], current_layer_all, current_layer)
            tlos_frontac_fastlayer,tlos_back_fastlayer, number_ac_fastlayer = self.getDistanceToAC( bs.traf.lat[ac_id], bs.traf.lon[ac_id],current_layer_all, current_layer + 1)
            tlos_frontac_slowLayer, tlos_backac_slowlayer, number_ac_slowlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id],current_layer_all, current_layer + 2)
        elif current_layer_id == FAST_LAYER_ID:
            tlos_frontac_traffic_layer,  tlos_back_trafficlayer, number_ac_trafficlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer - 1)
            tlos_frontac_fastlayer, tlos_back_fastlayer, number_ac_fastlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer)
            tlos_frontac_slowLayer,tlos_backac_slowlayer,number_ac_slowlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer + 1)
        else:
            tlos_frontac_traffic_layer,tlos_back_trafficlayer, number_ac_trafficlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer - 2)
            tlos_frontac_fastlayer, tlos_back_fastlayer,  number_ac_fastlayer = self.getDistanceToAC( bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer - 1)
            tlos_frontac_slowLayer, tlos_backac_slowlayer, number_ac_slowlayer = self.getDistanceToAC(bs.traf.lat[ac_id], bs.traf.lon[ac_id], current_layer_all, current_layer)

        layer_wpt = current_layer
        next_wpt = iactwp
        while layer_wpt == current_layer and next_wpt < len(bs.traf.ap.route[ac_id].wpalt):
            layer_wpt = min(range(len(ALTS_all)), key=lambda i: abs(ALTS_all[i] - bs.traf.ap.route[ac_id].wpalt[next_wpt]))
            next_wpt += 1

	layer_change = 1
        if self.wpt_layer[ac_id] < self.current_layer[ac_id]:
            layer_change = -1

        next_state = [bs.traf.tas[ac_id], current_layer_id,
                      number_ac_trafficlayer, tlos_frontac_traffic_layer, tlos_back_trafficlayer,
                      number_ac_fastlayer, tlos_frontac_fastlayer,  tlos_back_fastlayer,
                      number_ac_slowlayer, tlos_frontac_slowLayer,tlos_backac_slowlayer,
                      layer_wpt -  layer_now, layer_change]

        reward = 0
        reward = self.getLoss(reward, los_begin_action, current_layer)
        reward = self.getNearMisses(reward, conf_begin_action, current_layer)
        bs.sim.RLresult(bs.stack.get_scenname(), action[:STATE_DIM], next_state, reward, action[STATE_DIM:])

    def areStructureTheSame(self, old, new):
        for it in range(len(old)):
            if type(old[it]) in [int, np.int32, np.int64] and type(new[it]) in [int, np.int32, np.int64]:
                if old[it] != new[it]:
                    print('different Structures')
                    return False
            elif type(old[it]) not in [int, np.int32, np.int64] and type(new[it]) not in [int, np.int32, np.int64]:
                if len(set(old[it]) & set(new[it])) < len(old[it]):
                    print('different Structures')
                    return False
            else:
                print('different Structures')
                return False

        return True

    def createAircraft(self, aicraftname, alts, route, vars):
        lat0, lon0 = cb.getLatLonnode(cb.GRAPH, route[0])
        lat1, lon1 = cb.getLatLonnode(cb.GRAPH, route[1])

        qdr, dummy = geo.kwikqdrdist(lat0, lon0, lat1, lon1)
        creation_speed = TAS_MAX
        if vars[0] or vars[1]:
            creation_speed = TAS_CURVE

        if alts[0] < 0: # exception
            alt = ALTS[0] # - (LAYER_SEP + ASAS_PZH)s
        else:
            alt = ALTS[int(alts[0])]

        bs.traf.create(1, AC_TYPE, alt* ft, creation_speed, None, lat0, lon0, qdr,
                       aicraftname)

    def setAircraftRoute(self, aicraftname, alts, route, vars):
        # get last point to lead the aircraft to leave the simulation area
        lat_final, lon_final, alt_final = self.getLastPoint(route[-1])
        if alt_final != alts[-1]:
            vars[-1] = True
        else:
            vars[-1] = False
        vars = np.append(vars, False)

        # bs.traf.ap.route[aircraft].delrte()
        aircraft = bs.traf.id2idx(aicraftname)
        bs.traf.ap.setdestorig("DEST", aircraft, str(lat_final), str(lon_final))
        for it in range(1, len(vars)):
            speed = TAS_MAX
            if vars[it] or ( it+1 < len(vars) and vars[it+1]):
                bs.traf.ap.route[aircraft].addwptStack(aircraft, 'FLYTURN')
                bs.traf.ap.route[aircraft].addwptStack(aircraft, 'TURNRAD', TURNRAD * ft)
                speed = TAS_CURVE
            else:
                bs.traf.ap.route[aircraft].addwptStack(aircraft, 'FLYBY')
            if it < len(route):
                lat0, lon0 = cb.getLatLonnode(cb.GRAPH, route[it])
                alt = alts[it]
            else:
                lat0, lon0 = lat_final, lon_final
                # no need to go up or down when leaving the area
                alt = alts[-1]

            bs.traf.ap.route[aircraft].addwptStack(aircraft, str(lat0) + ',' + str(lon0), ALTS[int(alt)] * ft, speed )
            # add one more for the last point outside of the simulation square
            bs.traf.ap.route[aircraft].setRouteNodes(np.append(route[1:], [0]))
            bs.traf.ap.route[aircraft].setRouteStart(route[0])

        # set VNAV ON
        bs.traf.ap.setVNAV(aircraft, True)

    def getLastPoint(self, last_node):
        # add last point to delete aircraft from map as fast as possible
        destination_side = cb.getCurrentBorder(last_node, cb.GRAPH)
        # 0 - north, 1-bottom, 2-east, 3-west
        heading_finalpoint = 0  # up
        alt_final = 2
        if destination_side == 1:  # bottom
            heading_finalpoint = 180
            alt_final = 6
        elif destination_side == 2:  # east
            heading_finalpoint = 90
            alt_final = 4
        elif destination_side == 3:  # left
            heading_finalpoint = 270
            alt_final = 8

        lat0, lon0 = cb.getLatLonnode(cb.GRAPH, last_node)
        lat_final, lon_final = geo.kwikpos(lat0, lon0, heading_finalpoint, 1000 / nm)
        return lat_final, lon_final, alt_final

    def getPointFileName(self):
        scenarioname = bs.stack.get_scenname()
        if 'SSD' in scenarioname:
            repetition = scenarioname.split('-')[-2]
            instance_name = scenarioname.split('-')[-3]
        else:
            # instance_name = scenarioname.split('-')[2]
            # instance_name = instance_name.split('Inst')[1]
            # repetition = scenarioname.split('-')[-1]
            # repetition = repetition.split('_')[0]
            repetition = scenarioname.split('Rep')[1]
            repetition = repetition.split('.')[0]
        repetition = int(repetition)

        return 'point-test-Rep' + str(repetition) + '.scn'

    def readDeliveryPoints(self):
        scenarioname = bs.stack.get_scenname()
        if len(scenarioname) > 0:
            file_name = self.getPointFileName()
            with open(os.path.join(bs.settings.scenario_path, file_name), "r") as reader:
                self.previous_read_points = file_name
                lines = reader.readlines()
                for line in lines:
                    if not line.startswith("#"):
                        line1 = line.split(',')
                        line2 = line.split('(')
                        line2 = line2[1].split(')')
                        line3 = line.split('[')
                        aircraftname = line1[0]
                        self.aircraftNames = np.append(self.aircraftNames, aircraftname)
                        time = datetime.strptime(line1[1].split(',')[0], '%H:%M:%S.%f')
                        time = time.hour * 3600 + time.minute * 60 + time.second
                        self.generation_times = np.append(self.generation_times, time)
                        if line2[0][-1]==',':
                            line2[0] = line2[0][:-1]
                        self.deliverypoints[aircraftname] = [int(float(i)) for i in line2[0].replace(' ', '').split(',')]
                        self.aircraftRoutes[aircraftname] = [int(float(i)) for i in line3[1][:-2].replace(' ', '').split(',')]
