''' Conflict resolution based on the SSD algorithm. '''
import geopandas

from bluesky.traffic import route
from bluesky.traffic.asas import ConflictResolution
from bluesky.tools import geo
from bluesky.tools.aero import nm
import numpy as np
import bluesky as bs
from bluesky import sim
from shapely.ops import unary_union
from shapely.geometry import Point
from shapely.geometry.polygon import Polygon, LineString
from shapely.ops import nearest_points
from bluesky.tools.aero import ft, nm

G0 = 9.80665  # m/s2    Sea level gravity constant
R_EARTH = 6378000  # m    Radius of the Earth

LAYER_SEP = 5  #5 ft
ASAS_PZH = bs.settings.asas_pzh # ft
ALT_MIN = 65 # ft
ALTS_ALL = np.arange(ALT_MIN, ALT_MIN + (LAYER_SEP + ASAS_PZH)*17, LAYER_SEP + ASAS_PZH) # in ft
ALTS_TRAFFIC = np.arange(ALT_MIN, 400, ASAS_PZH*3 + LAYER_SEP*3)

# Try to import pyclipper
try:
    import pyclipper
except ImportError:
    print("Could not import pyclipper, RESO SSD will not function")


# TODO: not completely migrated yet to class-based implementation


def init_plugin():
    # Addtional initilisation code

    # Configuration parameters
    config = {
        # The name of your plugin
        'plugin_name': 'SSD',

        # The type of this plugin. For now, only simulation plugins are possible.
        'plugin_type': 'sim'
    }

    # init_plugin() should always return these two dicts.
    return config, {}


class SSD(ConflictResolution):
    def setprio(self, flag=None, priocode=''):
        '''Set the prio switch and the type of prio '''
        if flag is None:
            return True, "PRIORULES [ON/OFF] [PRIOCODE]" + \
                   "\nAvailable priority codes: " + \
                   "\n     RS1:  Shortest way out" + \
                   "\n     RS2:  Clockwise turning" + \
                   "\n     RS3:  Heading first, RS1 second" + \
                   "\n     RS4:  Speed first, RS1 second" + \
                   "\n     RS5:  Shortest from target" + \
                   "\n     RS6:  Rules of the air" + \
                   "\n     RS7:  Sequential RS1" + \
                   "\n     RS8:  Sequential RS5" + \
                   "\n     RS9:  Counterclockwise turning" + \
                   "\nPriority is currently " + ("ON" if self.swprio else "OFF") + \
                   "\nPriority code is currently: " + \
                   str(self.priocode)
        options = ["RS1", "RS2", "RS3", "RS4",
                   "RS5", "RS6", "RS7", "RS8", "RS9"]
        if priocode not in options:
            return False, "Priority code Not Understood. Available Options: " + str(options)
        return super().setprio(flag, priocode)

    def loaded_pyclipper(asas):
        """ Return true if pyclipper is successfully loaded """
        import sys
        return "pyclipper" in sys.modules

    def calcTurn(self, latintruder, lonintruder, bank, routeintruder, currentLeg, tas):
        wpqdr, dist = geo.qdrdist(latintruder, lonintruder, routeintruder.wplat[currentLeg],
                                  routeintruder.wplon[currentLeg])
        # Get distance from [nm] to [m]
        dist = dist * nm

        if routeintruder.nwp > currentLeg + 1:  # and \
            #   ( routeintruder.wplat[currentLeg] !=   routeintruder.wplat[currentLeg + 1] and \
            #   routeintruder.wplon[currentLeg] != routeintruder.wplon[currentLeg + 1]):
            nextqdr, distance = geo.qdrdist(routeintruder.wplat[currentLeg], routeintruder.wplon[currentLeg],
                                            routeintruder.wplat[currentLeg + 1], routeintruder.wplon[currentLeg + 1])
            # Turn radius in meters!
            turnrad = np.square(tas) / (np.maximum(0.01, np.tan(bank)) * G0)

            # turn angle
            if distance == 0:
                turnangle = 0
            else:
                turnangle = np.abs(wpqdr % 360. - nextqdr % 360.)

            # define turn direction
            if turnangle > 180:
                turnangle = 360. - turnangle

            turnangle = np.deg2rad(turnangle)

            # turndist in meters
            if np.pi - turnangle < 1:  # singularity
                turndist = turnrad
            else:
                turndist = np.where(nextqdr is -999, nextqdr, np.abs(turnrad * np.tan(0.5 * turnangle)))

            # turndist = np.abs(turnrad *
            #                   np.tan(np.radians(0.5 * np.abs(degto180(wpqdr % 360. - next_wpqdr % 360.)))))

            turnrate = G0 * np.tan(bank) / tas
        else:
            turndist = turnrad = turnrate = turnangle = 0

        return np.array([float(dist), float(turndist), float(turnrad), float(turnrate), float(turnangle),
                         np.deg2rad(float(wpqdr) % 360.)])

    def calculateIntruderTurns(self, lat, lon, route, bank, tas, turns):
        if route.iactwp >= 0:
            for i in range(route.iactwp, len(route.wplat)):
                if (route.wplat[i] == lat and route.wplon[i] == lon):  # the last waypoint is normally repeated
                    continue
                new_turn = self.calcTurn(lat, lon, bank, route, i, tas)
                turns = np.vstack([turns, new_turn])
                lat = route.wplat[i]
                lon = route.wplon[i]
        return turns

    def getNewLocationWithOffset(self, lat0, lon0, distance, heading):
        # flat-earth approximation

        # Coordinate offsets in radians
        dLat = distance * np.cos(heading) / R_EARTH
        newlat = lat0 + np.rad2deg(dLat)

        coslat = np.cos(np.deg2rad(newlat))

        dLon = distance * np.sin(heading) / coslat / R_EARTH

        newlon = lon0 + np.rad2deg(dLon)

        return newlat, newlon

    def getTimeAtLocation(self, lat0, lon0, lat_end, lon_end, tas, ax, acceltime):
        dummy, distance = geo.qdrdist_matrix(lat0, lon0, lat_end,
                                             lon_end)

        distance = distance * nm

        time = distance / tas

        return time

    def getNewLocationAfterTime(self, time, lat0, lon0, tas, newbearing, ax, acceltime):
        gseast = tas * np.sin(newbearing)
        gsnorth = tas * np.cos(newbearing)

        aeast = ax * np.sin(newbearing)
        anorth = ax * np.cos(newbearing)

        dLat = acceltime * gsnorth + 0.5 * anorth * acceltime ** 2 + (gsnorth + anorth * acceltime) * (time - acceltime)
        dLon = acceltime * gseast + 0.5 * aeast * acceltime ** 2 + (gseast + aeast * acceltime) * (time - acceltime)

        newlat = lat0 + np.rad2deg(dLat / R_EARTH)
        coslat = np.cos(np.deg2rad(newlat))
        newlon = lon0 + np.rad2deg(dLon / coslat / R_EARTH)

        return newlat, newlon

    def addToVector(self, vector, newelement):
        return np.append(vector, newelement)

    def gettimetocurve(self, distancetoNextCurve, tas, ax, accelerationtime):
        distanceinaccelerationtime = tas * accelerationtime + 0.5 * ax * accelerationtime ** 2

        if distanceinaccelerationtime >= distancetoNextCurve:
            timetoCurve = np.roots([ax * 0.5, tas, -distancetoNextCurve])
            timetoCurve = timetoCurve[np.isreal(timetoCurve)]
            timetoCurve = timetoCurve[timetoCurve > 0]
            timetoCurve = np.amin(timetoCurve)
        else:
            timetoCurve = (distancetoNextCurve - distanceinaccelerationtime) / tas + accelerationtime

        return timetoCurve

    def determineIntruderPosition(self, turns, initialLat, initialLon, bearing, tas, routeintruder, ax, minimumTime,
                                  maximumTime):
        lat = np.array([])
        lon = np.array([])
        tc = np.array([])
        last_lat = -1

        currentLeg = 0
        newlat = initialLat
        newlon = initialLon

        # currenttc = 1
        currenttc = minimumTime
        timestartleg = 0
        newbearing = np.deg2rad(bearing % 360.)
        if (currentLeg < len(turns)):
            newbearing = turns[currentLeg][5]
            distanceToNextWPT = turns[currentLeg][0]

        # lets consider that the acceleration will go for most for 10 seconds
        # otherwise for a high "time" the speed change will be considerable
        accelerationtime = 30  # seconds
        isThereNextCurve = True
        while isThereNextCurve and currenttc < maximumTime:  # ( len(turns) == 0 and):

            isThereNextCurve = currentLeg < len(turns) - 1
            if (isThereNextCurve):
                distancetoNextCurve = distanceToNextWPT - turns[currentLeg][1]
                if distancetoNextCurve > 0:
                    timetoCurve = self.gettimetocurve(distancetoNextCurve, tas, ax,
                                                      max(accelerationtime - timestartleg, 0))
                    timeToStartCurve = timetoCurve + timestartleg
                    timeToCompleteCurve = timeToStartCurve + turns[currentLeg][4] / turns[currentLeg][3]
                else:
                    timeToStartCurve = 0
                    # how much time do we still need to achieve the desired angle
                    turnangle = np.abs(np.rad2deg(turns[currentLeg + 1][5]) % 360. - np.rad2deg(newbearing) % 360.)
                    if turnangle > 180:
                        turnangle = 360. - turnangle
                    timeToCompleteCurve = (np.deg2rad(turnangle) / turns[currentLeg][3])
                    if timeToCompleteCurve < 0 or not np.isfinite(timeToCompleteCurve):
                        timeToCompleteCurve = 0

            if isThereNextCurve and currenttc >= timeToCompleteCurve:  # we are in next leg
                distanceToNextWPT = turns[currentLeg + 1][0] - turns[currentLeg][1]
                # get the position and the end of the curve
                timestartleg = timeToCompleteCurve
                newbearing = turns[currentLeg + 1][5] % (2 * np.pi)
                currentWaypoint = currentLeg + np.where(routeintruder.iactwp >= 0, routeintruder.iactwp, 0)
                newlat, newlon = self.getNewLocationWithOffset(routeintruder.wplat[currentWaypoint],
                                                               routeintruder.wplon[currentWaypoint],
                                                               turns[currentLeg][1], newbearing)
                last_lat = routeintruder.wpalt[currentWaypoint]
                if (int(timeToCompleteCurve) > 0):
                    currenttc = timeToCompleteCurve
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currenttc)  # end of the curve
                else:
                    newlat, newlon = self.getNewLocationAfterTime(currenttc + 1, newlat, newlon, tas, newbearing, ax,
                                                                  max(accelerationtime - currenttc, 0))
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currenttc + 1)
                currenttc += 1
                currentLeg += 1

            elif not isThereNextCurve or currenttc < timeToStartCurve:  # we are before curve
                # using flat-earth approximationnewlon
                timesincecurve = np.where(currentLeg == 0, currenttc, currenttc - timestartleg)
                newlat, newlon = self.getNewLocationAfterTime(timesincecurve, newlat, newlon, tas, newbearing, ax,
                                                              max(accelerationtime - currenttc, 0))
                lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                              currenttc)
                previoustc = currenttc
                if isThereNextCurve:  # beginning of the curve
                    # lets get the beginning of the curve
                    addtocurrenttc = (timeToStartCurve - currenttc) / 3
                    currenttc += addtocurrenttc
                    newlat, newlon = self.getNewLocationAfterTime(currenttc - previoustc, newlat, newlon, tas,
                                                                  newbearing,
                                                                  ax,
                                                                  max(accelerationtime - currenttc, 0))
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currenttc)

                    previoustc = currenttc
                    currenttc += addtocurrenttc
                    newlat, newlon = self.getNewLocationAfterTime(currenttc - previoustc, newlat, newlon, tas,
                                                                  newbearing,
                                                                  ax,
                                                                  max(accelerationtime - currenttc, 0))
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currenttc)

                    previoustc = currenttc
                    currenttc += addtocurrenttc
                    newlat, newlon = self.getNewLocationAfterTime(currenttc - previoustc, newlat, newlon, tas,
                                                                  newbearing,
                                                                  ax,
                                                                  max(accelerationtime - currenttc, 0))
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currenttc)

                    newbearing = turns[currentLeg][5] % (2 * np.pi)
                    currenttc = timeToStartCurve
                else:  # last one
                    lat_destination = routeintruder.wplat[-1]
                    lon_destination = routeintruder.wplon[-1]
                    # get at what time the aicraft will finish its path
                    time_end = self.getTimeAtLocation(newlat, newlon, lat_destination, lon_destination, tas, ax,
                                                      max(accelerationtime - currenttc, 0))
                    lat, lon, tc = self.addToVector(lat, lat_destination), self.addToVector(lon,
                                                                                            lon_destination), self.addToVector(
                        tc,
                        time_end + currenttc)

            elif isThereNextCurve and currenttc >= timeToStartCurve and currenttc < timeToCompleteCurve:  # we are in curve
                delhdg = (np.rad2deg(turns[currentLeg + 1][5]) - np.rad2deg(
                    turns[currentLeg][5]) + 180) % 360 - 180  # [deg]
                # timeleft = timeToCompleteCurve - timeToStartCurve
                quarterTime = (timeToCompleteCurve - currenttc) / 4
                currentTime = currenttc
                for curve_piece in range(4):
                    newlat, newlon = self.getNewLocationAfterTime(quarterTime, newlat, newlon, tas, newbearing, 0, 0)
                    newbearing += turns[currentLeg][3] * np.sign(delhdg) * quarterTime
                    newbearing = newbearing % (2 * np.pi)
                    currentTime += quarterTime
                    currentTime = min(currentTime, timeToCompleteCurve)
                    lat, lon, tc = self.addToVector(lat, newlat), self.addToVector(lon, newlon), self.addToVector(tc,
                                                                                                                  currentTime)
                newbearing = turns[currentLeg + 1][5]
                currenttc = timeToCompleteCurve

        return lat, lon, tc, last_lat

    def buildVO_intent(self, intruderLat, intruderLon, tc, hsepm, latownship, lonownship,
                       vmax):  # , to_graph_comparison):

        firstlinex = np.array([])
        firstliney = np.array([])
        secondlinex = np.array([])
        secondliney = np.array([])
        r = np.array([])
        center = np.empty((0, 2), float)
        VOdata = []

        for i3 in range(len(intruderLat)):

            if tc[i3] == 0:  # singularity
                continue

            # in tc, the onwship will have to travel this amount ohsepf time
            [qrd1, dist1] = geo.qdrdist_matrix(latownship, lonownship, intruderLat[i3], intruderLon[i3])
            # Get distance from [nm] to [m]
            dist1 = dist1 * nm

            qrd1 = np.deg2rad(qrd1)

            # In LoS the VO can't be defined, act as if dist is on edge
            if dist1 < hsepm:
                dist1 = hsepm

            distx = dist1 * np.sin(qrd1)
            disty = dist1 * np.cos(qrd1)
            # vc
            vcx = distx / tc[i3]
            vcy = disty / tc[i3]

            dvcx = float(-distx / np.square(tc[i3]))
            dvcy = float(-disty / np.square(tc[i3]))

            radius = hsepm / tc[i3]
            dr = - hsepm / np.square(tc[i3])

            if np.sqrt(np.square(dvcx) + np.square(dvcy)) >= abs(dr) and np.isfinite(dvcx) \
                    and np.isfinite(dvcy) and np.isfinite(dr):

                thetaupper = np.roots([-dvcx + dr, 2 * dvcy, dvcx + dr])
                thetalower = 2 * np.arctan(thetaupper)
                vx = vcx + radius * np.cos(thetalower)
                vy = vcy + radius * np.sin(thetalower)

                if vx.item(0) == vx.item(1):
                    continue

                if (vx.size == 0 and vy.size == 0) or np.any(np.iscomplex(vx)) or np.any(np.iscomplex(vy)):
                    continue

                first_ind = 0
                second_ind = 1

                if firstlinex.size > 0 and secondlinex.size > 0 and firstliney.size > 0 and secondliney.size > 0:
                    line1 = LineString([[firstlinex[-1], firstliney[-1]], [vx.item(first_ind), vy.item(first_ind)]])
                    line2 = LineString([[secondlinex[-1], secondliney[-1]], [vx.item(second_ind), vy.item(second_ind)]])
                    if line1.intersection(line2):
                        second_ind = 0
                        first_ind = 1

                firstlinex = np.append(firstlinex, vx.item(first_ind))
                secondlinex = np.append(secondlinex, vx.item(second_ind))

                firstliney = np.append(firstliney, vy.item(first_ind))
                secondliney = np.append(secondliney, vy.item(second_ind))

                # x = np.append(x, vx)
                # y = np.append(y,vy)
                # n = np.append(n, int(i3 == len(intruderLat) - 1))
                r = np.append(r, radius)
                center = np.append(center, [[float(vcx), float(vcy)]], axis=0)

        # build the tip of the triangle
        # Get vertices in an x- and y-array of size (ntraf-1)*3x1
        # if not build_SSD:
        #     x = np.array((gseast[i_other],
        #                   firstlinex[-1],
        #                   secondlinex[-1]))
        #     y = np.array((gsnorth[i_other],
        #                   firstliney[-1],
        #                   secondliney[-1]))
        #     xy_vo = np.dstack((x, y))
        #     #to_graph_comparison.append(xy_vo[0])
        #     if not ((abs(x) > 1e7).any() or (abs(y) > 1e7).any()):
        #         VOdata.append(xy_vo)

        it_vo = 0

        while it_vo < len(firstlinex) - 1:
            increment = 1
            valid_poly = False
            # make sure that the lines do not cross
            while not valid_poly:
                increment += 1
                if it_vo + increment < len(firstlinex):
                    vo_x = np.array(
                        [firstlinex[it_vo], firstlinex[it_vo + increment], secondlinex[it_vo + increment],
                         secondlinex[it_vo]])
                    vo_y = np.array(
                        [firstliney[it_vo], firstliney[it_vo + increment], secondliney[it_vo + increment],
                         secondliney[it_vo]])
                    # if all points are outside of the performance limits of the ownship there is no point in drawing this.
                    if (abs(vo_x) <= vmax).any() and (abs(vo_y) <= vmax).any():
                        vo_values = np.dstack((vo_x, vo_y))
                        poly = Polygon(vo_values[0])
                        valid_poly = poly.is_valid
                else:
                    break

            if valid_poly:
                VOdata.append(poly)
            it_vo += increment

        # if len(center) > 1:
        #     firstCircle = center[0] + xyc * r[0]
        #     secondCircle = center[-1] + xyc * r[-1]
        #     # first circle - so we don't cut half of the circle off
        #     # if not (abs(firstCircle) > vmax).all():
        #     if not (abs(firstCircle) > 1e7).any():
        #         firstCircle = tuple(map(tuple, np.flipud(firstCircle)))
        #         VOdata.append([np.flipud(firstCircle)])
        #
        #     # second circle  - so we don't cut half of the circle off
        #     # if not (abs(secondCircle) > vmax).all():
        #     if not (abs(secondCircle) > 1e7).any():
        #         secondCircle = tuple(map(tuple, np.flipud(secondCircle)))
        #         # if the radius is too small, then the coordinates will be too close to each other and pyclipper will throw an error
        #         # if secondCircle and len(secondCircle) > 2 and r[-1] >1:
        #         VOdata.append([np.flipud(secondCircle)])

        return VOdata

    def getXYPosition(self, plat, plon, p0lat, p0lon):
        if type(plat) is not np.ndarray:
            qdr, distance = geo.kwikqdrdist(p0lat, p0lon, plat, plon)
        else:
            [qdr, distance] = geo.kwikqdrdist_matrix(p0lat, p0lon, plat, plon)
            qdr = np.squeeze(np.asarray(qdr))
            distance = np.squeeze(np.asarray(distance))

        qdr_radians = np.deg2rad(qdr)
        distance = distance * nm

        return np.multiply(distance, np.sin(qdr_radians)), np.multiply(distance, np.cos(qdr_radians))

    def detect(asas, traf):
        """ Detect all current conflicts """

        # Check if ASAS is ON first!
        if not asas.swasas:
            return

        # Construct the SSD
        # constructSSD(asas, traf)

    def resolve(self, conf, ownship, intruder, active, only_active=False, computeVOS=True,
                polygons_VOs=[], polygons_VOs_intent=[], polygons_geofences=[]):
        self.priocode = "RS5"

        # Initialize SSD variables with ntraf
        self.initializeSSD(conf, ownship.ntraf)

        # Construct the SSD (twice in sequential methods)
        if self.priocode == "RS7":
            self.constructSSD(conf, ownship, "RS1")
        if self.priocode == "RS8":
            self.constructSSD(conf, ownship, "RS5")

        if computeVOS:
            polygons_VOs, polygons_VOs_intent, polygons_geofences = self.constructSSD(conf, ownship, active,
                                                                                      self.priocode,
                                                                                      only_active)

        # Get resolved speed-vector
        found_solution = self.calculate_resolution(conf, ownship, polygons_VOs, polygons_VOs_intent, polygons_geofences, active,
                                  only_active)

        # Now assign resolutions to variables in the ASAS class
        # Start with current states, need a copy, otherwise it changes traf!
        newtrack = np.copy(ownship.hdg)
        newgs = np.copy(ownship.gs)
        # Calculate new track and speed
        # No need to cap the speeds, since SSD implicitly caps
        new_trk = np.arctan2(conf.asase, conf.asasn) * 180 / np.pi
        new_gs = np.sqrt(conf.asase ** 2 + conf.asasn ** 2)

        # Sometimes an aircraft is in conflict, but no solutions could be found
        # In that case it is assigned 0 by ASAS, but needs to handled
        asas_cmd = np.logical_and(np.logical_or(conf.inconf, active), new_gs > 0)

        # Assign new track and speed for those that are in conflict
        newtrack[asas_cmd] = new_trk[asas_cmd]
        newgs[asas_cmd] = new_gs[asas_cmd]
        # we are going to vertical deviation through all layers
        # the ones that do not have a solution yet, we can use vertical avoidance
        # and check if it is safe for aicraft that wasnt tot move,to actually move (calculate free spaces)
        newvs, alt, active = self.getVS(ownship, conf, found_solution, active)
        # newvs = ownship.vs
        # alt = ownship.selalt

        # Cap the velocity
        vmin, vmax, vsmin, vsmax = ownship.perf.currentlimits()
        newgscapped = np.maximum(vmin, np.minimum(vmax, newgs))

        return newtrack, newgscapped, newvs, alt, np.copy(
            ownship.ap.trk), polygons_VOs, polygons_VOs_intent, polygons_geofences

    def getVS(self,ownship, conf, found_solution, active):
        vs = ownship.ap.vs

        current_alts = np.array([])
        next_alts = np.array([])

        # aircraft moving to occupy fast layer
        aircraft_speed_layer = np.array([])

        # get current and next alt for all
        for it in range(ownship.ntraf):
            # must the aircraft move up or down according to its route?
            iactwp = bs.traf.ap.route[it].iactwp
            next_alts = np.append(next_alts, bs.traf.ap.route[it].wpalt[iactwp] / ft)
            current_alts = np.append(current_alts, bs.traf.alt[it] / ft)

        alt = current_alts[:]
        for it in range(ownship.ntraf):
            closest_traffic_layer = min(range(len(ALTS_TRAFFIC)), key=lambda i: abs(ALTS_TRAFFIC[i] - current_alts[it]))
            if abs(next_alts[it] - current_alts[it]) > 5: # have to go down or up
                # if we have a vertical conflict do not let the aircraft move vertically just now
                if self.isInVertConf(conf, next_alts[it] , current_alts, it):
                    vs[it] = 0  # delay moving
                    alt[it] = current_alts[it] # stop the moving
            elif conf.inconf[it] and abs(current_alts[it]-ALTS_TRAFFIC[closest_traffic_layer] < 5) \
                    and not found_solution[it] and self.check_there_is_still_conflict(it, conf, current_alts, next_alts):
                # lets use the transition fast layer to perform vertical avoidance
                next_alts[it] += LAYER_SEP + ASAS_PZH
                #print('vertical avoidance', self.isInVertConf(conf, it, current_alts, next_alts))
                if not self.isInVertConf(conf, next_alts[it], current_alts, it):
                    # do not move, if the other aircraft this aircraft is in conflict with, is already moving
                    intruders = self.getIntruders(conf.confpairs, bs.traf.id[it])
                    #print('not vertical conflicts', intruders, not np.all(np.in1d(intruders, aircraft_speed_layer)))
                    if not np.all(np.in1d(intruders, aircraft_speed_layer)):
                        aircraft_speed_layer = np.append(aircraft_speed_layer, it)
                        alt[it] = next_alts[it]
                        vs[it] = 7.62
                        active[it] = True
                        #print('aircraft it', it, 'will climb')
                else:
                    next_alts[it] -= LAYER_SEP + ASAS_PZH
        return vs, alt, active

    def getIntruders(self, confpairs, aircraft):
        own, intr = zip(*confpairs)
        index_own = [i for i, x in enumerate(own) if x == aircraft]
        intruders = np.array([])
        for index in index_own:
            intruders = np.append(intruders, bs.traf.id2idx(intr[index]))

        return intruders

    def isInVertConf(self, conf, next_alt, current_alts, it):
        aircraft_in_next_alt = np.where(abs(current_alts - next_alt) < 5)[0]

        # are aircraft in conflict?
        if len(aircraft_in_next_alt) > 0:
            for intruder in aircraft_in_next_alt:
                dist = geo.kwikdist(bs.traf.lat[it], bs.traf.lon[it], bs.traf.lat[intruder], bs.traf.lon[intruder])
                if dist*nm < conf.rpz/ft:
                    return True

        return False

    def check_there_is_still_conflict(self, it, conf, current_alts,next_alts):
        in_conf_with = self.getIntruders(conf.confpairs, bs.traf.id[it])
        climb_ownship = next_alts[it] > current_alts[it] #true - climb, false- descend or stay the same

        for intruder in np.where(in_conf_with)[0]:
            climb_intruder = next_alts[intruder] > current_alts[intruder]  # true - climb, false- descend or stay the same
            # they are in conflict if they have opposite maneuvers: one is climbling and the other one is descending
            # or they are both doing the same starting from the same altitude
            if (current_alts[it] == current_alts[intruder] and climb_ownship == climb_intruder) \
                or climb_ownship != climb_intruder:
                return True

        return False

    def initializeSSD(self, conf, ntraf):
        """ Initialize variables for SSD """
        # Need to do it here, since ASAS.reset doesn't know ntraf
        conf.FRV = [None] * ntraf
        conf.ARV = [None] * ntraf
        # For calculation purposes
        conf.ARV_calc = [None] * ntraf
        conf.inrange = [None] * ntraf
        # asas.inconf       = np.zeros(ntraf, dtype=bool)
        # Index 2 for sequential solutions (RS7, RS8)
        conf.ARV_calc2 = [None] * ntraf
        conf.inrange2 = [None] * ntraf
        conf.inconf2 = np.zeros(ntraf, dtype=bool)
        # Stores resolution vector, also used in visualization
        conf.asasn = np.zeros(ntraf, dtype=np.float32)
        conf.asase = np.zeros(ntraf, dtype=np.float32)
        # Area calculation
        conf.FRV_area = np.zeros(ntraf, dtype=np.float32)
        conf.ARV_area = np.zeros(ntraf, dtype=np.float32)
        conf.ap_free = np.ones(ntraf, dtype=bool)

        # asas is an object of the ASAS class defined in asas.py

    def constructSSD(self, conf, ownship, active, priocode="RS1", only_active=False):
        """ Calculates the FRV and ARV of the SSD """
        N = 0
        # Parameters
        N_angle = 180  # [-] Number of points on circle (discretization)
        hsep = conf.rpz  # [m] Horizontal separation (5 NM)
        margin = self.resofach  # [-] Safety margin for evasion
        hsepm = hsep * margin  # [m] Horizontal separation with safety margin
        alpham = 0.4999 * np.pi  # [rad] Maximum half-angle for VO
        betalos = np.pi / 4  # [rad] Minimum divertion angle for LOS (45 deg seems optimal)
        adsbmax = 65. * nm  # [m] Maximum ADS-B range
        beta = np.pi / 4 + betalos / 2
        if priocode == "RS7" or priocode == "RS8":
            adsbmax /= 2

        # Relevant info from traf
        gsnorth = ownship.gsnorth
        gseast = ownship.gseast
        lat = ownship.lat
        lon = ownship.lon
        ntraf = ownship.ntraf
        hdg = ownship.hdg
        gs_ap = ownship.ap.tas
        hdg_ap = ownship.ap.trk
        apnorth = np.cos(hdg_ap / 180 * np.pi) * gs_ap
        apeast = np.sin(hdg_ap / 180 * np.pi) * gs_ap

        # Local variables, will be put into asas later
        FRV_loc = [None] * ownship.ntraf
        ARV_loc = [None] * ownship.ntraf
        # For calculation purposes
        ARV_calc_loc = [None] * ownship.ntraf
        FRV_area_loc = np.zeros(ownship.ntraf, dtype=np.float32)
        ARV_area_loc = np.zeros(ownship.ntraf, dtype=np.float32)

        # # Use velocity limits for the ring-shaped part of the SSD
        # Discretize the circles using points on circle
        angles = np.arange(0, 2 * np.pi, 2 * np.pi / N_angle)
        # Put points of unit-circle in a (180x2)-array (CW)
        xyc = np.transpose(np.reshape(np.concatenate((np.sin(angles), np.cos(angles))), (2, N_angle)))

        # If no traffic
        if ntraf == 0:
            return

        route = ownship.ap.route
        bank = ownship.bank
        ax = ownship.ax

        # # If only one aircraft
        # elif ntraf == 1:
        #     # Map them into the format ARV wants. Outercircle CCW, innercircle CW
        #     ARV_loc[0] = circle_lst
        #     FRV_loc[0] = []
        #     ARV_calc_loc[0] = ARV_loc[0]
        #     # Calculate areas and store in asas
        #     FRV_area_loc[0] = 0
        #     ARV_area_loc[0] = np.pi * (vmax ** 2 - vmin ** 2)
        #     return

        # Function qdrdist_matrix needs 4 vectors as input (lat1,lon1,lat2,lon2)
        # To be efficient, calculate all qdr and dist in one function call
        # Example with ntraf = 5:   ind1 = [0,0,0,0,1,1,1,2,2,3]
        #                           ind2 = [1,2,3,4,2,3,4,3,4,4]
        # This way the qdrdist is only calculated once between every aircraft
        # To get all combinations, use this function to get the indices
        ind1, ind2 = self.qdrdist_matrix_indices(ntraf)
        # Get absolute bearing [deg] and distance [nm]
        # Not sure abs/rel, but qdr is defined from [-180,180] deg, w.r.t. North
        [qdr, dist] = geo.qdrdist_matrix(lat[ind1], lon[ind1], lat[ind2], lon[ind2])
        # Put result of function from matrix to ndarray
        qdr = np.reshape(np.array(qdr), np.shape(ind1))
        dist = np.reshape(np.array(dist), np.shape(ind1))
        # SI-units from [deg] to [rad]
        qdr = np.deg2rad(qdr)
        # Get distance from [nm] to [m]
        dist = dist * nm

        # In LoS the VO can't be defined, act as if dist is on edge
        dist[dist < hsepm] = hsepm

        # Calculate vertices of Velocity Obstacle (CCW)
        # These are still in relative velocity space, see derivation in appendix
        # Half-angle of the Velocity obstacle [rad]
        # Include safety margin
        alpha = np.arcsin(hsepm / dist)
        # Limit half-angle alpha to 89.982 deg. Ensures that VO can be constructed
        alpha[alpha > alpham] = alpham
        # Relevant sin/cos/tan
        sinqdr = np.sin(qdr)
        cosqdr = np.cos(qdr)
        tanalpha = np.tan(alpha)
        cosqdrtanalpha = cosqdr * tanalpha
        sinqdrtanalpha = sinqdr * tanalpha

        polygons_VOs = []
        polygons_VOs_intent = []
        polygons_geofences = []

        # Consider every aircraft
        for i in range(ntraf):

            nextnode_lat = ownship.ap.route[i].wplat[ownship.ap.route[i].iactwp]
            nextnode_lon = ownship.ap.route[i].wplon[ownship.ap.route[i].iactwp]
            # bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(0), [lat[i], lon[i], nextnode_lat, nextnode_lon])
            # for route_i in range(ownship.ap.route[i].iactwp, len(ownship.ap.route[i].wplat) - 1):
            #     line_route = np.array([ownship.ap.route[i].wplat[route_i],
            #                            ownship.ap.route[i].wplon[route_i],
            #                            ownship.ap.route[i].wplat[route_i + 1],
            #                            ownship.ap.route[i].wplon[route_i + 1]])
            #     bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(route_i), line_route)

            polygons_VOs_i = []
            polygons_VOs_intent_i = []
            polygons_geofences_i = []

            # Calculate SSD only for aircraft in conflict (See formulas appendix)
            if (not only_active and (conf.inconf[i] or active[i])) or active[i]:

                # vmin = ownship.perf.vmin[i]
                # vmax = ownship.perf.vmax[i]
                vmin, vmax, vsmin, vsmax = ownship.perf.currentlimits()

                # in the first time step, ASAS runs before perf, which means that his value will be zero
                # and the SSD cannot be constructed
                if vmin == vmax == 0:
                    continue

                # Map them into the format pyclipper wants. Outercircle CCW, innercircle CW
                # circle_tup = (tuple(map(tuple, np.flipud(xyc * vmax))), tuple(map(tuple, xyc * vmin)))
                # circle_lst = [list(map(list, np.flipud(xyc * vmax))), list(map(list, xyc * vmin))]

                # Relevant x1,y1,x2,y2 (x0 and y0 are zero in relative velocity space)
                x1 = (sinqdr + cosqdrtanalpha) * 2 * vmax
                x2 = (sinqdr - cosqdrtanalpha) * 2 * vmax
                y1 = (cosqdr - sinqdrtanalpha) * 2 * vmax
                y2 = (cosqdr + sinqdrtanalpha) * 2 * vmax

                # SSD for aircraft i
                # Get indices that belong to aircraft i
                ind = np.where(np.logical_or(ind1 == i, ind2 == i))[0]
                # Check whether there are any aircraft in the vicinity
                if len(ind) == 0:
                    # No aircraft in the vicinity
                    # Map them into the format ARV wants. Outercircle CCW, innercircle CW
                    # ARV_loc[i] = circle_lst
                    FRV_loc[i] = []
                    ARV_calc_loc[i] = ARV_loc[i]
                    # Calculate areas and store in asas
                    FRV_area_loc[i] = 0
                    ARV_area_loc[i] = np.pi * (vmax ** 2 - vmin ** 2)
                else:
                    # The i's of the other aircraft
                    i_other = np.delete(np.arange(0, ntraf), i)
                    # Aircraft that are within ADS-B range
                    ac_adsb = np.where(dist[ind] < adsbmax)[0]
                    # Now account for ADS-B range in indices of other aircraft (i_other)
                    ind = ind[ac_adsb]
                    i_other = i_other[ac_adsb]
                    if not priocode == "RS7" and not priocode == "RS8":
                        # Put it in class-object (not for RS7 and RS8)
                        conf.inrange[i] = i_other
                    else:
                        conf.inrange2[i] = i_other
                    # VO from 2 to 1 is mirror of 1 to 2. Only 1 to 2 can be constructed in
                    # this manner, so need a correction vector that will mirror the VO
                    fix = np.ones(np.shape(i_other))
                    fix[i_other < i] = -1
                    # Relative bearing [deg] from [-180,180]
                    # (less required conversions than rad in RotA)
                    fix_ang = np.zeros(np.shape(i_other))
                    fix_ang[i_other < i] = 180.

                    # Get vertices in an x- and y-array of size (ntraf-1)*3x1
                    x = np.concatenate((gseast[i_other],
                                        x1[ind] * fix + gseast[i_other],
                                        x2[ind] * fix + gseast[i_other]))
                    y = np.concatenate((gsnorth[i_other],
                                        y1[ind] * fix + gsnorth[i_other],
                                        y2[ind] * fix + gsnorth[i_other]))
                    # Reshape [(ntraf-1)x3] and put arrays in one array [(ntraf-1)x3x2]
                    x = np.transpose(x.reshape(3, np.shape(i_other)[0]))
                    y = np.transpose(y.reshape(3, np.shape(i_other)[0]))
                    xy = np.dstack((x, y))

                    # for dist_to_center in np.arange(vmin, vmax, vmax):
                    #     circle_tup = tuple(map(tuple, np.flipud(xyc * dist_to_center)))
                    #     # Make a clipper object
                    #     pc = pyclipper.Pyclipper()
                    #     # Add circles (ring-shape) to clipper as subject
                    #     pc.AddPaths(pyclipper.scale_to_clipper(circle_tup), pyclipper.PT_SUBJECT, True)

                    # Extra stuff needed for RotA
                    # if priocode == "RS6":
                    #     # Make another clipper object for RotA
                    #     pc_rota = pyclipper.Pyclipper()
                    #     pc_rota.AddPaths(pyclipper.scale_to_clipper(circle_tup), pyclipper.PT_SUBJECT, True)
                    #     # Bearing calculations from own view and other view
                    #     brg_own = np.mod((np.rad2deg(qdr[ind]) + fix_ang - hdg[i]) + 540., 360.) - 180.
                    #     brg_other = np.mod((np.rad2deg(qdr[ind]) + 180. - fix_ang - hdg[i_other]) + 540., 360.) - 180.

                    # add static objects
                    # nextnode_lat = ownship.ap.route[i].wplat[ownship.ap.route[i].iactwp]
                    # nextnode_lon = ownship.ap.route[i].wplon[ownship.ap.route[i].iactwp]
                    # if ownship.ap.route[i].iactwp == 0:
                    #     previousnode_lat = None
                    #     previousnode_lon = None
                    # else:
                    #     previousnode_lat = ownship.ap.route[i].wplat[ownship.ap.route[i].iactwp - 1]
                    #     previousnode_lon = ownship.ap.route[i].wplon[ownship.ap.route[i].iactwp - 1]
                    # obs_lat1, obs_lon1, obs_lat2, obs_lon2, latnode, lonnode = \
                    #     getclosestnodes(lat[i], lon[i], hdg[i], nextnode_lat, nextnode_lon, previousnode_lat,
                    #                     previousnode_lon)
                    # print(i, obs_lat1, obs_lon1, obs_lat2, obs_lon2)

                    # for border in range(len(obs_lat1)):
                    # geofence1 = np.array([obs_lat1[0], obs_lon1[0], obs_lat2[0], obs_lon2[0]])
                    # geofence2 = np.array([obs_lat1[1], obs_lon1[1], obs_lat2[1], obs_lon2[1]])
                    #
                    # # give longitude as x and latitude as y
                    # # x, y = self.getXYPosition(
                    # #     np.array([obs_lat1[0], obs_lat2[0], obs_lat1[1], obs_lat2[1], obs_lat1[0]]),
                    # #     np.array([obs_lon1[0], obs_lon2[0], obs_lon1[1], obs_lon2[1], obs_lon1[0]]),
                    # #     np.zeros(5), np.zeros(5))
                    # # area_valid = Polygon(list(zip(x, y)))
                    # area_valid = Polygon(
                    #     [(geofence1[1], geofence1[0]), (geofence1[3], geofence1[2]), (geofence2[1], geofence2[0]),
                    #      (geofence2[3], geofence2[2]), (geofence1[1], geofence1[0])])
                    # #ownship_point = Point(self.getXYPosition(lat[i], lon[i], 0, 0))
                    # ownship_point = Point(lon[i], lat[i])
                    #
                    # if i >= len(conf.geo_breach):
                    #     conf.geo_breach = np.append(conf.geo_breach,
                    #                                 np.zeros(i - len(conf.geo_breach_max_dist) + 1))
                    #     conf.geo_breach_max_dist = np.append(conf.geo_breach_max_dist,
                    #                                          np.zeros(
                    #                                              i - len(conf.geo_breach_max_dist) + 1))
                    #
                    #
                    #
                    # # consider also edges cases
                    # if not area_valid.contains(ownship_point) and not area_valid.touches(ownship_point):
                    #     # bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(0), geofence1)
                    #     # bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(1), geofence2)
                    #     if not conf.geo_breach[i]:
                    #         conf.geo_breach_total += 1
                    #         conf.geo_breach_exit = np.append(conf.geo_breach_exit, np.array([lat[i], lon[i]]))
                    #     p1, p2 = nearest_points(area_valid, ownship_point)
                    #     dist1 = geo.kwikdist(p1.y, p1.x, lat[i], lon[i])
                    #     dist2 = geo.kwikdist(p2.y, p2.x, lat[i], lon[i])
                    #     conf.geo_breach_max_dist[i] = max(conf.geo_breach_max_dist[i],max(dist1, dist2))
                    #     conf.geo_breach[i] = True
                    #
                    #     # area_valid = LineString([[geofence1[1], geofence1[0]], [geofence1[3], geofence1[2]]])
                    #     # p1, p2 = nearest_points(area_valid, ownship_point)
                    #     # dist = geo.kwikdist(p2, p1, lat[i], lon[i])
                    #     # dist1 = geo.kwikdist(p1.y, p1.x, lat[i], lon[i])*nm
                    #     # dist2 = geo.kwikdist(p2.y, p2.x, lat[i], lon[i])*nm
                    #     # area_valid = LineString([[geofence2[1], geofence2[0]], [geofence2[3], geofence2[2]]])
                    #     # p1, p2 = nearest_points(area_valid, ownship_point)
                    #     # dist1 = geo.kwikdist(p1.y, p1.x, lat[i], lon[i])*nm
                    #     # dist2 = geo.kwikdist(p2.y, p2.x, lat[i], lon[i])*nm
                    # else:
                    #     if not conf.geo_breach[i]:
                    #         conf.geo_breach_reentrance = np.append(conf.geo_breach_reentrance,
                    #                                                np.array([lat[i], lon[i]]))
                    #         conf.geo_breach_dist = np.append(conf.geo_breach_dist, conf.geo_breach_max_dist[i])
                    #     conf.geo_breach[i] = False

                    # bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(0), geofence1)
                    # bs.scr.objappend('LINE', str(i) + str(sim.simt) + str(1), geofence2)
                    # bs.scr.objappend('LINE', "connection" + str(sim.simt) + str(i),
                    #                  [latnode, lonnode, nextnode_lat, nextnode_lon])

                    # print('aircraft', str(i), geo.kwikqdrdist(lat[i], lon[i], nextnode_lat, nextnode_lat),
                    #       coordinates)

                    # fig2, axes2 = plt.subplots(1, 1)
                    # color = ['red', 'black', 'yellow', 'blue', 'orange', 'gray', 'pink', 'purple']
                    # axes2.set_prop_cycle(color=color)
                    # for it_obs in range(len(coordinates)):
                    #     if it_obs + 1 < len(coordinates):
                    #         axes2.plot( [coordinates[it_obs][1], coordinates[it_obs+1][1]],
                    #                      [coordinates[it_obs][0], coordinates[it_obs+1][0]])
                    #     else:
                    #         axes2.plot([coordinates[it_obs][1], coordinates[0][1]],
                    #                    [coordinates[it_obs][0], coordinates[0][0]])
                    # axes2.plot(lon[i], lat[i], marker='o', markersize=3, color="red")
                    # fig2.savefig(
                    #     r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\solution_space_images\\' + str(
                    #         ownship.id[i]) + '_' + str(sim.simt) + "_static.png")

                    # xs_gf = []  # [deg] in hdg CW
                    # ys_gf = []  # [m]
                    # # for k in range(len(coordinates)):
                    # #     # Calculate relative qdrs and distances of geofence points w.r.t. ownship
                    # #     qdr_gf, dist_gf = geo.qdrdist(ownship.lat[i], ownship.lon[i], coordinates[k][0],
                    # #                                   coordinates[k][1])
                    # #     qdrs_gf = np.append(qdrs_gf, qdr_gf)
                    # #     dists_gf = np.append(dists_gf, dist_gf * nm)
                    # geofences = []
                    # geofences.append(geofence1)
                    # geofences.append(geofence2)
                    #
                    # for geofence in geofences:
                    #     d_ownlat = np.ones(int(len(geofence) / 2)) * ownship.lat[i]
                    #     d_ownlon = np.ones(int(len(geofence) / 2)) * ownship.lon[i]
                    #     qdr_gf, dist_gf = geo.kwikqdrdist_matrix(d_ownlat, d_ownlon, geofence[0::2], geofence[1::2])
                    #     qdrs_gf_rad = np.deg2rad(qdr_gf)
                    #     dists_gf = dist_gf * nm
                    #     xs_gf.append(dists_gf * np.sin(qdrs_gf_rad))
                    #     ys_gf.append(dists_gf * np.cos(qdrs_gf_rad))
                    #
                    # #
                    # # # Generate data for each geofence segment 0 to 1, 1 to 2, 2 to 3 ..... n to 0.
                    # dxs_gf = np.array([])
                    # dys_gf = np.array([])
                    # # for k in range(len(coordinates)):
                    # #     x_from = xs_gf[k]
                    # #     y_from = ys_gf[k]
                    # #     # if last elament (needs to be connected to first element)
                    # #     if k == (len(coordinates) - 1):
                    # #         x_to = xs_gf[0]
                    # #         y_to = ys_gf[0]
                    # #     else:
                    # #         x_to = xs_gf[k + 1]
                    # #         y_to = ys_gf[k + 1]
                    # #
                    # #     dxs_gf = np.append(dxs_gf, x_to - x_from)
                    # #     dys_gf = np.append(dys_gf, y_to - y_from)
                    # #
                    # # dxs_gf = dxs_gf[::-1]
                    # # dys_gf = dys_gf[::-1]
                    # for geofence in range(len(xs_gf)):
                    #     dxs_gf = np.append(dxs_gf, xs_gf[geofence][1] - xs_gf[geofence][0])
                    #     dys_gf = np.append(dys_gf, ys_gf[geofence][1] - ys_gf[geofence][0])
                    #
                    # # calculate values (phis) of rotation of geofence segments(conf.tcpa_all[j])
                    # phis_gf = np.arctan2(dys_gf, dxs_gf) % (2 * np.pi)
                    # x_hats_prime = np.transpose(np.array([np.cos(phis_gf), np.sin(phis_gf)]))
                    # y_hats_prime = np.transpose(np.array([-np.sin(phis_gf), np.cos(phis_gf)]))

                    ind_colors = []
                    # Add each other other aircraft to clipper as clip
                    for j in range(np.shape(i_other)[0]):
                        # intruder_tlos = conf.tLOS_all[i][i_other[j]]

                        #if conf.inconf[i_other[j]]:
                        # ind_closest = np.argmin(conf.tcpa_all[i])
                        # j = np.where(i_other == ind_closest)[0][0]
                        ## Debug prints
                        ## print(traf.id[i] + " - " + traf.id[i_other[j]])
                        ## print(dist[ind[j]])
                        # Scale VO when not in LOS
                        vos_graph = []
                        static_graph = []
                        if dist[ind[j]] > hsepm:
                            # Normally VO shall be added of this other a/c
                            # VO = pyclipper.scale_to_clipper(tuple(map(tuple, xy[j, :, :])))
                            polygon_points = xy[j, :, :]
                            polygon_points = np.vstack((polygon_points, polygon_points[0]))
                            polygons_VOs_i.append(Polygon(polygon_points))
                            # vos_graph.append(xy[j, :, :])
                        elif dist[ind[j]] < hsepm:
                            # Pair is in LOS, instead of triangular VO, use darttip
                            # Check if bearing should be mirrored
                            if i_other[j] < i:
                                qdr_los = qdr[ind[j]] + np.pi
                            else:
                                qdr_los = qdr[ind[j]]
                            # Length of inner-leg of darttip
                            leg = 1.1 * vmax / np.cos(beta) * np.array([1, 1, 1, 0])
                            # Angles of darttip
                            angles_los = np.array([qdr_los + 2 * beta, qdr_los, qdr_los - 2 * beta, 0.])
                            # Calculate coordinates (CCW)
                            x_los = leg * np.sinse(angles_los)
                            y_los = leg * np.cos(angles_los)
                            # Put in array of correct format
                            xy_los = np.vstack((x_los, y_los)).T
                            xy_los = np.vstack((xy_los, xy_los[0]))
                            # Scale darttip
                            # VO = pyclipper.scale_to_clipper(tuple(map(tuple, xy_los)))
                            polygons_VOs_i.append(Polygon(xy_los))
                            # Add scaled VO to clipper
                            # pc.AddPath(VO, pyclipper.PT_CLIP, True)
                        #else:
                        # if dist[ind[j]] > hsepm:
                        #     # dist, turndist,turnrad, turnrate, turnangble, wpqdr
                        #     turns = np.empty((0, 6), float)
                        #     turns = self.calculateIntruderTurns(lat[i_other[j]], lon[i_other[j]],
                        #                                         route[int(i_other[j])],
                        #                                         bank[i_other[j]], gs_ap[i_other[j]], turns)
                        #
                        #     minimumTime = 0
                        #     maximumTime = 10
                        #
                        #     # if turns[0][0] < asas.vmax*maximumTime:
                        #     intruderLat, intruderLon, tc, last_alt = self.determineIntruderPosition(turns,
                        #                                                                             lat[i_other[j]],
                        #                                                                             lon[i_other[j]],
                        #                                                                             hdg[i_other[j]],
                        #                                                                             gs_ap[
                        #                                                                                 i_other[j]],
                        #                                                                             route[int(
                        #                                                                                 i_other[
                        #                                                                                     j])],
                        #                                                                             ax[i_other[j]],
                        #                                                                             minimumTime,
                        #                                                                             maximumTime)
                        #
                        #     add_VO = True
                        #     if last_alt > 0:
                        #         if abs(last_alt - ownship.alt[i]) > conf.hpz:
                        #             add_VO = False
                        #
                        #     if add_VO:
                        #         xy_intent = self.buildVO_intent(intruderLat, intruderLon, tc, hsepm, lat[i], lon[i],
                        #                                         vmax)  # , vos_graph)
                        #
                        #         for poly in xy_intent:
                        #             if poly.is_valid:
                        #                 polygons_VOs_intent_i.append(poly)

                        # if len(obs_lat1) > 0:
                        #
                        #     # for border in range(len(obs_lat1)):
                        #     #     print("border, ", str(border), obs_lat1[border], obs_lon1[border], obs_lat2[border],
                        #     #           obs_lon2[border])
                        #
                        #     # VOs = construct_geofence_VO(obs_lat1, obs_lon1, obs_lat2,obs_lon2,
                        #     #                            lat[i], lon[i], lat[i_other[j]], lon[i_other[j]],
                        #     #                            gseast[i_other[j]], gsnorth[i_other[j]], vmax)
                        #
                        #     # if len(VO) > 0 and len(VO[0]) > 0:
                        #     #     fig2, axes2 = plt.subplots(1, 1)
                        #     #     axes2.set_prop_cycle(
                        #     #         color=['red', 'black', 'yellow', 'blue', 'orange', 'gray', 'pink', 'purple'])
                        #     #     axes2.set_xlim([- 2 * vmax, 2 * vmax])
                        #     #     axes2.set_ylim([- 2 * vmax, 2 * vmax])
                        #     #     axes2.plot(np.sin(angles) * vmax, np.cos(angles) * vmax,
                        #     #                np.sin(angles) * vmin, np.cos(angles) * vmin, 'b')
                        #     #     axes2.plot(np.append(vos_graph[0][:, 0], vos_graph[0][:, 0][0]),
                        #     #                np.append(vos_graph[0][:, 1], vos_graph[0][:, 1][0]),
                        #     #                color='green')
                        #     #     axes2.fill(np.append(VO[0][:, 0], VO[0][:, 0][0]),
                        #     #                np.append(VO[0][:, 1], VO[0][:, 1][0]),
                        #     #                color=color[border])
                        #     #     fig2.savefig(
                        #     #         r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\solution_space_images\\' + str(
                        #     #             ownship.id[i]) + "_border_" + str(border) + ".png")
                        #
                        #     # VO = construct_geofence_VO(obs_lat2[border], obs_lon2[border], obs_lat1[border],
                        #     #                            obs_lon1[border],
                        #     #                            lat[i], lon[i], lat[i_other[j]], lon[i_other[j]],
                        #     #                            gseast[i_other[j]], gsnorth[i_other[j]], vmax)
                        #
                        #     # if len(VO) > 0 and len(VO[0]) > 0:
                        #     #     print('other way around')
                        #     #     fig2, axes2 = plt.subplots(1, 1)
                        #     #     axes2.set_prop_cycle(
                        #     #         color=['red', 'black', 'yellow', 'blue', 'orange', 'gray', 'pink', 'purple'])
                        #     #     axes2.set_xlim([- 2 * vmax, 2 * vmax])
                        #     #     axes2.set_ylim([- 2 * vmax, 2 * vmax])
                        #     #     axes2.plot(np.sin(angles) * vmax, np.cos(angles) * vmax,
                        #     #                np.sin(angles) * vmin, np.cos(angles) * vmin, 'b')
                        #     #     axes2.plot(np.append(vos_graph[0][:, 0], vos_graph[0][:, 0][0]),
                        #     #                np.append(vos_graph[0][:, 1], vos_graph[0][:, 1][0]),
                        #     #                color='green')
                        #     #     axes2.fill(np.append(VO[0][:, 0], VO[0][:, 0][0]),
                        #     #                np.append(VO[0][:, 1], VO[0][:, 1][0]),
                        #     #                color=color[border])
                        #     #     fig2.savefig(
                        #     #         r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\solution_space_images\\' + str(
                        #     #             ownship.id[i]) + "_border_" + str(border) + "2.png")
                        #
                        #     # Determine relative distance vector w.r.t. intruder
                        #     qdr_int, dist_int = geo.qdrdist(ownship.lat[i], ownship.lon[i], ownship.lat[i_other[j]],
                        #                                     ownship.lon[i_other[j]])
                        #     qdr_int_rad = np.deg2rad(qdr_int)
                        #     x_int = dist_int * nm * np.sin(qdr_int_rad)
                        #     y_int = dist_int * nm * np.cos(qdr_int_rad)
                        #     d_int = np.array([x_int, y_int])
                        #     trk_int = np.deg2rad(ownship.trk[i_other[j]])
                        #     gs_int = ownship.gs[i_other[j]]
                        #     v_int = np.array([gs_int * np.sin(trk_int), gs_int * np.cos(trk_int)])
                        #
                        #     v_int_dot_y_hats_prime = np.array([])
                        #     for k in range(len(y_hats_prime)):
                        #         v_int_dot_y_hats_prime = np.append(v_int_dot_y_hats_prime,
                        #                                            np.dot(v_int, y_hats_prime[k]))
                        #     candidate_gf_segments = np.where(v_int_dot_y_hats_prime < 0)[0]
                        #     # candidate_gf_segments = np.arange(len(geofences))
                        #
                        #     d_int_dot_x_hats_prime = np.array([])
                        #     d_int_dot_y_hats_prime = np.array([])
                        #
                        #     ds_geo = np.array([])  # [m] Array of distanced w.r.t. geofence of wonship
                        #     for k in candidate_gf_segments:
                        #         d_int_dot_x_hats_prime = np.append(d_int_dot_x_hats_prime,
                        #                                            np.dot(d_int, x_hats_prime[k]))
                        #         d_int_dot_y_hats_prime = np.append(d_int_dot_y_hats_prime,
                        #                                            np.dot(d_int, y_hats_prime[k]))
                        #
                        #         ds_geo = np.append(ds_geo, -np.dot(np.array([xs_gf[k], ys_gf[k]]), y_hats_prime[k]))
                        #
                        #     phis_prime_gf = 0.5 * np.arctan2(-1. * d_int_dot_x_hats_prime, d_int_dot_y_hats_prime)
                        #
                        #     # Total rotation angle
                        #     phis_total_gf = phis_gf[candidate_gf_segments] + phis_prime_gf
                        #
                        #     # Secondary axis system primary axes
                        #     x_hats_2prime = np.transpose(np.array([np.cos(phis_total_gf), np.sin(phis_total_gf)]))
                        #     y_hats_2prime = np.transpose(np.array([-np.sin(phis_total_gf), np.cos(phis_total_gf)]))
                        #
                        #     # Arrays of dot products
                        #     d_int_dot_x_hats_2prime = np.array([])
                        #     d_int_dot_y_hats_2prime = np.array([])
                        #     v_int_dot_x_hats_2prime = np.array([])
                        #     v_int_dot_y_hats_2prime = np.array([])
                        #     d_int_dot_v_int = np.array([])
                        #
                        #     for k in range(len(x_hats_2prime)):
                        #         d_int_dot_x_hats_2prime = np.append(d_int_dot_x_hats_2prime,
                        #                                             np.dot(d_int, x_hats_2prime[k]))
                        #         d_int_dot_y_hats_2prime = np.append(d_int_dot_y_hats_2prime,
                        #                                             np.dot(d_int, y_hats_2prime[k]))
                        #
                        #         v_int_dot_x_hats_2prime = np.append(v_int_dot_x_hats_2prime,
                        #                                             np.dot(v_int, x_hats_2prime[k]))
                        #         v_int_dot_y_hats_2prime = np.append(v_int_dot_y_hats_2prime,
                        #                                             np.dot(v_int, y_hats_2prime[k]))
                        #
                        #         d_int_dot_v_int = np.append(d_int_dot_v_int, np.dot(d_int, v_int))
                        #
                        #     # Constants needed to compute geometry of geofence VO's
                        #     C1s = 1. + np.sin(phis_prime_gf) * d_int_dot_x_hats_2prime / ds_geo[candidate_gf_segments]
                        #     C2s = 1. + np.cos(phis_prime_gf) * d_int_dot_y_hats_2prime / ds_geo[candidate_gf_segments]
                        #     C3s = -2. * v_int_dot_x_hats_2prime - np.sin(phis_prime_gf) * d_int_dot_v_int / ds_geo[
                        #         candidate_gf_segments]
                        #     C4s = -2. * v_int_dot_y_hats_2prime - np.cos(phis_prime_gf) * d_int_dot_v_int / ds_geo[
                        #         candidate_gf_segments]
                        #
                        #     # Center points of geofence VO geometries in double rotated axis system
                        #     Cxs_2prime = - C3s / (2. * C1s)
                        #     Cys_2prime = - C4s / (2. * C2s)
                        #
                        #     # semi major axes squared (in case of ellipse)
                        #     a2s = (- gs_int ** 2 + C2s * Cys_2prime ** 2) / C1s + Cxs_2prime ** 2
                        #     b2s = (- gs_int ** 2 + C1s * Cxs_2prime ** 2) / C2s + Cys_2prime ** 2
                        #
                        #     # Loop trough a2s and b2s to construct VOs, categorize them
                        #     numberVos = 0
                        #     for k in range(len(a2s)):
                        #         # if ownship outside gf segment
                        #         if (a2s[k] <= 0):
                        #             continue
                        #         # If Ellipse
                        #         if (b2s[k] > 0):
                        #             numberVos += 1
                        #             ellipse_angles = np.linspace(0., 2. * np.pi, N_angle)
                        #             rotated_xs = np.sqrt(a2s[k]) * np.cos(ellipse_angles) + Cxs_2prime[k]
                        #             rotated_ys = np.sqrt(b2s[k]) * np.sin(ellipse_angles) + Cys_2prime[k]
                        #         # If hyperbola
                        #         else:
                        #             numberVos += 1
                        #             tmax = np.log(
                        #                 (2. * vmax + np.sqrt(2. ** 2 * vmax ** 2 + a2s[k])) / np.sqrt(a2s[k]))
                        #             tmin = -tmax
                        #             if (phis_prime_gf[k] > 0):
                        #                 t = np.linspace(tmin, tmax, N_angle)
                        #                 rotated_xs = -np.sqrt(a2s[k]) * np.cosh(t) + Cxs_2prime[k]
                        #             else:
                        #                 t = np.linspace(tmax, tmin, N_angle)
                        #                 rotated_xs = np.sqrt(a2s[k]) * np.cosh(t) + Cxs_2prime[k]
                        #             rotated_ys = np.sqrt(-b2s[k]) * np.sinh(t) + Cys_2prime[k]
                        #         ind_colors.append(k)
                        #         non_rotated_xs = rotated_xs * np.cos(phis_total_gf[k]) - rotated_ys * np.sin(
                        #             phis_total_gf[k])
                        #         non_rotated_ys = rotated_xs * np.sin(phis_total_gf[k]) + rotated_ys * np.cos(
                        #             phis_total_gf[k])
                        #
                        #     if numberVos > 0:
                        #         # Add non roted VO's to clipper
                        #         xy_gf = []
                        #         xy_gf.append(non_rotated_xs)
                        #         xy_gf.append(non_rotated_ys)
                        #         xy_gf = np.array(xy_gf)
                        #         xy_gf = np.transpose(xy_gf)
                        #         polygons_geofences_i.append(Polygon(xy_gf))

                        # try:
                        #     pass
                        # VO_static = pyclipper.scale_to_clipper(xy_gf_tuple)
                        # pc.AddPath(VO_static, pyclipper.PT_CLIP, True)
                        # except:
                        #     print("An exception occurred with", VO)

                        # for VO in VOs:
                        #
                        #
                        # fig, axes = plt.subplots(1, 1)
                        # # asas.fig, asas.axes2 = plt.subplots()
                        # plt.grid(True)
                        # axes.set_xlim([- 2 * vmax, 2 * vmax])
                        # axes.set_ylim([- 2 * vmax, 2 * vmax])
                        # axes.plot(np.sin(angles) * vmax, np.cos(angles) * vmax,
                        #           np.sin(angles) * vmin, np.cos(angles) * vmin, 'b')
                        # for it99 in range(len(vos_graph_total)):
                        #     axes.plot(np.append(vos_graph_total[it99][:, 0], vos_graph_total[it99][:, 0][0]),
                        #               np.append(vos_graph_total[it99][:, 1], vos_graph_total[it99][:, 1][0]),
                        #               color='green')
                        # for it99 in range(len(static_graph)):
                        #     axes.fill(np.append(static_graph_total[it99][:, 0], static_graph_total[it99][:, 0][0]),
                        #               np.append(static_graph_total[it99][:, 1], static_graph_total[it99][:, 1][0]))
                        # fig.savefig(
                        #     r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\solution_space_images\\' + str(
                        #         ownship.id[i]) + '_intr ' + str(i_other[j]) +".png")

                        # For RotA it is possible to ignore
                        # if priocode == "RS6":
                        #     if brg_own[j] >= -20. and brg_own[j] <= 110.:
                        #         # Head-on or converging from right
                        #         pc_rota.AddPath(VO, pyclipper.PT_CLIP, True)
                        #     elif brg_other[j] <= -110. or brg_other[j] >= 110.:
                        #         # In overtaking position
                        #         pc_rota.AddPath(VO, pyclipper.PT_CLIP, True)
                        # # Detect conflicts for smaller layer in RS7 and RS8
                        # if priocode == "RS7" or priocode == "RS8":
                        #     if pyclipper.PointInPolygon(pyclipper.scale_to_clipper((gseast[i], gsnorth[i])), VO):
                        #         conf.inconf2[i] = True
                        # if priocode == "RS5":
                        #     if pyclipper.PointInPolygon(pyclipper.scale_to_clipper((apeast[i], apnorth[i])), VO):
                        #         conf.ap_free[i] = False

                        # break

                    polygons_VOs.append(polygons_VOs_i)
                    polygons_VOs_intent.append(polygons_VOs_intent_i)
                    polygons_geofences.append(polygons_geofences_i)
            else:
                polygons_VOs.append([])
                polygons_VOs_intent.append([])
                polygons_geofences.append([])

                if i < len(conf.geo_breach):
                    conf.geo_breach[i] = False

                # # Execute clipper command
                # FRV = pyclipper.scale_from_clipper(
                #     pc.Execute(pyclipper.CT_INTERSECTION, pyclipper.PFT_NONZERO, pyclipper.PFT_NONZERO))
                #
                # ARV = pc.Execute(pyclipper.CT_DIFFERENCE, pyclipper.PFT_NONZERO, pyclipper.PFT_NONZERO)
                #
                # if not priocode == "RS1" and not priocode == "RS5" and not priocode == "RS7" and not priocode == "RS8":
                #     # Make another clipper object for extra intersections
                #     pc2 = pyclipper.Pyclipper()
                #     # When using RotA clip with pc_rota
                #     if priocode == "RS6":
                #         # Calculate ARV for RotA
                #         ARV_rota = pc_rota.Execute(pyclipper.CT_DIFFERENCE, pyclipper.PFT_NONZERO,
                #                                    pyclipper.PFT_NONZERO)
                #         if len(ARV_rota) > 0:
                #             pc2.AddPaths(ARV_rota, pyclipper.PT_CLIP, True)
                #     else:
                #         # Put the ARV in thpere, make sure it's not empty
                #         if len(ARV) > 0:
                #             pc2.AddPaths(ARV, pyclipper.PT_CLIP, True)
                #
                # # Scale back
                # ARV = pyclipper.scale_from_clipper(ARV)
                #
                # # Check if ARV or FRV is empty
                # if len(ARV) == 0:
                #     # No aircraft in the vicinity
                #     # Map them into the format ARV wants. Outercircle CCW, innercircle CW
                #     ARV_loc[i] = []
                #     FRV_loc[i] = circle_lst
                #     ARV_calc_loc[i] = []
                #     # Calculate areas and store in asas
                #     FRV_area_loc[i] = np.pi * (vmax ** 2 - vmin ** 2)
                #     ARV_area_loc[i] = 0
                # elif len(FRV) == 0:
                #     # Should not happen with one a/c or no other a/c in the vicinity.
                #     # These are handled earlier. Happens when RotA has removed all
                #     # Map them into the format ARV wants. Outercircle CCW, innercircle CW
                #     ARV_loc[i] = circle_lst
                #     FRV_loc[i] = []
                #     ARV_calc_loc[i] = circle_lst
                #     # Calculate areas and store in asas
                #     FRV_area_loc[i] = 0
                #     ARV_area_loc[i] = np.pi * (vmax ** 2 - vmin ** 2)
                # else:
                #     # Check multi exteriors, if this layer is not a list, it means it has no exteriors
                #     # In that case, make it a list, such that its format is consistent with further code
                #     if not type(FRV[0][0]) == list:
                #         FRV = [FRV]
                #     if not type(ARV[0][0]) == list:
                #         ARV = [ARV]
                #     # Store in asas
                #     FRV_loc[i] = FRV
                #     ARV_loc[i] = ARV
                #     # Calculate areas and store in asas
                #     FRV_area_loc[i] = self.area(FRV)
                #     ARV_area_loc[i] = self.area(ARV)
                #
                #     # For resolution purposes sometimes extra intersections are wanted
                #     if priocode == "RS2" or priocode == "RS9" or priocode == "RS6" or priocode == "RS3" or priocode == "RS4":
                #         # Make a box that covers right or left of SSD
                #         own_hdg = hdg[i] * np.pi / 180
                #         # Efficient calculation of box, see notes
                #         if priocode == "RS2" or priocode == "RS6":
                #             # CW or right-turning
                #             sin_table = np.array([[1, 0], [-1, 0], [-1, -1], [1, -1]], dtype=np.float64)
                #             cos_table = np.array([[0, 1], [0, -1], [1, -1], [1, 1]], dtype=np.float64)
                #         elif priocode == "RS9":
                #             # CCW or left-turning
                #             sin_table = np.array([[1, 0], [1, 1], [-1, 1], [-1, 0]], dtype=np.float64)
                #             cos_table = np.array([[0, 1], [-1, 1], [-1, -1], [0, -1]], dtype=np.float64)
                #         # Overlay a part of the full SSD
                #         if priocode == "RS2" or priocode == "RS9" or priocode == "RS6":
                #             # Normalized coordinates of box
                #             xyp = np.sin(own_hdg) * sin_table + np.cos(own_hdg) * cos_table
                #             # Scale with vmax (and some factor) and put in tuple
                #             part = pyclipper.scale_to_clipper(tuple(map(tuple, 1.1 * vmax * xyp)))
                #             pc2.AddPath(part, pyclipper.PT_SUBJECT, True)
                #         elif priocode == "RS3":
                #             # Small ring
                #             xyp = (tuple(map(tuple, np.flipud(xyc * min(vmax, gs_ap[i] + 0.1)))),
                #                    tuple(map(tuple, xyc * max(vmin, gs_ap[i] - 0.1))))
                #             part = pyclipper.scale_to_clipper(xyp)
                #             pc2.AddPaths(part, pyclipper.PT_SUBJECT, True)
                #         elif priocode == "RS4":
                #             hdg_sel = hdg[i] * np.pi / 180
                #             xyp = np.array([[np.sin(hdg_sel - 0.0087), np.cos(hdg_sel - 0.0087)],
                #                             [0, 0],
                #                             [np.sin(hdg_sel + 0.0087), np.cos(hdg_sel + 0.0087)]],
                #                            dtype=np.float64)
                #             part = pyclipper.scale_to_clipper(tuple(map(tuple, 1.1 * vmax * xyp)))
                #             pc2.AddPath(part, pyclipper.PT_SUBJECT, True)
                #         # Execute clipper command
                #         ARV_calc = pyclipper.scale_from_clipper(
                #             pc2.Execute(pyclipper.CT_INTERSECTION, pyclipper.PFT_NONZERO, pyclipper.PFT_NONZERO))
                #         N += 1
                #         # If no smaller ARV is found, take the full ARV
                #         if len(ARV_calc) == 0:
                #             ARV_calc = ARV
                #         # Check multi exteriors, if this layer is not a list, it means it has no exteriors
                #         # In that case, make it a list, such that its format is consistent with further code
                #         if not type(ARV_calc[0][0]) == list:
                #             ARV_calc = [ARV_calc]
                #     # Shortest way out prio, so use full SSD (ARV_calc = ARV)
                #     else:
                #         ARV_calc = ARV
                #     # Update calculatable ARV for resolutions
                #     ARV_calc_loc[i] = ARV_calc

        # If sequential approach, the local should go elsewhere
        if not priocode == "RS7" and not priocode == "RS8":
            conf.FRV = FRV_loc
            conf.ARV = ARV_loc
            conf.ARV_calc = ARV_calc_loc
            conf.FRV_area = FRV_area_loc
            conf.ARV_area = ARV_area_loc
        else:
            conf.ARV_calc2 = ARV_calc_loc
        return polygons_VOs, polygons_VOs_intent, polygons_geofences

    def calculate_resolution(self, conf, ownship, polygons_vos, polygons_vos_intent, polygons_geofences, active,
                             only_active):
        ntraf = ownship.ntraf

        all_solution_found = []

        # Loop through SSDs of all aircraft
        for i in range(ntraf):
            lat_next, lon_next = ownship.ap.route[i].wplat[ownship.ap.route[i].iactwp], \
                                 ownship.ap.route[i].wplon[ownship.ap.route[i].iactwp]
            qdr_next, dummy = geo.kwikqdrdist(ownship.lat[i], ownship.lon[i], lat_next, lon_next)
            qdr_next_rad = np.deg2rad(qdr_next)
            vmax = ownship.ap.route[i].wpspd[ownship.ap.route[i].iactwp]
            gsnorth = np.cos(qdr_next_rad) * vmax
            gseast = np.sin(qdr_next_rad) * vmax

            solution_found = False
            if (not only_active and (conf.inconf[i] or active[i])) or active[i]:
                # first see if the ap intention is free of conflict
                all_VOs = unary_union(polygons_vos[i])# + polygons_vos_intent[i])
                # all_VOsGeofences = unary_union(   polygons_vos[i]  )
                if not Point(gseast, gsnorth).within(all_VOs):
                    conf.asase[i] = gseast
                    conf.asasn[i] = gsnorth
                    all_solution_found.append(True)
                # then see if other velocities with the same heading are free of conflict
                else:
                    intent_v_line = LineString([[0, 0], [np.sin(qdr_next_rad) * vmax, np.cos(qdr_next_rad) * vmax]])
                    # differences = intent_v_line.difference(all_VOs)
                    try:
                        differences = intent_v_line.difference(all_VOs)
                    except:
                        # print("ERROR on intersection:")
                        differences = intent_v_line.difference(unary_union(polygons_vos[i]))

                    if differences.is_empty:
                        # print("intersection is empty")
                        differences = intent_v_line.difference(unary_union(polygons_vos[i]))

                    if not differences.is_empty:
                        if type(differences) is LineString:
                            differences = [differences]
                        max_speed2 = 0
                        # when the lines encounters multi polygons it gets divided
                        for line_it in range(len(differences)):
                            for it in range(len(differences[line_it].xy)):
                                aux = differences[line_it].xy[0][it] ** 2 + differences[line_it].xy[1][it] ** 2
                                if aux > max_speed2:
                                    max_speed2 = aux
                                    conf.asase[i] = differences[line_it].xy[0][it]
                                    conf.asasn[i] = differences[line_it].xy[1][it]
                        # solution closest to vmax is the last difference point
                        if max_speed2 > 0:
                            solution_found = True

                            # fig, axes = plt.subplots(1, 1)
                            # # asas.fig, asas.axes2 = plt.subplots()
                            # plt.grid(True)
                            # axes.set_xlim([- 2*vmax, 2*vmax])
                            # axes.set_ylim([- 2*vmax, 2*vmax])
                            # angles = np.arange(0, 2 * np.pi, 2 * np.pi / 180)
                            # axes.plot(np.sin(angles) * vm.cr.trk[1], bs.traf.aax, np.cos(angles) * vmax,
                            #           np.sin(angles) * 0, np.cos(angles) * 0, 'b')
                            # for it_plot in range(len(polygons_vos[i])):
                            #     axes.fill(polygons_vos[i][it_plot].exterior.xy[0],
                            #               polygons_vos[i][it_plot].exterior.xy[1])
                            # axes.plot([0, np.sin(qdr_next_rad) * vmax], [0, np.cos(qdr_next_rad) * vmax], color="red")
                            # axes.plot(gseast, gsnorth, marker='o', markersize=10, color="red")
                            # axes.plot(conf.asase[i], conf.asasn[i], marker='o', markersize=3, color="black")
                            # fig.savefig(
                            #     r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\images\intent_' + str(
                            #         ownship.id[i]) + '_' + str(sim.simt) + ".png")

                    if not solution_found:
                        conf.asase[i] = gseast
                        conf.asasn[i] = gsnorth
                    all_solution_found.append(solution_found)

            # Those that are not in conflict will be assigned zeros
            else:
                conf.asase[i] = 0.
                conf.asasn[i] = 0.
                all_solution_found.append(solution_found)

        return all_solution_found

    # def calculate_resolution(self, conf, ownship):
    #     """ Calculates closest conflict-free point according to ruleset """
    #     # It's just linalg, however credits to: http://stackoverflow.com/a/1501725
    #     # Variables
    #     ARV = conf.ARV_calc
    #     if self.priocode == "RS7" or self.priocode == "RS8":
    #         ARV2 = conf.ARV_calc2
    #     # Select AP-setting as reference point for closest to target rulesets
    #     if self.priocode == "RS5" or self.priocode == "RS8":
    #         gsnorth = np.cos(ownship.ap.trk / 180 * np.pi) * ownship.ap.tas
    #         gseast = np.sin(ownship.ap.trk / 180 * np.pi) * ownship.ap.tas
    #     else:
    #         gsnorth = ownship.gsnorth
    #         gseast = ownship.gseast
    #     ntraf = ownship.ntraf
    #
    #     # Loop through SSDs of all aircraft
    #     for i in range(ntraf):
    #         # Only those that are in conflict need to resolve
    #         if conf.inconf[i] and ARV[i] is not None and len(ARV[i]) > 0:
    #             # First check if AP-setting is free
    #             if conf.ap_free[i] and self.priocode == "RS5":
    #                 conf.asase[i] = gseast[i]
    #                 conf.asasn[i] = gsnorth[i]
    #             else:
    #                 # Loop through all exteriors and append. Afterwards concatenate
    #                 p = []
    #                 q = []
    #
    #                 fig, axes = plt.subplots(1, 1)
    #                 # asas.fig, asas.axes2 = plt.subplots()
    #                 plt.grid(True)
    #
    #                 for j in range(len(ARV[i])):
    #                     axes.plot(np.array(ARV[i][j])[:,0], np.array(ARV[i][j])[:,1])
    #
    #                     p.append(np.array(ARV[i][j]))
    #                     q.append(np.diff(np.row_stack((p[j], p[j][0])), axis=0))
    #
    #                 fig.savefig(
    #                         r'C:\Users\mjribeiro\Documents\my documents\papers\urban+intent\solution_space_images\\total_' + str(
    #                             ownship.id[i]) + '_' + str(sim.simt) + "hi.png")
    #
    #                 p = np.concatenate(p)
    #                 q = np.concatenate(q)
    #                 # Calculate squared distance between edges
    #                 l2 = np.sum(q ** 2, axis=1)
    #                 # Catch l2 == 0 (exception)
    #                 same = l2 < 1e-8
    #                 l2[same] = 1.
    #                 # Calc t
    #                 t = np.sum((np.array([gseast[i], gsnorth[i]]) - p) * q, axis=1) / l2
    #                 # Speed of boolean indices only slightly faster (negligible)
    #                 # t must be limited between 0 and 1
    #                 t = np.clip(t, 0., 1.)
    #                 t[same] = 0.
    #                 # Calculate closest point to each edge
    #                 x1 = p[:, 0] + t * q[:, 0]
    #                 y1 = p[:, 1] + t * q[:, 1]
    #                 # Get distance squared
    #                 d2 = (x1 - gseast[i]) ** 2 + (y1 - gsnorth[i]) ** 2
    #                 # Sort distance
    #                 ind = np.argsort(d2)
    #                 x1 = x1[ind]
    #                 y1 = y1[ind]
    #
    #                 if self.priocode == "RS7" or self.priocode == "RS8" and conf.inconf2[i]:
    #                     # Loop through all exteriors and append. Afterwards concatenate
    #                     p = []
    #                     q = []
    #                     for j in range(len(ARV2[i])):
    #                         p.append(np.array(ARV2[i][j]))
    #                         q.append(np.diff(np.row_stack((p[j], p[j][0])), axis=0))
    #                     p = np.concatenate(p)
    #                     q = np.concatenate(q)
    #                     # Calculate squared distance between edges
    #                     l2 = np.sum(q ** 2, axis=1)
    #                     # Catch l2 == 0 (exception)
    #                     same = l2 < 1e-8
    #                     l2[same] = 1.
    #                     # Calc t
    #                     t = np.sum((np.array([gseast[i], gsnorth[i]]) - p) * q, axis=1) / l2
    #                     # Speed of boolean indices only slightly faster (negligible)
    #                     # t must be limited between 0 and 1
    #                     t = np.clip(t, 0., 1.)
    #                     t[same] = 0.
    #                     # Calculate closest point to each edge
    #                     x2 = p[:, 0] + t * q[:, 0]
    #                     y2 = p[:, 1] + t * q[:, 1]
    #                     # Get distance squared
    #                     d2 = (x2 - gseast[i]) ** 2 + (y2 - gsnorth[i]) ** 2
    #                     # Sort distance
    #                     ind = np.argsort(d2)
    #                     x2 = x2[ind]
    #                     y2 = y2[ind]
    #                     d2 = d2[ind]
    #
    #                 # Store result in conf
    #                 if not conf.inconf2[i] or not self.priocode == "RS7" and not self.priocode == "RS8":
    #                     conf.asase[i] = x1[0]
    #                     conf.asasn[i] = y1[0]
    #                 else:
    #                     # Sequential method, check if both result in very similar resolutions
    #                     if (x1[0] - x2[0]) * (x1[0] - x2[0]) + (x1[0] - x2[0]) * (x1[0] - x2[0]) < 1:
    #                         # In that case take the full layer
    #                         conf.asase[i] = x1[0]
    #                         conf.asasn[i] = y1[0]
    #                     else:
    #                         # In that case take the partial layer solution and see which
    #                         # results in lower TLOS
    #                         conf.asase[i] = x1[0]
    #                         conf.asasn[i] = y1[0]
    #                         # dv2 for the RS1-solution
    #                         dist12 = (x1[0] - gseast[i]) ** 2 + (y1[0] - gsnorth[i]) ** 2
    #                         # distances for the partial layer solution stored in d2
    #                         ind = d2 < dist12
    #                         if sum(ind) == 1:
    #                             conf.asase[i] = x2[0]
    #                             conf.asasn[i] = y2[0]
    #                         elif sum(ind) > 1:
    #                             x2 = x2[ind]
    #                             y2 = y2[ind]
    #                             # Get solution with minimum TLOS
    #                             i_other = conf.inrange[i]
    #                             idx = self.minTLOS(conf, ownship, i, i_other, x1, y1, x2, y2)
    #                             # Get solution with maximum TLOS
    #                             conf.asase[i] = x2[idx]
    #                             conf.asasn[i] = y2[idx]
    #                         else:
    #                             # This case should not happen...
    #                             conf.asase[i] = x1[0]
    #                             conf.asasn[i] = y1[0]
    #
    #         # Those that are not in conflict will be assigned zeros
    #         # Or those that have no solutions (full ARV)
    #         else:
    #             conf.asase[i] = 0.
    #             conf.asasn[i] = 0.

    def area(self, vset):
        """ This function calculates the area of the set of FRV or ARV """
        # Initialize A as it could be calculated iteratively
        A = 0
        # Check multiple exteriors
        if type(vset[0][0]) == list:
            # Calc every exterior separately
            for i in range(len(vset)):
                A += pyclipper.scale_from_clipper(
                    pyclipper.scale_from_clipper(pyclipper.Area(pyclipper.scale_to_clipper(vset[i]))))
        else:
            # Single exterior
            A = pyclipper.scale_from_clipper(
                pyclipper.scale_from_clipper(pyclipper.Area(pyclipper.scale_to_clipper(vset))))
        return A

    def qdrdist_matrix_indices(self, ntraf):
        """ This function gives the indices that can be used in the lon/lat-vectors """
        # The indices will be n*(n-1)/2 long
        # Only works for n >= 2, which is logical...
        # This is faster than np.triu_indices :)
        tmp_range = np.arange(ntraf - 1, dtype=np.int32)
        ind1 = np.repeat(tmp_range, (tmp_range + 1)[::-1])
        ind2 = np.ones(ind1.shape[0], dtype=np.int32)
        inds = np.cumsum(tmp_range[1:][::-1] + 1)
        np.put(ind2, inds, np.arange(ntraf * -1 + 3, 1))
        ind2 = np.cumsum(ind2, out=ind2)
        return ind1, ind2

    # def minTLOS(self, conf, ownship, i, i_otaher, x1, y1, x, y):
    #     """ This function calculates the aggregated TLOS for all resolution points """
    #     # Get speeds of other AC in range
    #     x_other = ownship.gseast[i_other]
    #     y_other = ownship.gsnorth[i_other]
    #     # Get relative bearing [deg] and distance [nm]
    #     qdr, dist = geo.qdrdist(ownship.lat[i], ownship.lon[i], ownship.lat[i_other], ownship.lon[i_other])
    #     # Convert to SI
    #     qdr = np.deg2rad(qdr)
    #     dist *= nm
    #     # For vectorization, store lengths as W and L
    #     W = np.shape(x)[0]
    #     L = np.shape(x_other)[0]
    #     # Relative speed-components
    #     du = np.dot(x_other.reshape((L, 1)), np.ones((1, W))) - np.dot(np.ones((L, 1)), x.reshape((1, W)))
    #     dv = np.dot(y_other.reshape((L, 1)), np.ones((1, W))) - np.dot(np.ones((L, 1)), y.reshape((1, W)))
    #     # Relative speed + zero check
    #     vrel2 = du * du + dv * dv
    #     vrel2 = np.where(np.abs(vrel2) < 1e-6, 1e-6, vrel2)  # limit lower absolute value
    #     # X and Y distance
    #     dx = np.dot(np.reshape(dist * np.sin(qdr), (L, 1)), np.ones((1, W)))
    #     dy = np.dot(np.reshape(dist * np.cos(qdr), (L, 1)), np.ones((1, W)))
    #     # Time to CPA
    #     tcpa = -(du * dx + dv * dy) / vrel2
    #     # CPA distance
    #     dcpa2 = np.square(np.dot(dist.reshape((L, 1)), np.ones((1, W)))) - np.square(tcpa) * vrel2
    #     # Calculate time to LOS
    #     R2 = conf.rpz * conf.rpz
    #     swhorconf = dcpa2 < R2
    #     dxinhor = np.sqrt(np.maximum(0, R2 - dcpa2))
    #     dtinhor = dxinhor / np.sqrt(vrel2)
    #     tinhor = np.where(swhorconf, tcpa - dtinhor, 0.)
    #     tinhor = np.where(tinhor > 0, tinhor, 1e6)
    #     # Get index of best solution
    #     idx = np.argmax(np.sum(tinhor, 0))
    #
    #     return idx
