""" BlueSky deletion area plugin. This plugin can use an area definition to
    delete aircraft that exit the area. Statistics on these flights can be
    logged with the FLSTLOG logger. """
import numpy as np
from bluesky.tools.geo import kwikqdrdist_matrix
# Import the global bluesky objects. Uncomment the ones you need
from bluesky import traf, sim, stack  # , settings, navdb, traf, sim, scr, tools
from bluesky.tools import datalog, areafilter, \
    TrafficArrays, RegisterElementParameters, plugin
from bluesky.tools.aero import ft, kts, nm, fpm
from bluesky.tools.simtime import timed_function
import sys


# Log parameters for the flight statistics log
flstheader = \
    '#######################################################\n' + \
    'FLST LOG\n' + \
    'Flight Statistics\n' + \
    '#######################################################\n\n' + \
    'Parameters [Units]:\n' + \
    'Deletion Time [s], ' + \
    'Call sign [-], ' + \
    'Spawn Time [s], ' + \
    'Flight time [s], ' + \
    'Time in conflict [s], ' + \
    'Actual Distance 2D [nm], ' + \
    'Actual Distance 3D [nm], ' + \
    'Work Done [MJ], ' + \
    'Latitude [deg], ' + \
    'Longitude [deg], ' + \
    'Altitude [ft], ' + \
    'TAS [kts], ' + \
    'Vertical Speed [fpm], ' + \
    'Heading [deg], ' + \
    'Origin Lat [deg], ' + \
    'Origin Lon [deg], ' + \
    'Destination Lat [deg], ' + \
    'Destination Lon [deg], ' + \
    'ASAS Active [bool], ' + \
    'Pilot ALT [ft], ' + \
    'Pilot SPD (TAS) [kts], ' + \
    'Pilot HDG [deg], ' + \
    'Pilot VS [fpm]' + '\n'

confheader = \
    '#######################################################\n' + \
    'CONF LOG\n' + \
    'Conflict Statistics\n' + \
    '#######################################################\n\n' + \
    'Parameters [Units]:\n' + \
    'Simulation time [s], ' + \
    'Total number of aircraft in exp area [-], ' + \
    'Total number of conflicts in exp area [-], ' + \
    'Total number of losses of separation in exp area [-]\n'

structure_header = \
    '#######################################################\n' + \
    'STRUCTURE CHANGES LOG\n' + \
    '#######################################################\n\n' + \
    'Parameters [Units]:\n' + \
    'Simulation time [s], ' + \
    'New Structure Set by RL [-], ' + \
    'Resulting Structure Change [-] '


CRSOL_header = \
    '#######################################################\n' + \
    'Conflict Resolution LOG\n' + \
    '#######################################################\n\n' 
    

    # Global data
area = None


### Initialization function of your plugin. Do not change the name of this
### function, as it is the way BlueSky recognises this file as a plugin.
def init_plugin():
    # Addtional initilisation code
    global area
    area = Area()

    # Configuration parameters
    config = {
        # The name of your plugin
        'plugin_name': 'AREA',

        # The type of this plugin. For now, only simulation plugins are possible.
        'plugin_type': 'sim',

        # The update function is called after traffic is updated.
        'update': area.update,

        # The reset function
        'reset': area.reset
    }

    stackfunctions = {
        'AREA': [
            'AREA Shapename/OFF or AREA lat,lon,lat,lon,[top,bottom]',
            '[float/txt,float,float,float,alt,alt]',
            area.set_area,
            'Define deletion area (aircraft leaving area are deleted)'
        ],
        'EXP': [
            'EXP Shapename/OFF or EXP lat,lon,lat,lon,[top,bottom]',
            '[float/txt,float,float,float,alt,alt]',
            lambda *args: area.set_area(*args, exparea=True),
            'Define experiment area (area of interest)'
        ],
        'TAXI': [
            'TAXI ON/OFF [alt] : OFF auto deletes traffic below 1500 ft',
            'onoff[,alt]',
            area.set_taxi,
            'Switch on/off ground/low altitude mode, prevents auto-delete at 1500 ft'
        ]
    }
    # init_plugin() should always return these two dicts.
    return config, stackfunctions


class Area(TrafficArrays):
    ''' Traffic area: delete traffic when it leaves this area (so not when outside)'''

    def __init__(self):
        super(Area, self).__init__()
        # Parameters of area
        self.active = False
        self.delarea = ''
        self.exparea = ''
        self.swtaxi = True  # Default ON: Doesn't do anything. See comments of set_taxi function below.
        self.swtaxialt = 1500.0  # Default alt for TAXI OFF
        self.prevconfpairs = set()
        self.intrusions_all = np.array([])
        self.prevlospairs = set()
        self.confinside_all = 0
        self.prevgeofencebreach = 0

        # The FLST logger
        self.flst = datalog.crelog('FLSTLOG', None, flstheader)
        self.conflog = datalog.crelog('CONFLOG', None, confheader)
        #self.flpath = datalog.crelog('PATHLOG', None, flstheader)
        self.vchanges = datalog.crelog('VCHANGESLOG', None, flstheader)
        #self.stlog = datalog.crelog('STRUCTURELOG', None, structure_header)
        #self.CRSolution = datalog.crelog('CRSOL', None, CRSOL_header)

        with RegisterElementParameters(self):
            self.insdel = np.array([], dtype=np.bool)  # In deletion area or not
            self.insexp = np.array([], dtype=np.bool)  # In experiment area or not
            self.oldalt = np.array([])
            self.distance2D = np.array([])
            self.distance3D = np.array([])
            self.dstart2D = np.array([])
            self.average_speed = np.array([])
            self.dstart3D = np.array([])
            self.workstart = np.array([])
            self.work = np.array([])
            self.entrytime = np.array([])
            self.create_time = np.array([])
            self.timeinconf = np.array([])
            self.flightaltitude= []
            self.flightpath = []
            self.flight_inConflict  = []
            self.flight_vs = []

    def reset(self):
        ''' Reset area state when simulation is reset. '''
        super().reset()
        self.active = False
        self.delarea = ''
        self.exparea = ''
        self.swtaxi = True
        self.swtaxialt = 1500.0
        self.confinside_all = 0
        self.losinside_all = 0

    def create(self, n=1):
        super(Area, self).create(n)
        self.oldalt[-n:] = traf.alt[-n:]
        self.insdel[-n:] = False
        self.insexp[-n:] = False
        self.create_time[-n:] = sim.simt

    @timed_function('AREA', dt=1.0)
    def update(self, dt):
        ''' Update flight efficiency metrics
            2D and 3D distance [m], and work done (force*distance) [J] '''
        if self.active:
            resultantspd = np.sqrt(traf.gs * traf.gs + traf.vs * traf.vs)
            self.distance2D += dt * traf.gs
            self.distance3D += dt * resultantspd
            self.work += (traf.perf.thrust * dt * resultantspd)
            self.average_speed += traf.gs
            for aircraft in range(traf.ntraf):
                self.flightpath[aircraft] = np.append(self.flightpath[aircraft], (traf.lat[aircraft],traf.lon[aircraft]))
                self.flightaltitude[aircraft] = np.append(self.flightaltitude[aircraft],traf.alt[aircraft])
                self.flight_inConflict[aircraft] = np.append(self.flight_inConflict[aircraft], traf.cd.inconf[aircraft])
                self.flight_vs[aircraft] = np.append(self.flight_vs[aircraft], traf.cr.vs[aircraft])

            # Find out which aircraft are currently inside the experiment area, and
            # determine which aircraft need to be deleted.
            insdel = areafilter.checkInside(self.delarea, traf.lat, traf.lon, traf.alt)
            insexp = insdel if not self.exparea else \
                areafilter.checkInside(self.exparea, traf.lat, traf.lon, traf.alt)
            # Find all aircraft that were inside in the previous timestep, but no
            # longer are in the current timestep
            delidx = np.where(np.array(self.insdel) * (np.array(insdel) == False))[0]
            self.insdel = insdel

            # Count new conflicts where at least one of the aircraft is inside
            # the experiment area
            # Store statistics for all new conflict pairs
            # Conflict pairs detected in the current timestep that were not yet present in the previous timestep
            # confpairs_new = list(set(traf.cd.confpairs) - self.prevconfpairs)
            #
            # lospairs_new = list(set(traf.cd.lospairs) - self.prevlospairs)
            #
            #
            # if (confpairs_new or lospairs_new) and sim.simt % 1800 ==0:
            #     # If necessary: select conflict geometry parameters for new conflicts
            #     # idxdict = dict((v, i) for i, v in enumerate(traf.cd.confpairs))
            #     # idxnew = [idxdict.get(i) for i in confpairs_new]
            #     # dcpa_new = np.asarray(traf.cd.dcpa)[idxnewmer]
            #     # tcpa_new = np.asarray(traf.cd.tcpa)[idxnew]
            #     # tLOS_new = np.asarray(traf.cd.tLOS)[idxnew]
            #     # qdr_new = np.asarray(traf.cd.qdr)[idxnew]
            #     # dist_new = np.asarray(traf.cd.dist)[idxnew]
            #
            #     if confpairs_new:
            #         newconf_unique = {frozenset(pair) for pair in confpairs_new}
            #         ac1, ac2 = zip(*newconf_unique)
            #         idx1 = traf.id2idx(ac1)
            #         idx2 = traf.id2idx(ac2)
            #         # do not count conflicts with aircraft that were just spawned
            #         #allidx = np.asarray(idx1 + idx2)
            #         # dummy, dist = kwikqdrdist_matrix(traf.lat[allidx], traf.lon[allidx], \
            #         #                                  np.asarray([traf.ap.route[x].wplat[0] for x in allidx]),
            #         #                                  np.asarray([traf.ap.route[x].wplon[0] for x in allidx]))
            #         # newconf_inside = np.logical_or(
            #         #     np.logical_and(insexp[idx1], dist[:len(idx1)] * nm > traf.cd.rpz),
            #         #     np.logical_and(insexp[idx2], dist[len(idx1):] * nm > traf.cd.rpz))
            #         newconf_inside = np.logical_or(insexp[idx1], insexp[idx2])
            #
            #         nnewconf_exp = np.count_nonzero(newconf_inside)
            #         if nnewconf_exp:
            #             self.confinside_all += nnewconf_exp
            #
            #     if lospairs_new:
            #         newlos_unique = {frozenset(pair) for pair in lospairs_new}
            #         ac1, ac2 = zip(*newlos_unique)
            #         idx1 = traf.id2idx(ac1)
            #         idx2 = traf.id2idx(ac2)
            #         # # do not count losses of separation with aircraft that were just spawned
            #         #allidx = np.asarray(idx1 + idx2)
            #         # dummy, dist = kwikqdrdist_matrix(traf.lat[allidx], traf.lon[allidx], \
            #         #                                  np.asarray([traf.ap.route[x].wplat[0] for x in allidx]),
            #         #                                  np.asarray([traf.ap.route[x].wplon[0] for x in allidx]))
            #         # newlos_inside = np.logical_or(
            #         #     np.logical_and(insexp[idx1], dist[:len(idx1)] * nm > traf.cd.rpz),
            #         #     np.logical_and(insexp[idx2], dist[len(idx1):] * nm > traf.cd.rpz))
            #         newlos_inside = np.logical_or(insexp[idx1], insexp[idx2])
            #
            #         nnewlos_exp = np.count_nonzero(newlos_inside)
            #         if newlos_unique:
            #             self.losinside_all += nnewlos_exp
            #             indexes = np.where(newlos_inside == True)[0]
            #             new_array = [(traf.cd.rpz - traf.cd.dcpa[index]) / nm for index in indexes]
            #             self.intrusions_all = np.append(self.intrusions_all, new_array)
            np.set_printoptions(threshold=np.inf)
            if sim.simt % 300 == 0:
                self.conflog.log(traf.ntraf,
                                 len(traf.cd.confpairs_all),
                                 len(traf.cd.lospairs_all),
                                 #len(traf.cd.confpairs_all_cruising),
                                 #len(traf.cd.lospairs_all_cruising),
                                 #np.array2string(traf.alt, precision=8, separator=' ', max_line_width=np.inf),
                                 #np.array2string(traf.gs, precision=8, separator=' ', max_line_width=np.inf),
                                 np.array2string(traf.cd.intrusions, precision=8, separator=' ', max_line_width=np.inf),
                                 #  np.array2string(traf.cd.intrusionstime, precision=8, separator=' ', max_line_width=np.inf),
                                 #  np.array2string(traf.cd.intrusionsalt, precision=8, separator=' ',max_line_width=np.inf),
                                 np.array2string(traf.cd.intrusionsspeed, precision=8, separator=' ', max_line_width=np.inf),
                                 #  np.array2string(traf.cd.intrusionslat, precision=8, separator=' ',max_line_width=np.inf),
                                 #  np.array2string(traf.cd.intrusionslon, precision=8, separator=' ', max_line_width=np.inf)
                                 )
                #
                # self.prevconfpairs = set(traf.cd.confpairs)
                # self.prevlospairs = set(traf.cd.lospairs)
            # if sim.simt % 1 == 0:
            #     self.CRSolution.log(np.array2string(traf.cr.CR_RL.print_ac, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_state, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_current_heading, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_current_speed, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_heading_change, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_speed_change, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_MVP_heading, precision=8, separator=' ', max_line_width=np.inf),
            #         np.array2string(traf.cr.CR_RL.print_MVP_speed, precision=8, separator=' ', max_line_width=np.inf)
            #     )

            # Register distance values upon entry of experiment area
            newentries = np.logical_not(self.insexp) * insexp
            self.dstart2D[newentries] = self.distance2D[newentries]
            self.dstart3D[newentries] = self.distance3D[newentries]
            self.workstart[newentries] = self.work[newentries]
            self.entrytime[newentries] = sim.simt

            if traf.cd.inconf.any():
                in_conf = np.where(traf.cd.inconf == True)
                self.timeinconf[in_conf and insexp] += dt

            # Log flight statistics when exiting experiment area
            exits = np.logical_and(self.insexp, np.logical_not(insexp))
            # Update insexp
            self.insexp = insexp

            if np.any(exits):
                for exit_id in np.where(exits)[0]:
                    self.flst.log(
                    np.array(traf.id)[exits],
                    self.create_time[exits],
                    sim.simt - self.entrytime[exits],
                    self.timeinconf[delidx],
                    (self.distance2D[exits] - self.dstart2D[exits]) / nm,
                    (self.distance3D[exits] - self.dstart3D[exits]) / nm,
                    (self.work[exits] - self.workstart[exits]) * 1e-6,
                    traf.lat[exits],
                    traf.lon[exits],
                    traf.alt[exits] / ft,
                    traf.tas[exits] / kts,
                    traf.vs[exits] / fpm,
                    traf.hdg[exits],
                    self.average_speed[exits]/(sim.simt - self.create_time[exits]),
                    traf.cr.active[exits],
                    traf.pilot.alt[exits] / ft,
                    traf.pilot.tas[exits] / kts,
                    traf.pilot.vs[exits] / fpm,
                    traf.pilot.hdg[exits])

            # if np.any(exits):
            #     for aircraft_exit in range(traf.ntraf):
            #         if exits[aircraft_exit]:
            #             self.flpath.log(
            #                 np.array(traf.id)[aircraft_exit],
            #                 np.array2string(self.flightpath[aircraft_exit], precision=8, separator=' ', max_line_width=np.inf),
            #                 np.array2string(self.flightaltitude[aircraft_exit], precision=8, separator=' ', max_line_width=np.inf),
            #                 np.array2string(self.flight_inConflict[aircraft_exit], precision=8, separator=' ',max_line_width=np.inf),
            #                 np.array2string(self.flight_vs[aircraft_exit], precision=8, separator=' ',max_line_width=np.inf)
            #             )



            # if sim.simt % 600 == 0:
            #     self.stlog.log(
            #         np.array2string(np.array(plugin.active_plugins['STRUCTURE_RL'].structure_RL.history_structures), precision=8, separator=' ', max_line_width=np.inf)
            #     )


            # delete all aicraft in self.delidx
            if len(delidx) > 0:
                traf.delete(delidx)

        # Autodelete for descending with swTaxi:
        if not self.swtaxi:
            delidxalt = np.where((self.oldalt >= self.swtaxialt)
                                 * (traf.alt < self.swtaxialt))[0]
            self.oldalt = traf.alt
            if len(delidxalt) > 0:
                traf.delete(list(delidxalt))


    def set_area(self, *args, exparea=False):
        ''' Set Experiment Area. Aircraft leaving the experiment area are deleted.
        Input can be existing shape name, or a box with optional altitude constraints.'''
        curname = self.exparea if exparea else self.delarea
        msgname = 'Experiment area' if exparea else 'Deletion area'
        # if all args are empty, then print out the current area status
        if not args:
            return True, f'{msgname} is currently ON (name={curname})' if self.active else \
                f'{msgname} is currently OFF'

        # start by checking if the first argument is a string -> then it is an area name
        if isinstance(args[0], str) and len(args) == 1:
            if areafilter.hasArea(args[0]):
                # switch on Area, set it to the shape name
                if exparea:
                    self.exparea = args[0]
                else:
                    self.delarea = args[0]

                self.active = True
                self.flst.start()
                self.conflog.start()
                #self.CRSolution.start()
                #self.stlog.start()
                #self.flpath.start()
                return True, f'{msgname} is set to {args[0]}'
            if args[0][:2] == 'OF':
                # switch off the area and reset the logger
                self.active = False
                return True, f'{msgname} is switched OFF'
            if args[0][:2] == 'ON':
                if not self.name:
                    return False, 'No area defined.'
                else:
                    self.active = True
                    return True, f'{msgname} switched ON (name={curname})'
            # shape name is unknown
            return False, 'Shapename unknown. ' + \
                   'Please create shapename first or shapename is misspelled!'
        # if first argument is a float -> then make a box with the arguments
        if isinstance(args[0], (float, int)) and 4 <= len(args) <= 6:
            self.active = True
            if exparea:
                self.exparea = 'EXPAREA'
                areafilter.defineArea('EXPAREA', 'BOX', args[:4], *args[4:])
            else:
                self.delarea = 'DELAREA'
                areafilter.defineArea('DELAREA', 'BOX', args[:4], *args[4:])
            self.flst.start()
            self.conflog.start()
            #self.CRSolution.start()
            #self.stlog.start()
            #self.flpath.start()
            return True, f'{msgname} is ON. Area name is: {"EXP" if exparea else "DEL"}AREA'

        return False, 'Incorrect arguments' + \
               '\nAREA Shapename/OFF or\n Area lat,lon,lat,lon,[top,bottom]'

    def set_taxi(self, flag, alt=1500 * ft):
        ''' Taxi ON/OFF to autodelete below a certain altitude if taxi is off'''
        self.swtaxi = flag  # True =  taxi allowed, False = autodelete below swtaxialt
        self.swtaxialt = alt
