""" Pilot logic."""
import numpy as np
import bluesky as bs
from bluesky.tools.aero import vtas2eas, vcas2tas, vcas2mach, vtas2cas
from bluesky.tools.trafficarrays import TrafficArrays, RegisterElementParameters
from bluesky.tools import plugin
from bluesky import settings
import copy


class Pilot(TrafficArrays):
    def __init__(self):
        super(Pilot, self).__init__()
        with RegisterElementParameters(self):
            # Desired aircraft states
            self.alt = np.array([])  # desired altitude [m]
            self.hdg = np.array([])  # desired heading [deg]
            self.trk = np.array([])  # desired track angle [deg]s
            self.vs = np.array([])  # desired vertical speed [m/s]
            self.tas = np.array([])  # desired speed [m/s]mv

    def create(self, n=1):
        super(Pilot, self).create(n)

        self.alt[-n:] = bs.traf.alt[-n:]
        self.tas[-n:] = bs.traf.tas[-n:]
        self.hdg[-n:] = bs.traf.hdg[-n:]
        self.trk[-n:] = bs.traf.trk[-n:]

    def APorASAS(self):
        # --------- Input to Autopilot settings to follow: destination or ASAS ----------
        # Convert the ASAS commanded speed from ground speed to TAS
        if bs.traf.wind.winddim > 0:
            vwn, vwe = bs.traf.wind.getdata(bs.traf.lat, bs.traf.lon, bs.traf.alt)
            asastasnorth = bs.traf.cr.tas * np.cos(np.radians(bs.traf.cr.trk)) - vwn
            asastaseast = bs.traf.cr.tas * np.sin(np.radians(bs.traf.cr.trk)) - vwe
            asastas = np.sqrt(asastasnorth ** 2 + asastaseast ** 2)
        # no wind, then ground speed = TAS
        else:
            asastas = bs.traf.cr.tas  # TAS [m/s]

        # Determine desired states from ASAS or AP. Select asas if there is a conflict AND resolution is on.
        #bs.traf.cr.active = np.zeros(len(bs.traf.cr.active))
        #print("deciding", bs.traf.cr.active, bs.traf.cr.intent_trk, bs.traf.ap.trk, len(bs.traf.cr.intent_trk), len(bs.traf.ap.trk))
        # #are in a turn and asas at the same time? lets not wait for the next asas hit (just one every second)
        # if len(bs.traf.cr.intent_trk) == len( bs.traf.ap.trk):
        #     ind_asas = \
        #         np.where(
        #             np.logical_and(bs.traf.cr.active == True, np.logical_or(abs(bs.traf.cr.intent_trk - bs.traf.ap.trk)>5, abs(bs.traf.cr.alt_no_vertical_deviation - bs.traf.selalt)>5)))[0]
        #     if len(ind_asas) > 0:
        #         active = np.zeros(bs.traf.ntraf)
        #         active[ind_asas] = True
        #         newtrack, newgscapped, newvs, alt, new_intent, new_selalt = \
        #         bs.traf.cr.resolve(bs.traf.cd, bs.traf, bs.traf, active)
        #         bs.traf.cr.trk[ind_asas] = newtrack[ind_asas]
        #         bs.traf.cr.tas[ind_asas] = newgscapped[ind_asas]
        #         asastas = bs.traf.cr.tas
        #         bs.traf.cr.alt[ind_asas] = alt[ind_asas]
        #         bs.traf.cr.vs[ind_asas] = newvs[ind_asas]
        #         bs.traf.cr.intent_trk[ind_asas] = new_intent[ind_asas]
        #         bs.traf.cr.alt_no_vertical_deviation[ind_asas] = new_selalt[ind_asas]

        self.trk = np.where(bs.traf.cr.active, bs.traf.cr.trk, bs.traf.ap.trk)
        #self.tas0 = copy.deepcopy(self.tas)
        self.tas = np.where(bs.traf.cr.active, bs.traf.cr.tas, bs.traf.ap.tas)
        self.alt = np.where(bs.traf.cr.active, bs.traf.cr.alt, bs.traf.ap.alt)
        #self.alt = copy.deepcopy(bs.traf.ap.alt)
        #self.vs = np.where(bs.traf.cr.active, bs.traf.cr.vs, bs.traf.ap.vs)
        self.vs = np.where(bs.traf.cr.active, bs.traf.cr.vs, bs.traf.ap.vs)
        #bs.traf.cr.active = np.zeros(bs.traf.ntraf)
        #
        # for it in range(len(bs.traf.cr.active)):
        #     if not bs.traf.cr.active[it]:
        #         self.vs[it] = 0

        #print(bs.traf.ap.vs)
        # merging_actions = plugin.active_plugins['MERGING_RL'].merging_RL.getActions()
        # # if len(merging_actions) > 0:
        # #     print('pilot', bs.sim.simt)
        # for it in range(len(merging_actions[0])):
        #     # ac_idx = bs.traf.id2idx(merging_actions[0][it])
        #     # vert_speed = merging_actions[1][it]
        #     # # print('merging', vert_speed)
        #     # next_alt = merging_actions[2][it]
        #     # self.vs[ac_idx] = vert_speed
        #     # if vert_speed > 0:
        #     #     self.alt[ac_idx] = next_alt
        #     # else:
        #     #     self.alt[ac_idx] = bs.traf.alt[ac_idx]
        #     ac_idx = bs.traf.id2idx(merging_actions[0][it])
        #     vert_speed = merging_actions[1][it]
        #     next_alt = merging_actions[2][it]
        #     hor_speed = merging_actions[3][it]
        #     self.alt[ac_idx] = next_alt
        #     self.vs[ac_idx] = vert_speed
        #     if not bs.traf.cr.active[ac_idx]:
        #         self.tas[ac_idx] = hor_speed

        #print(bs.traf.id, self.vs,self.alt )

        # apply limits maximum
        #for it in bs.traf.ntraf:

        # self.alt = np.where(bs.traf.cr.active, bs.traf.cr.alt, self.alt)
        # self.vs = np.where(bs.traf.cr.active, bs.traf.cr.vs, self.vs)
        #self.gs = np.where(bs.traf.cr.active, bs.traf.cr.vs, self.vs)

        # ASAS can give positive and negative VS, but the sign of VS is determined using delalt in Traf.ComputeAirSpeed
        # Therefore, ensure that pilot.vs is always positive to prevent opposite signs of delalt and VS in Traf.ComputeAirSpeed
        self.vs = np.abs(self.vs)


        # Compute the desired heading needed to compensate for the wind
        if bs.traf.wind.winddim > 0:

            # Calculate wind correction
            vwn, vwe = bs.traf.wind.getdata(bs.traf.lat, bs.traf.lon, bs.traf.alt)
            Vw = np.sqrt(vwn * vwn + vwe * vwe)
            winddir = np.arctan2(vwe, vwn)
            drift = np.radians(self.trk) - winddir  # [rad]
            steer = np.arcsin(np.minimum(1.0, np.maximum(-1.0,
                                                         Vw * np.sin(drift) / np.maximum(0.001, bs.traf.tas))))
            # desired heading
            self.hdg = (self.trk + np.degrees(steer)) % 360.
        else:
            self.hdg = self.trk % 360.

    def applylimits(self):
        # check for the flight envelope
        if settings.performance_model == 'openap':
            self.tas, self.vs, self.alt = bs.traf.perf.limits(self.tas, self.vs, self.alt, bs.traf.ax)
        else:
            bs.traf.perf.limits()  # Sets limspd_flag and limspd when it needs to be limited

            # Update desired sates with values within the flight envelope
            # When CAs is limited, it needs to be converted to TAS as only this TAS is used later on!

            self.tas = np.where(bs.traf.limspd_flag, vcas2tas(bs.traf.limspd, bs.traf.alt), self.tas)

            # Autopilot selected altitude [m]
            self.alt = np.where(bs.traf.limalt_flag, bs.traf.limalt, self.alt)

            # Autopilot selected vertical speed (V/S)
            self.vs = np.where(bs.traf.limvs_flag, bs.traf.limvs, self.vs)
