# -*- coding: utf-8 -*-
# Created by mjribeiro at 4-6-2020

import numpy as np
from bluesky.tools import geo
from bluesky.tools.aero import nm

N_points = 180


# p1: [x1,y1]   p2: [x2,y2]
def geofence_segment(obs_x1, obs_y1, obs_x2, obs_y2, dist1, dist2, qdr1, qdr2):
    dp = np.vstack((obs_x2, obs_y2)).T - np.vstack((obs_x1, obs_y1)).T
    #dp = np.vstack((obs_x1, obs_y1)).T - np.vstack((obs_x2, obs_y2)).T

    invert = True

    # the shape of the obstacles must be built CCW
    # if both quadrants are the same, start with the farthest point
    # if qdr1 > qdr2 or (qdr1 == qdr2 and dist1 > dist2):
    #     dp = np.vstack((obs_x1, obs_y1)).T - np.vstack((obs_x2, obs_y2)).T
    #     print('p1,p2')
    # else:
    #     print('p2,p1')

    theta = np.arctan2(dp[:, 1], dp[:, 0])
    #print(np.rad2deg(theta))

    # dp1 = np.vstack((obs_x1, obs_y1)).T - np.vstack((obs_x2, obs_y2)).T

    x_ = np.transpose(np.array([np.cos(theta), np.sin(theta)]))
    y_ = np.transpose(np.array([-np.sin(theta), np.cos(theta)]))

    d_geo = [-np.dot(np.vstack((obs_x1, obs_y1)).T[x], y_[x]) for x in range(len(y_))]
    #print('d_geo', d_geo)

    return x_, y_, d_geo, theta, invert


def geofence_VO_params(obs_lat1, obs_lon1, obs_lat2, obs_lon2, dist1, dist2, qdr1, qdr2, d_intlat, d_intlon, vx_int,
                       vy_int):
    x_, y_, d_geo, segment_theta, invert = geofence_segment(obs_lat1, obs_lon1, obs_lat2, obs_lon2, dist1, dist2, qdr1, qdr2)

    d_int = np.array([d_intlat, d_intlon])
    dint_x_ = [-np.dot(d_int, x_[x]) for x in range(len(x_))]
    dint_y_ = [np.dot(d_int, y_[x]) for x in range(len(y_))]
    theta_ = 0.5 * np.arctan2(dint_x_, dint_y_)

    # Total rotation angle
    theta_total = segment_theta + theta_

    # Secondary axis system primary axes
    x__ = np.transpose(np.array([np.cos(theta_total), np.sin(theta_total)]))
    y__ = np.transpose(np.array([-np.sin(theta_total), np.cos(theta_total)]))

    v_int = np.array([vx_int, vy_int])
    dint_x__ = [np.dot(d_int, x__[x]) for x in range(len(x__))]
    dint_y__ = [np.dot(d_int, y__[x]) for x in range(len(y__))]
    dint_v_int = np.dot(d_int, v_int)

    vint_x__ = [np.dot(v_int, x__[x]) for x in range(len(x__))]
    vint_y__ = [np.dot(v_int, y__[x]) for x in range(len(y__))]

    C1 = 1 + np.sin(theta_) * dint_x__ / d_geo
    C2 = 1 + np.cos(theta_) * dint_y__ / d_geo
    C3 = np.multiply(-2, vint_x__) - np.sin(theta_) * dint_v_int / d_geo
    C4 = np.multiply(-2, vint_y__) - np.cos(theta_) * dint_v_int / d_geo

    Cx__ = -C3 / (2 * C1)
    Cy__ = -C4 / (2 * C2)

    a2 = (- np.dot(v_int, v_int) + C2 * Cy__ ** 2) / C1 + Cx__ ** 2
    b2 = (- np.dot(v_int, v_int) + C1 * Cx__ ** 2) / C2 + Cy__ ** 2

    vint_y_ = [np.dot(v_int, y_[x]) for x in range(len(y_))]
    # valid_segments = np.where(np.asarray(vint_y_) < 0)[0]
    #print('vint_y_', vint_y_)

    return a2, b2, Cx__, Cy__, segment_theta, theta_, invert  # , valid_segments


# def latlontoCartesian(p):
#     p = np.deg2rad(p)
#
#     # Compute flat Earth correction at the center of the experiment circle
#     re = 6371000.  # radius earth [m]
#     x = re * p[1] * np.cos(0.5 * p[0])
#     y = re * p[0]
#
#     return np.array([x, y])

def getXYPosition(plat, plon, p0lat, p0lon):
    [qdr, distance] = geo.qdrdist_matrix(p0lat, p0lon, plat, plon)
    qdr = np.squeeze(np.asarray(qdr))
    distance = np.squeeze(np.asarray(distance))

    qdr_radians = np.deg2rad(qdr)
    distance = distance * nm

    return np.multiply(distance, np.sin(qdr_radians)), np.multiply(distance,
                                                                   np.cos(qdr_radians)), distance , qdr % 360


def construct_geofence_VO(obs_lat1, obs_lon1, obs_lat2, obs_lon2, d_ownlat, d_ownlon, d_intlat, \
                          d_intlon, vx_int, vy_int, vmax):
    # assume ownship at the center of the referential
    qdr, distance = geo.qdrdist(d_ownlat, d_ownlon, d_intlat, d_intlon)
    d_intx, d_inty = distance * nm * np.sin(np.deg2rad(qdr)), distance * nm * np.cos(np.deg2rad(qdr))
    d_ownlat = np.ones(len(obs_lat1)) * d_ownlat
    d_ownlon = np.ones(len(obs_lat1)) * d_ownlon
    obs_x1, obs_y1, dist1, qdr1 = getXYPosition(obs_lat1, obs_lon1, d_ownlat, d_ownlon)
    obs_x2, obs_y2, dist2, qdr2 = getXYPosition(obs_lat2, obs_lon2, d_ownlat, d_ownlon)

    a2, b2, Cx__, Cy__, segment_theta, vo_theta, invert = geofence_VO_params(obs_x1, obs_y1, obs_x2, obs_y2, dist1, dist2, qdr1,
                                                                     qdr2, d_intx, d_inty, vx_int, vy_int)

    VOs = []
    for it in range(len(a2)):
        if a2[it] <= 0:  # it not in valid_segments or
            # print('could not build geofence for ', obs_lat1[it], obs_lon1[it], obs_lat2[it], obs_lon2[it], \
            #       dist1[it], qdr1[it], dist2[it], qdr2[it])
            continue

        if b2[it] > 1e10:
            VOs.append(
                parabolic_VO(a2[it], b2[it], segment_theta[it], vo_theta[it], vmax, Cx__[it], Cy__[it], qdr1, qdr2, invert))
        elif b2[it] > 0:
            VOs.append(elliptical_VO(a2[it], b2[it], segment_theta[it], vo_theta[it], Cx__[it], Cy__[it], qdr1, qdr2, invert))
        else:
            VOs.append(
                hyperbolic_VO(a2[it], b2[it], segment_theta[it], vo_theta[it], vmax, Cx__[it], Cy__[it], qdr1, qdr2, invert))

    return VOs


def parabolic_VO(a2, b2, segment_theta, vo_theta, vmax, Cx__, Cy__, qdr1, qdr2, invert):
    b = 1e10
    theta_total = segment_theta + vo_theta
    tmax = np.log((2 * vmax + np.sqrt(4 * vmax ** 2 + a2)) / np.sqrt(a2))
    tmin = - tmax
    dt = np.linspace(tmax, tmin, N_points)

    if vo_theta > 0:
        x_ = - np.sqrt(a2) * np.cosh(dt) + Cx__
    else:
        x_ = np.sqrt(a2) * np.cosh(dt) + Cx__
    y_ = np.sqrt(b) * np.sinh(dt) + Cy__
    x = x_ * np.cos(theta_total) - y_ * np.sin(theta_total)
    y = x_ * np.sin(theta_total) + y_ * np.cos(theta_total)

    return np.vstack((x, y)).T


def elliptical_VO(a2, b2, segment_theta, vo_theta, Cx__, Cy__, qdr1, qdr2, invert):
    theta_total = segment_theta + vo_theta
    dt = np.linspace(0, 2 * np.pi, N_points)

    x_ = np.sqrt(a2) * np.cos(dt) + Cx__
    y_ = np.sqrt(b2) * np.sin(dt) + Cy__
    x = x_ * np.cos(theta_total) - y_ * np.sin(theta_total)
    y = x_ * np.sin(theta_total) + y_ * np.cos(theta_total)

    qdrs = np.arctan2(y, x)
    # valid_indices = np.where(np.logical_and(np.greater_equal(qdrs, np.deg2rad(min(qdr1, qdr2))), \
    #                                         np.less_equal(qdrs, np.deg2rad(max(qdr1, qdr2)))))[0]

    return np.vstack((x, y)).T


def hyperbolic_VO(a2, b2, segment_theta, vo_theta, vmax, Cx__, Cy__, qdr1, qdr2, invert):
    theta_total = segment_theta + vo_theta
    tmax = np.log((2 * vmax + np.sqrt(4 ** 2 * vmax ** 2 + a2)) / np.sqrt(a2))
    tmin = - tmax

    if vo_theta > 0:
        dt = np.linspace(tmin, tmax, N_points)
        x_ = - np.sqrt(a2) * np.cosh(dt) + Cx__
    else:
        dt = np.linspace(tmax, tmin, N_points)
        x_ = np.sqrt(a2) * np.cosh(dt) + Cx__
    y_ = np.sqrt(-b2) * np.sinh(dt) + Cy__
    x = x_ * np.cos(theta_total) - y_ * np.sin(theta_total)
    y = x_ * np.sin(theta_total) + y_ * np.cos(theta_total)

    # qdrs = np.arcsin(x/y)
    # valid_indices = np.where( np.logical_and(np.greater_equal(qdrs, np.deg2rad(min(qdr1, qdr2))-90),\
    #                                          np.less_equal(qdrs,np.deg2rad(max(qdr1, qdr2)-90))))[0]



    #return np.vstack((x[valid_indices], y[valid_indices])).T
    return np.vstack((x, y)).T
