%-------------------------------------------------------------------------%
% Background
% Shape function information (standard triangular and quadrilateral elements)
% Gaussian integration scheme, default element type
% Created by Lei, TU/e, 2017
% Reference: pyfem-1.2TUE, written by J.J.C. Remmers
%-------------------------------------------------------------------------%

%% Get shape data
function elemData = getElemShapeData(elemCoords,order)

% element type
elemType = getElemType(elemCoords);

% Gaussian integration points and weights
[intCrds,intWghts] = getIntegrationPoints(elemType,order);

% initialize
elemData = cell(1,length(intWghts));

% element shape data sets
for i = 1:size(intCrds,1)
    
    % shape functions and local derivatives
    sData = getShape(intCrds(i,:),elemType);
    
    % jacobian
    jac = elemCoords'*sData.dhdxi;  
    
    % global derivatives
    if size(jac,1) == size(jac,2)
       sData.dhdx = sData.dhdxi*inv(jac);
    end
    
    % global weights and coordinates
    wJ = calweight(jac);
    sData.weight = wJ*intWghts(i);
    sData.x = sData.h*elemCoords;
    
    % collect
    elemData{i} = sData;
    
end

end


%% Get weight of J
function wJ = calweight(jac)

dim = size(jac);
if dim(1) == dim(2)
    wJ = det(jac);
else if dim(1) ==2 && dim(2) ==1
        wJ = sqrt(sum(sum(jac*jac)));   
    end
end

end


%% Get element type (2D)
function elemType = getElemType(elemCoords)

nNel = length(elemCoords);          % node number

if nNel ==3
    elemType = 'Tria3';
else if nNel ==4
        elemType = 'Quad4';
    end
end

end


%% Get integration points
function [xi,weight] = getIntegrationPoints(elemType,order)

% triangular element
if strcmpi(elemType(1:end-1),'Tria')
   orderArray = [1,3,7];            % integration point number
   if strcmpi(elemType,'Tria3')
      stdOrder = 1;                 % standard form
   end
   [xi,weight] = tria_scheme(orderArray(stdOrder+order));
end

% quadrilateral element
if strcmpi(elemType(1:end-1),'Quad')
    if strcmpi(elemType,'Quad4')
       stdOrder = 2;
    end
    stdOrder = stdOrder+order;      % final form
    [xi,weight] = quad_scheme(stdOrder);    
end

end


%% Get integration scheme for triangluar element
function [xi,weight] = tria_scheme(order)

% 3 schemes
if order ==1
    xi =[1/3,1/3];                 % local coordinates
    weight = [1];                  % weights
else if order ==3
        r1 = 1/6; 
        r2 = 2/3;
        
        xi = [r1 r1; r2 r1; r1 r2];
        
        w1 = 1/3;
        weight = [w1,w1,w1];
    else if order==7
           r1 = 0.1012865073235;
           r2 = 0.7974269853531;
           r4 = 0.4701420641051;
           r6 = 0.0597158717898;
           r7 = 1.0/3.0;
           
           xi = [r1 r1; r2 r1; r1 r2; r4 r6; r4 r4; r6 r4; r7 r7];
           
           w1 = 0.1259391805448;
           w4 = 0.1323941527885;
           w7 = 0.225;
           weight = [w1,w1,w1,w4,w4,w4,w7];            
        end
    end
end

end


%% Get integration scheme for quadrilateral element
function [xi,weight] = quad_scheme(order)

% 1 shceme
if order ==2
    r1 = 1/sqrt(3);
    r2= -1/sqrt(3);
    xi = [ r2, r2; r2 r1; r1 r2; r1 r1];
           
    w1 = 1;    
    weight = [w1,w1,w1,w1];    
end

end


%% Get derivatives of shape functions
function sData = getShape(xi,elemType)

% triangular element
if strcmpi(elemType,'Tria3')
    sData.h = zeros(1,3);
    sData.dhdxi = zeros(3,2);             % local derivatives
    sData.xi = xi;
    
    % calcualte shape functions
    sData.h(1) = 1-xi(1)-xi(2);
    sData.h(2) = xi(1);
    sData.h(3) = xi(2);
    
    % derivatives 3*2
    sData.dhdxi(1,1) = -1;
    sData.dhdxi(2,1) = 1;
    sData.dhdxi(3,1) = 0;
    
    sData.dhdxi(1,2) = -1;
    sData.dhdxi(2,2) = 0;
    sData.dhdxi(3,2) = 1;
    
     % quadrilateral element
else if strcmpi(elemType,'Quad4')
        sData.h = zeros(1,4);
        sData.dhdxi = zeros(4,2);             % local derivatives
        sData.xi = xi;
        
        % calcualte shape functions
        sData.h(1) = 0.25*(1-xi(1))*(1-xi(2));
        sData.h(2) = 0.25*(1+xi(1))*(1-xi(2));
        sData.h(3) = 0.25*(1+xi(1))*(1+xi(2));
        sData.h(4) = 0.25*(1-xi(1))*(1+xi(2));
        
        % derivatives 4*2
        sData.dhdxi(1,1) = -0.25*(1-xi(2));
        sData.dhdxi(2,1) =  0.25*(1-xi(2));
        sData.dhdxi(3,1) =  0.25*(1+xi(2));
        sData.dhdxi(4,1) = -0.25*(1+xi(2));
        
        sData.dhdxi(1,2) = -0.25*(1-xi(1));
        sData.dhdxi(2,2) = -0.25*(1+xi(1));
        sData.dhdxi(3,2) =  0.25*(1+xi(1));
        sData.dhdxi(4,2) =  0.25*(1-xi(1));
    end
    
end

end
